/*                                                              coding: utf-8
 ============================================================================
 Name        : src/SaEngineModuleDispatch.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Engine's module for dispatching SMS into specified messaging
               folder.
 Notes:

 ============================================================================
*/


// INCLUDES

#include "SaEngineModuleDispatch.h"


// locally needed, system:

#include <msvuids.h>                        // KUidMsvFolderEntry
#include <smut.h>                           // KUidMsgTypeSMS
#include <msvids.h>                         // KMsvGlobalInBoxIndexEntryId
#include <smuthdr.h>                        // CSmsHeader
#include <gsmupdu.h>                        // CSmsDeliver
#include <txtrich.h>                        // CRichText


// LIBS
//
// euser.lib      // CBase
// msgs.lib       // CMsvSession, CClientMtmRegistry, CMsvEntrySelection,
//                // CMsvEntry, TMsvEntry, CMsvStore
// smcm.lib       // CSmsClientMtm
// etext.lib      // CRichText
// msgs.lib       // CMsvSession, CClientMtmRegistry;
// gsmu.lib       // CSmsDeliver
//
// NOTE: following libraries are mentioned in the API documentation,
//       but not present in SDK:
//       * smcm_gsm.lib          // CSmsClientMtm, CSmsHeader
//       * smcm_cdma.lib         // CSmsHeader
// fortunately, these can be replaced by reference to single library "smcm.lib".


// CAPS (complete)
//
// ReadUserData (est.)


// METHODS IMPLEMENTATION

// ===========================================================================
// CSaEngineModuleDispatch
// ===========================================================================

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::CSaEngineModuleDispatch()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CSaEngineModuleDispatch::CSaEngineModuleDispatch()
    : iStatus(EDispatchEngineNotReady)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch* CSaEngineModuleDispatch::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineModuleDispatch*
CSaEngineModuleDispatch::NewLC()
    {
    CSaEngineModuleDispatch* self = new (ELeave) CSaEngineModuleDispatch();
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch* CSaEngineModuleDispatch::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineModuleDispatch*
CSaEngineModuleDispatch::NewL()
    {
    CSaEngineModuleDispatch* self = CSaEngineModuleDispatch::NewLC();
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::ConstructL()
// Second-phase constructor.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleDispatch::ConstructL()
    {
    iMsvSession = CMsvSession::OpenAsyncL(*this);
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::~CSaEngineModuleDispatch()
// Destructor.
// ---------------------------------------------------------------------------
//
CSaEngineModuleDispatch::~CSaEngineModuleDispatch()
    {
    CloseSession();
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::HandleSessionEventL()
// Handler from MMsvSessionObserver.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleDispatch::HandleSessionEventL( TMsvSessionEvent aEvent,
                                              TAny* /*aArg1*/, TAny* /*aArg2*/,
                                              TAny* /*aArg3*/ )
    {
    switch (aEvent)
        {
        case EMsvServerReady:
            // Received after a client has used CMsvSession::OpenAsyncL() to create a session.
            //
            // The session can now be used.
            {
            iClientMtmRegistry = CClientMtmRegistry::NewL(*iMsvSession);
            iSmsClientMtm = static_cast<CSmsClientMtm*>(iClientMtmRegistry->NewMtmL(KUidMsgTypeSMS));

            iStatus = EDispatchEngineReady;
            break;
            }

        case EMsvCloseSession:
            // The client should immediately close the session with the Message Server.
        case EMsvServerTerminated:
            // The Message Server has been terminated.
            //
            // All clients must close their sessions immediately.
            {
            iStatus = EDispatchEngineNotReady;
            CloseSession(); // TODO: is it possible?
            break;
            }

        default:
            return;
        }
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::CloseSession()
// Closes session.
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleDispatch::CloseSession()
    {
    delete iSmsClientMtm;
    iSmsClientMtm = NULL;

    delete iClientMtmRegistry;
    iClientMtmRegistry = NULL;

    delete iMsvSession;
    iMsvSession = NULL;
    };

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::FindOrCreateFolderInMyFoldersL()
// Tries to find folder of specified name within "My Folders" folder.
// ---------------------------------------------------------------------------
//
TMsvId
CSaEngineModuleDispatch::FindOrCreateFolderInMyFoldersL(const TDesC& aFolderName)
    {

    // entry of "My Folders" folder of default messaging application
    CMsvEntry* entryMyFolders = iMsvSession->GetEntryL(KMsvMyFoldersEntryId);
    CleanupStack::PushL(entryMyFolders);

    // set of folders within "My Folders" folder
    CMsvEntrySelection* entries = entryMyFolders->ChildrenWithTypeL(KUidMsvFolderEntry);

    CleanupStack::PushL(entries);

    TMsvId folderId;
    CMsvEntry* entrySubfolder = NULL;

    // try to find folder with specified name within "My Folders"

    TInt nCount = entries->Count();
    TBool found = EFalse;
    TInt err;
    for (TInt i=1; i < nCount; i++) // because 0 is reserved for Templates in MyFolder
        {
        folderId = entries->At(i);
        TRAP(err, entrySubfolder = iMsvSession->GetEntryL(folderId));

        if (err != KErrNone || !entrySubfolder)
            {
            return 0;
            }

        // compare the name
        TPtrC details = entrySubfolder->Entry().iDetails;

        if (details.Compare(aFolderName) == 0)
            { // bingo
            found = ETrue;
            break;
            }
        }

    delete entrySubfolder;
    entrySubfolder = NULL;

    CleanupStack::PopAndDestroy(entries);

    // if no folder found create one
    if (!found)
        {
        TMsvEntry indexEntry;
        indexEntry.iDate.UniversalTime();
        indexEntry.iSize = 0;
        indexEntry.iType = KUidMsvFolderEntry;
        indexEntry.iMtm = KUidMsvLocalServiceMtm;
        indexEntry.iServiceId = KMsvLocalServiceIndexEntryId;
        indexEntry.iDetails.Set(aFolderName);
        indexEntry.iDescription.Set(KNullDesC);
        indexEntry.SetVisible(ETrue);

        entryMyFolders->CreateL(indexEntry);
        folderId = indexEntry.Id();
        }

    CleanupStack::PopAndDestroy(entryMyFolders);

    return folderId;
    }

// ---------------------------------------------------------------------------
// CSaEngineModuleDispatch::CreateSmsL()
// Creates an SMS in specified subfolder of "My Folders".
// ---------------------------------------------------------------------------
//
void
CSaEngineModuleDispatch::CreateSmsL( const TDesC& aSenderNumber,
                                     const TDesC& aSenderName,
                                     const TDesC& aText,
                                     const TDesC& aFolderName )
    {
    // test whether the engine is ready
    if (iStatus != EDispatchEngineReady)
        User::Leave(KErrNotReady);

    TMsvId folderId;
    if (aFolderName != KNullDesC)
        { // folder is specified
        folderId = FindOrCreateFolderInMyFoldersL(aFolderName);
        if (folderId == KNullUidValue)
            folderId = KMsvGlobalInBoxIndexEntryId;
        }
    else
        { // folder not specified -> standard "Inbox" is used
        folderId = KMsvGlobalInBoxIndexEntryId;
        }

    iSmsClientMtm->SwitchCurrentEntryL(folderId);
    iSmsClientMtm->CreateMessageL(KUidMsgTypeSMS.iUid);

    TMsvEntry entry = iSmsClientMtm->Entry().Entry();
    entry.SetInPreparation(EFalse);
    entry.SetVisible(ETrue);
    entry.SetNew(ETrue);
    entry.SetUnread(ETrue);
//  entry.iDate.HomeTime();
    entry.iDate.UniversalTime();
//  entry.iDescription.Set(KNullDesC);
    entry.iDescription.Set(aText);
    entry.iDetails.Set((aSenderName != KNullDesC) ? aSenderName : aSenderNumber);

// TODO: clean -vvv
// signalizace
//  entry.SetUnread(ETrue);
//  entry.SetNew( EFalse);

// nic
//  entry.SetUnread(EFalse);
//  entry.SetNew( ETrue);

    iSmsClientMtm->Entry().ChangeL(entry);
    iSmsClientMtm->SaveMessageL();

    CMsvStore* msvStore = iSmsClientMtm->Entry().EditStoreL();
    CleanupStack::PushL(msvStore);
    CRichText& body = iSmsClientMtm->Body();
    body.Reset();
    body.InsertL(0, aText);

    CSmsHeader* header = CSmsHeader::NewL(CSmsPDU::ESmsDeliver, body);
    CleanupStack::PushL(header);
// not necessary:
//    header->SetFromAddressL((aSenderName != KNullDesC) ? aSenderName : aSenderNumber);

    CSmsDeliver& deliverPdu = header->Deliver(); // get the Deliver PDU from the Header
    TTime timestamp;
    timestamp.UniversalTime();
    deliverPdu.SetServiceCenterTimeStamp(timestamp, 0);
    deliverPdu.SetToFromAddressL(aSenderNumber);

    header->StoreL(*msvStore);
    msvStore->StoreBodyTextL(body);
    msvStore->CommitL();

    CleanupStack::PopAndDestroy(header);
    CleanupStack::PopAndDestroy(msvStore);
    }

