/*                                                              coding: utf-8
 ============================================================================
 Name        : src/SaEngineCore.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Core of the engine handling incoming SMS.

 Notes:      * parts inspired by
               <http://discussion.forum.nokia.com/forum/showthread.php?t=79397>

 ============================================================================
*/


// INCLUDES

#include "SaEngineCore.h"

// for SaEngineCore.h:

#include <eikappui.h>                   // CEikAppUi
#include "SaDataModel.h"
#include "SaEngineModuleDispatch.h"
#include "SaEngineModuleNotify.h"

// locally needed, system:

#include <cntdb.h>                      // CContactDatabase, CContactTextDef,
                                        // TContactTextDefItem
#include <cntitem.h>                    // CContactItem
#include <cntdef.h>                     // KUidContactFieldGivenName,
                                        // KUidContactFieldFamilyName

// LIBS
//
// euser.lib             // CBase
// cntmodel.lib          // CContactDatabase, CContactItem, CContactTextDef,
                         // TContactTextDefItem
// eikcore.lib           // CEikAppUi

// CAPS
//
//

const TInt KMaxMatchPhoneNumber = 9;

_LIT(KGivenNameFamilyNameSeparator, " ");


// METHODS IMPLEMENTATION


// ===========================================================================
// CSaEngineCore
// ===========================================================================

// ---------------------------------------------------------------------------
// CSaEngineCore::CSaEngineCore()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CSaEngineCore::CSaEngineCore(MSaDataCache& aStorage)
    : iStorage(aStorage)
    {
    // No implementation required
    }

// ---------------------------------------------------------------------------
// CSaEngineCore* CSaEngineCore::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineCore*
CSaEngineCore::NewLC(MSaDataCache& aStorage, CEikAppUi* aAppUi)
    {
    CSaEngineCore* self = new (ELeave) CSaEngineCore(aStorage);
    CleanupStack::PushL(self);
    self->ConstructL(aAppUi);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineCore* CSaEngineCore::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaEngineCore*
CSaEngineCore::NewL(MSaDataCache& aStorage, CEikAppUi* aAppUi)
    {
    CSaEngineCore* self = CSaEngineCore::NewLC(aStorage, aAppUi);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaEngineCore::ConstructL()
// Second-phase constructor.
// ---------------------------------------------------------------------------
//
void
CSaEngineCore::ConstructL(CEikAppUi* aAppUi)
    {
    iDispatchModule = CSaEngineModuleDispatch::NewL();
    iNotifyModule = CSaEngineModuleNotify::NewL(aAppUi);

#ifndef __WINS__
    // SaEngineModuleReceive module uses calls, that can't be used
    // in emulator build
    iReceiveModule = CSaEngineModuleReceive::NewL(*this);
    iReceiveModule->StartListeningL();
#endif
    }

// ---------------------------------------------------------------------------
// CSaEngineCore::~CSaEngineCore()
// Destructor.
// ---------------------------------------------------------------------------
//
CSaEngineCore::~CSaEngineCore()
    {
#ifndef __WINS__
    // SaEngineModuleReceive module uses calls, that can't be used
    // in emulator build
    if (iReceiveModule)
        {
        iReceiveModule->StopListening();
        }

    delete iReceiveModule;
    iReceiveModule = NULL;
#endif

    delete iNotifyModule;
    iNotifyModule = NULL;

    delete iDispatchModule;
    iDispatchModule = NULL;
    }

// ---------------------------------------------------------------------------
// CSaEngineCore::SmsReceivedL()
// Handler from MSmsAlertsReceiveEngineObserver.
// ---------------------------------------------------------------------------
//
void
CSaEngineCore::SmsReceivedL(const TDesC& aTelnum, const TDesC& aBody)
    {
    HBufC* formattedName = NULL;
    const TDesC* senderName = &KNullDesC;

    // Open the default contact database
    CContactDatabase* contactDb = CContactDatabase::OpenL();
    CleanupStack::PushL(contactDb);

    CContactIdArray* itemIds;
    itemIds = contactDb->MatchPhoneNumberL(aTelnum, KMaxMatchPhoneNumber);
    CleanupStack::PushL(itemIds);

    if (itemIds->Count() > 0)
        {
        // the number is matching to one or more stored contacts
        // only the first one is used
        // TODO: use country calling prefix?

        CContactItem* contactItem;
//        CContactCard* contactItem;
        contactItem = /*(CContactCard*)*/ contactDb->ReadContactL((*itemIds)[0]);

        if (contactItem->Type() == KUidContactCard)
            {
            // only "contact card" type is possible to find appropriate contact groups

            CleanupStack::PopAndDestroy(itemIds);
            CleanupStack::PushL(contactItem);

            // get the sender's name (to be used instead of his/her number)
            CContactTextDef* contactFormat = CContactTextDef::NewLC();
            contactFormat->AppendL(TContactTextDefItem( KUidContactFieldFamilyName,
                                                        KGivenNameFamilyNameSeparator ));
            contactFormat->AppendL(TContactTextDefItem(KUidContactFieldGivenName));

            int len = 2 * KCntMaxTextFieldLength + KGivenNameFamilyNameSeparator().Length();
            formattedName = HBufC::NewLC(len);
            TPtr formattedNamePtr(formattedName->Des());
            contactDb->ReadContactTextDefL(*contactItem, formattedNamePtr, contactFormat);
            CleanupStack::Pop(formattedName);
            CleanupStack::PopAndDestroy(contactFormat);
            CleanupStack::PushL(formattedName);

            // find appropriate contact groups
            itemIds = static_cast<CContactCard*>(contactItem)->GroupsJoinedLC();
            CleanupStack::Pop(itemIds);
            CleanupStack::Pop(formattedName);
            CleanupStack::PopAndDestroy(contactItem);
            CleanupStack::PushL(itemIds);
            CleanupStack::PushL(formattedName);
            }
        else
            {
            // unsupported type of contact
            delete contactItem;
            contactItem = NULL;
            itemIds->Reset();
            }
        }
    else
        {
        itemIds->Reset(); // TODO: maybe not necessary
        }

    // find settings
    TSaFilterItemSettings itemSettings;
    TFilterItemHeader itemHeader;
    TInt whichUsed = 0;
    if (iStorage.FirstMatchingSenderGroupL(*itemIds, itemHeader, itemSettings))
        {
        whichUsed = 1;
        }
    else if (iStorage.FirstMatchingTelnumPatternL(aTelnum, itemHeader, itemSettings))
        {
        whichUsed = 2;
        }

    if (formattedName)
        {
        senderName = formattedName;
        }

    if (whichUsed == 0)
        {
        iDispatchModule->CreateSmsL(aTelnum, *senderName, aBody);
        }
    else
        {
        //
        iDispatchModule->CreateSmsL(aTelnum, *senderName, aBody, itemHeader);

        // vibrate
        if (itemSettings.iVibratingAlert)
            {
                iNotifyModule->Vibrate();
            }

        // alert tone
        if (itemSettings.iAlertTone != KNullDesC)
            {
                iNotifyModule->PlayL( itemSettings.iAlertTone,
                                      itemSettings.iAlertVolume,
                                      itemSettings.iOverrideProfile );
            }

        iNotifyModule->ShowNoteL();
        }

    if (formattedName)
        {
        CleanupStack::PopAndDestroy(formattedName);
        }
    CleanupStack::PopAndDestroy(itemIds);
    CleanupStack::PopAndDestroy(contactDb);
    }

