/*                                                              coding: utf-8
 ============================================================================
 Name        : src/SaDataStorageDictionary.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Concrete implementation of data storage.

 Notes:

 ============================================================================
*/


// INCLUDES

#include "SaDataStorageDictionary.h"

#include <e32cmn.h>                   // TUid
#include <f32file.h>
#include <s32file.h>                  // CDictionaryFileStore
#include <s32stor.h>                  // RDictionaryReadStream, RDictionaryWriteStream

#include "SaDataModel.h"
#include "Application.h"             // KUidSmsAlertsApplication


// LIBS (complete)
// euser.lib      // CBase; TUid
// cntmodel.lib   //
// estor.lib      // RDictionaryReadStream, RDictionaryWriteStream


// CAPS (complete)
//

#define TO_STR(aStr)                 #aStr
#define FILE_IN_PRIVATE_FOLDER_PATH(aUid, aFile) \
                                    "c:\\private\\" TO_STR(aUid) "\\" aFile
//                                     "c:\\private\\" TO_STR(aUid) "\\" aFile

#define UID3_STR                     a9e722b4
#define DICTIONARY_STORAGE_FILE      "smsalerts.ini"
#define DICTIONARY_STORAGE_FULLPATH  FILE_IN_PRIVATE_FOLDER_PATH(UID3_STR, DICTIONARY_STORAGE_FILE)

const TPtrC8 KDictionaryFilePtr = reinterpret_cast<const TUint8*>(DICTIONARY_STORAGE_FULLPATH);


// METHODS IMPLEMENTATION


// ===========================================================================
// CSaDataStorageDictionary
// ===========================================================================

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::CSaDataStorageDictionary()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CSaDataStorageDictionary::CSaDataStorageDictionary(RFs& aFs)
    : iOpened(EFalse)
    , iFs(aFs)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::NewLC()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaDataStorageDictionary*
CSaDataStorageDictionary::NewLC(RFs& aFs)
    {
    CSaDataStorageDictionary* self = new (ELeave) CSaDataStorageDictionary(aFs);
    CleanupStack::PushL(self);
    self->ConstructL();
    return self;
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CSaDataStorageDictionary*
CSaDataStorageDictionary::NewL(RFs& aFs)
    {
    CSaDataStorageDictionary* self = CSaDataStorageDictionary::NewLC(aFs);
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::ConstructL()
// Second-phase constructor.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::ConstructL()
    {
//    User::LeaveIfError(iFs.Connect()); // file server session

    iDictionaryFile.Copy(KDictionaryFilePtr);

    TBuf<KMaxFileName> path;
    iFs.PrivatePath(path);

//    CCommandLineArguments* arguments = 0;
//#ifndef __WINS__
//    // not an emulator, drive letter can be variable
//    TRAPD(rc, arguments = CCommandLineArguments::NewL());
//    if (arguments)
//    {
//        iDictionaryFile[0] = arguments->Arg(0)[0];
//        delete arguments;
//        arguments = NULL;
//    }
//#endif

    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::~CSaDataStorageDictionary()
// Destructor.
// ---------------------------------------------------------------------------
//
CSaDataStorageDictionary::~CSaDataStorageDictionary()
    {
    if (iOpened && iDictionaryFileStore)
        {
        iOpened = EFalse;
        delete iDictionaryFileStore;
        iDictionaryFileStore = NULL;
        }

//    delete &iDictionaryFile;
//    iDictionaryFile = NULL;

//    iFs.Close();
    }

// ===========================================================================
// from MSaSenderGroupDataStorage

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::OpenL()
// Called if the owner of the class is about to begin working
// with the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::OpenL()
    {
    if (iOpened)
        {
        return;
        }

    TInt error;
    TRAP(error, iDictionaryFileStore = CDictionaryFileStore::OpenL(iFs, _L("c:\\data\\zzz.ini"), KUidSmsAlertsApplication));
    if (error != KErrNone)
        {
        User::Leave(error);
        }
    iOpened = ETrue;
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::IsOpened()
// Called if the owner of the class is about to stop working with the storage.
// ---------------------------------------------------------------------------
//
TBool
CSaDataStorageDictionary::IsOpened()
const
    {
    return iOpened;
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::Close()
// Called if the owner of the class is about to stop working with the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::Close()
    {
    if (!iOpened)
        {
        return;
        }

    iOpened = EFalse;
    delete iDictionaryFileStore;
    iDictionaryFileStore = NULL;
    }

// ===========================================================================
// from MSmsAlertsSenderGroupHandler

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupCountL()
// Gets the number of sender group filter items in the storage.
// ---------------------------------------------------------------------------
//
TUint
CSaDataStorageDictionary::SenderGroupCountL()
const
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUint count;

    if (!iDictionaryFileStore->IsPresentL(STREAM_UID_SENDER_GROUP(0)))
        {
        RDictionaryWriteStream writeStream;
        writeStream.AssignLC(*iDictionaryFileStore, STREAM_UID_SENDER_GROUP(0));
        writeStream.WriteUint32L(0);
        writeStream.CommitL();
        writeStream.Close();
        CleanupStack::PopAndDestroy(&writeStream);
        iDictionaryFileStore->CommitL();
        count = 0;
        }
    else
        {
        RDictionaryReadStream readStream;
        readStream.OpenLC(*iDictionaryFileStore, STREAM_UID_SENDER_GROUP(0));
        count = readStream.ReadUint32L();
        readStream.Close();
        CleanupStack::PopAndDestroy(&readStream);
        }
    return count;
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupsResetL();
// Cleans the storage from stored sender group filter items.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupsResetL()
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUid dictStreamUid;
    TUint count = SenderGroupCountL();

    for (TUint index = 1; index <= count; index++)
        {
        dictStreamUid = STREAM_UID_SENDER_GROUP(index);
        iDictionaryFileStore->Remove(dictStreamUid);
        }
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupAddL()
// Appends a sender group filter item to the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupAddL( TContactItemId aGroupId,
                                           const TSaFilterItemSettings& aSettings )
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUint position = SenderGroupCountL() + 1;

    // At first, add the item.
    TUid dictStreamUid = STREAM_UID_SENDER_GROUP(position);
    RDictionaryWriteStream writeStream;
    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream.WriteInt32L(aGroupId);
    writeStream << aSettings;
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);

    // Then increment the counter.
    writeStream.AssignLC(*iDictionaryFileStore, STREAM_UID_SENDER_GROUP(0));
    writeStream.WriteUint32L(position);
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupRemoveL()
// Removes a sender group filter item from the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupRemoveL(TUint aIndex)
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUint count = SenderGroupCountL();
    // item to remove does not exist
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    // at first, move items following the removed item to new positions,
    // i.e. one position backward each
    TUint groupIndex;
    TUid dictStreamUid;
    RDictionaryReadStream readStream;
    RDictionaryWriteStream writeStream;
    for (groupIndex = aIndex; groupIndex < count; groupIndex++)
        {
        dictStreamUid = STREAM_UID_SENDER_GROUP(groupIndex + 1);
        readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
        dictStreamUid = STREAM_UID_SENDER_GROUP(groupIndex);
        writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
        writeStream.WriteL(readStream);
        writeStream.CommitL();
        writeStream.Close();
        CleanupStack::PopAndDestroy(&writeStream);
        readStream.Close();
        CleanupStack::PopAndDestroy(&readStream);
        }

    // then decrement the number of items
    writeStream.AssignLC(*iDictionaryFileStore, STREAM_UID_SENDER_GROUP(0));
    writeStream.WriteUint32L(count-1);
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);

    // finally, clean the last item, which is redundant now
    dictStreamUid =STREAM_UID_SENDER_GROUP(groupIndex + 1);
    iDictionaryFileStore->Remove(dictStreamUid);
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupMoveUpDownL()
// Moves a sender group filter item logically up or down in the list
// (within the storage).
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupMoveUpDownL(TUint aIndex, TBool aUp)
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check -- vital one
    TUint count = SenderGroupCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    // boundary check -- optimalization one
    if (aUp && aIndex == 1 || !aUp && aIndex == count) User::Leave(KErrGeneral);

    // at first, read the content of item which is to be moved to temporary memory
    TContactItemId groupId;
    TSaFilterItemSettings itemSettings;
    TUid dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex);

    RDictionaryReadStream readStream;
    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    groupId = readStream.ReadInt32L();
    readStream >> itemSettings;
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);

    // then copy content of the other item of the swapping pair to previously
    // backed-up item position
    RDictionaryWriteStream writeStream;

    dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex + (aUp ? -1 : +1));
    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex);
    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream.WriteL(readStream);
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);
    // finally, copy content of backed-up item to specified position
    dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex + (aUp ? -1 : +1));
    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream.WriteInt32L(groupId);
    writeStream << itemSettings;
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupGetHeaderL()
// Gets the sender group filter item header at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupGetHeaderL(TUint aIndex, TContactItemId& aGroupId)
const
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check
    TUint count = SenderGroupCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    TUid dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex);
    RDictionaryReadStream readStream;

    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    aGroupId = readStream.ReadInt32L();
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupGetSettingsL()
// Gets the sender group filter item settings at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupGetSettingsL(TUint aIndex, TSaFilterItemSettings& aSettings)
const
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check
    TUint count = SenderGroupCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    TUid dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex);
    RDictionaryReadStream readStream;

    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    readStream.ReadInt32L(); // "empty" read, discard group ID
                             // at the beginning of the stream
    readStream >> aSettings;
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::SenderGroupSetL()
// Sets the sender group filter item (header and settings) at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::SenderGroupSetL( TUint aIndex, const TContactItemId& aGroupId,
                                           const TSaFilterItemSettings& aSettings )
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check
    TUint count = SenderGroupCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    TUid dictStreamUid = STREAM_UID_SENDER_GROUP(aIndex);
    RDictionaryWriteStream writeStream;

    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream.WriteInt32L(aGroupId);
    writeStream << aSettings;
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    iDictionaryFileStore->CommitL();
    }

// ===========================================================================
// from MSaTelnumPatternDataStorage

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternCountL()
// Gets the number of tel.num.pattern filter items in the storage.
// ---------------------------------------------------------------------------
//
TUint
CSaDataStorageDictionary::TelnumPatternCountL()
const
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUint count;

    if (!iDictionaryFileStore->IsPresentL(STREAM_UID_TELNUM_PATTERN(0)))
        {
        RDictionaryWriteStream writeStream;
        writeStream.AssignLC(*iDictionaryFileStore, STREAM_UID_TELNUM_PATTERN(0));
        writeStream.WriteUint32L(0);
        writeStream.CommitL();
        writeStream.Close();
        CleanupStack::PopAndDestroy(&writeStream);
        count = 0;
        iDictionaryFileStore->CommitL();
        }
    else
        {
        RDictionaryReadStream readStream;
        readStream.OpenLC(*iDictionaryFileStore, STREAM_UID_TELNUM_PATTERN(0));
        count = readStream.ReadUint32L();
        readStream.Close();
        CleanupStack::PopAndDestroy(&readStream);
        }

    return count;
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternResetL()
// Cleans the storage from stored tel.num.pattern filter items.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternResetL()
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUid dictStreamUid;
    TUint count = TelnumPatternCountL();

    for (TUint index = 1; index <= count; index++)
        {
        dictStreamUid = STREAM_UID_TELNUM_PATTERN(index);
        iDictionaryFileStore->Remove(dictStreamUid);
        }
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternAddL()
// Appends a tel.num.pattern filter item to the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternAddL( const TSaTelnumPatternCacheItem& aPatternHeader,
                                             const TSaFilterItemSettings& aSettings )
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUint position = TelnumPatternCountL() + 1;

    // At first, add the item.
    TUid dictStreamUid = STREAM_UID_TELNUM_PATTERN(position);
    RDictionaryWriteStream writeStream;
    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream << aPatternHeader;
    writeStream << aSettings;
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);

    // Then increment the counter.
    writeStream.AssignLC(*iDictionaryFileStore, STREAM_UID_TELNUM_PATTERN(0));
    writeStream.WriteUint32L(position);
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternRemoveL()
// Removes a tel.num.pattern filter item from the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternRemoveL(TUint aIndex)
    {
    if (!iOpened) User::Leave(KErrGeneral);

    TUint count = TelnumPatternCountL();
    // item to remove does not exist
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    // at first, move items following the removed item to new positions,
    // i.e. one position backward each
    TUint patternIndex;
    TUid dictStreamUid;
    RDictionaryReadStream readStream;
    RDictionaryWriteStream writeStream;
    for (patternIndex = aIndex; patternIndex < count; patternIndex++)
        {
        dictStreamUid = STREAM_UID_TELNUM_PATTERN(patternIndex + 1);
        readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
        dictStreamUid = STREAM_UID_TELNUM_PATTERN(patternIndex);
        writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
        writeStream.WriteL(readStream);
        writeStream.CommitL();
        writeStream.Close();
        CleanupStack::PopAndDestroy(&writeStream);
        readStream.Close();
        CleanupStack::PopAndDestroy(&readStream);
        }

    // then decrement the number of items
    writeStream.AssignLC(*iDictionaryFileStore, STREAM_UID_TELNUM_PATTERN(0));
    writeStream.WriteUint32L(count-1);
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);

    // finally, clean the last item, which is redundant now
    dictStreamUid = STREAM_UID_TELNUM_PATTERN(patternIndex + 1);
    iDictionaryFileStore->Remove(dictStreamUid);
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternMoveUpDownL()
// Moves a tel.num.pattern filter item logically up or down in the list
// (within the storage).
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternMoveUpDownL(TUint aIndex, TBool aUp)
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check -- vital one
    TUint count = TelnumPatternCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    // boundary check -- optimalization one
    if (aUp && aIndex == 1 || !aUp && aIndex == count) User::Leave(KErrGeneral);

    // at first, read the content of item which is to be moved to temporary memory
    TSaTelnumPatternCacheItem itemHeader;
    TSaFilterItemSettings itemSettings;
    TUid dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex);

    RDictionaryReadStream readStream;
    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    readStream >> itemHeader;
    readStream >> itemSettings;
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);

    // then copy content of the other item of the swapping pair to previously
    // backed-up item position
    RDictionaryWriteStream writeStream;

    dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex + (aUp ? -1 : +1));
    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex);
    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream.WriteL(readStream);
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);
    // finally, copy content of backed-up item to specified position
    dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex + (aUp ? -1 : +1));
    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream << itemHeader;
    writeStream << itemSettings;
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    iDictionaryFileStore->CommitL();
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternGetHeaderL()
// Gets the tel.num.pattern filter item header at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternGetHeaderL( TUint aIndex,
                                                   TSaTelnumPatternCacheItem& aPatternHeader )
const
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check
    TUint count = TelnumPatternCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    TUid dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex);
    RDictionaryReadStream readStream;

    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    readStream >> aPatternHeader;
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternGetSettingsL()
// Gets the tel.num.pattern filter item settings at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternGetSettingsL( TUint aIndex,
                                                     TSaFilterItemSettings& aSettings )
const
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check
    TUint count = TelnumPatternCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    TUid dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex);
    RDictionaryReadStream readStream;

    readStream.OpenLC(*iDictionaryFileStore, dictStreamUid);
    TSaTelnumPatternCacheItem a;
    readStream >> a;
//    readStream.ReadL(sizeof(TSaTelnumPatternCacheItem)); // discards filter item header
//                                                     // at the beginning of the stream
    readStream >> aSettings;
    readStream.Close();
    CleanupStack::PopAndDestroy(&readStream);
    }

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary::TelnumPatternSetL()
// Sets the tel.num.pattern filter item (header and settings) at specified
// index within the storage.
// ---------------------------------------------------------------------------
//
void
CSaDataStorageDictionary::TelnumPatternSetL( TUint aIndex,
                                             const TSaTelnumPatternCacheItem& aPatternHeader,
                                             const TSaFilterItemSettings& aSettings )
    {
    if (!iOpened) User::Leave(KErrGeneral);

    // boundary check
    TUint count = TelnumPatternCountL();
    if (aIndex >= count) User::Leave(KErrGeneral);

    aIndex++;

    TUid dictStreamUid = STREAM_UID_TELNUM_PATTERN(aIndex);
    RDictionaryWriteStream writeStream;

    writeStream.AssignLC(*iDictionaryFileStore, dictStreamUid);
    writeStream << aPatternHeader;
    writeStream << aSettings;
    writeStream.CommitL();
    writeStream.Close();
    CleanupStack::PopAndDestroy(&writeStream);
    iDictionaryFileStore->CommitL();
    }

