/*                                                              coding: utf-8
 ============================================================================
 Name        : src/Document.cpp
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Application's document model.

 Notes:

 ============================================================================
*/


// INCLUDES

#include "Document.h"

// for SmsAlertsDocument.h:

#include "SaEngineCore.h"            // CSaEngineCore


// locally needed, system:

#include <cntdb.h>                   // CContactDatabase
#include <cntitem.h>                 // CContactGroup


// locally needed, own:

#include "AppUi.h"
#include "SaDataStorageDictionary.h"


// LIBS
//
// avkon.lib     // CAknDocument
// eikcore.lib   // CAknDocument
// eiksrv.lib    // CAknDocument
// cntmodel.lib  // CContactDatabase, CContactIdArray, CContactGroup


// CAP
//
// ReadUserData (est.)


// CONSTANTS
_LIT(KUnspecifiedGroupLabel, "Unspecified");


// METHODS IMPLEMENTATION


// ===========================================================================
// CDocument
// ===========================================================================

// ---------------------------------------------------------------------------
// CDocument::CDocument()
// Default C++ constructor.
// ---------------------------------------------------------------------------
//
CDocument::CDocument(CEikApplication& aApplication, RFs& aFs)
    : CAknDocument(aApplication)
    , iFs(aFs)
    {
    // No implementation required.
    }

// ---------------------------------------------------------------------------
// CDocument::NewL()
// Two-phased construction.
// ---------------------------------------------------------------------------
//
CDocument*
CDocument::NewL(CEikApplication& aApplication, RFs& aFs)
    {
    CDocument* self = new (ELeave) CDocument(aApplication, aFs);
    CleanupStack::PushL(self);
    self->ConstructL();
    CleanupStack::Pop(self);
    return self;
    }

// ---------------------------------------------------------------------------
// CDocument::ConstructL()
// Second-phase constructor.
// ---------------------------------------------------------------------------
//
void
CDocument::ConstructL()
    {
    iStorage = CSaDataStorageDictionary::NewL(iFs);
    iStorage->OpenL();

    ConstructAndFillCachesL();

    }

// ---------------------------------------------------------------------------
// CDocument::~CDocument()
// Destructor.
// ---------------------------------------------------------------------------
//
CDocument::~CDocument()
    {
    delete iStorage;
    iStorage = NULL;

    delete iSenderGroupCache;
    iSenderGroupCache = NULL;

    delete iTelnumPatternCache;
    iTelnumPatternCache = NULL;

    if (iStorage && iStorage->IsOpened())
        {
        iStorage->Close();
        }
    delete iStorage;
    iStorage = NULL;
    }

// ---------------------------------------------------------------------------
// CDocument::CreateAppUiL()
// Creates the application UI object for this document.
// ---------------------------------------------------------------------------
//
CEikAppUi*
CDocument::CreateAppUiL()
    {
    CEikAppUi* appUi = new (ELeave) CAppUi(this);
    iMainEngine = CSaEngineCore::NewL(*this, appUi);
    return appUi;
    }

// ---------------------------------------------------------------------------
// CDocument::ConstructAndFillCachesL()
// Creates and fill caches for particular filter types.
// ---------------------------------------------------------------------------
//
void
CDocument::ConstructAndFillCachesL()
    {
    TUint senderGroupCount = iStorage->SenderGroupCountL();
    TUint telnumPatternCount =  iStorage->TelnumPatternCountL();

    if (iSenderGroupCache)
        {
        delete iSenderGroupCache;
        iSenderGroupCache = NULL;
        }
    if (iTelnumPatternCache)
        {
        delete iTelnumPatternCache;
        iTelnumPatternCache = NULL;
        }
    iSenderGroupCache =
        new (ELeave) CSaSenderGroupCache(senderGroupCount + KCacheReservedItems);
    iTelnumPatternCache =
        new (ELeave) CSaTelnumPatternCache(telnumPatternCount + KCacheReservedItems);

    TContactItemId groupId;
    TInt foundIndex;
    TSaSenderGroupCacheItem groupCacheItem;

    CContactDatabase* contactsDb = CContactDatabase::OpenL();
    CleanupStack::PushL(contactsDb);
    CContactIdArray* groupArray = contactsDb->GetGroupIdListL();
    CleanupStack::PushL(groupArray);

    for (TUint groupIndex = 0; groupIndex < senderGroupCount; )
        {
        iStorage->SenderGroupGetHeaderL(groupIndex, groupId);
        foundIndex = groupArray->Find(groupId);
        if (foundIndex == KErrNotFound)
            {
            // such previously set group ID not found
            iStorage->SenderGroupRemoveL(groupIndex);
            senderGroupCount--;
            }
        else
            {
            groupCacheItem.iGroupId = groupId;
            CContactGroup* groupItem = static_cast<CContactGroup*>(contactsDb->ReadContactLC(groupId));
            groupCacheItem.iGroupLabel.Copy(groupItem->GetGroupLabelL().Left(KMaxFilterItemHeaderLength));
            CleanupStack::PopAndDestroy(groupItem);
            iSenderGroupCache->AppendL(groupCacheItem);
            groupIndex++;
            }
        }

    CleanupStack::PopAndDestroy(groupArray);
    CleanupStack::PopAndDestroy(contactsDb);

    TSaTelnumPatternCacheItem patternCacheItem;
    for (TUint patternIndex = 0; patternIndex < telnumPatternCount; patternIndex++)
        {
        iStorage->TelnumPatternGetHeaderL(patternIndex, patternCacheItem);
        iTelnumPatternCache->AppendL(patternCacheItem);
        }
    }


// ===========================================================================
// from MSaSenderGroupDataStorage

// ---------------------------------------------------------------------------
// CDocument::SenderGroupCountL()
// Gets the number of sender group filter items in the storage.
// ---------------------------------------------------------------------------
//
TUint
CDocument::SenderGroupCountL()
const
    {
    return iSenderGroupCache->Count();
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupsResetL();
// Cleans the storage from stored sender group filter items.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupsResetL()
    {
    iStorage->SenderGroupsResetL();
    iSenderGroupCache->Reset();
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupAddL()
// Appends a sender group filter to the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupAddL( TContactItemId aGroupId,
                            const TSaFilterItemSettings& aSettings )
    {
    iStorage->SenderGroupAddL(aGroupId, aSettings);
    TSaSenderGroupCacheItem groupCacheItem;
    groupCacheItem.iGroupId = aGroupId;
    groupCacheItem.iGroupLabel.Copy(KUnspecifiedGroupLabel());
    iSenderGroupCache->AppendL(groupCacheItem);
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupAddL()
// Appends a sender group filter item to the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupAddL( TContactItemId aGroupId,
                            const TSaFilterItemSettings& aSettings,
                            const TFilterItemHeader& aGroupLabel )
    {
    iStorage->SenderGroupAddL(aGroupId, aSettings);
    TSaSenderGroupCacheItem groupCacheItem;
    groupCacheItem.iGroupId = aGroupId;
    groupCacheItem.iGroupLabel.Copy(aGroupLabel);
    iSenderGroupCache->AppendL(groupCacheItem);
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupRemoveL()
// Removes a sender group filter item from the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupRemoveL(TUint aIndex)
    {
    iStorage->SenderGroupRemoveL(aIndex);
    iSenderGroupCache->Delete(aIndex);
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupMoveUpDownL()
// Moves a sender group filter item logically up or down in the list
// (within the storage).
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupMoveUpDownL(TUint aIndex, TBool aUp)
    {
    if (aUp && aIndex == 0u  ||  !aUp && TInt(aIndex) == (iSenderGroupCache->Count()-1) )
        {
        return;
        }

    iStorage->SenderGroupMoveUpDownL(aIndex, aUp);
    TSaSenderGroupCacheItem groupCacheItem = iSenderGroupCache->At(aIndex);
    iSenderGroupCache->At(aIndex) = iSenderGroupCache->At(aUp ? aIndex-1 : aIndex+1);
    iSenderGroupCache->At(aUp ? aIndex-1 : aIndex+1) = groupCacheItem;

    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupGetHeaderL
// Gets the sender group filter item header at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupGetHeaderL(TUint aIndex, TContactItemId& aGroupId)
const
    {
    aGroupId = iSenderGroupCache->At(aIndex).iGroupId;
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupGetSettingsL()
// Gets the sender group filter item settings at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupGetSettingsL(TUint aIndex, TSaFilterItemSettings& aSettings)
const
    {
    iStorage->SenderGroupGetSettingsL(aIndex, aSettings);
    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupSetL()
// Sets the sender group filter item (header and settings) at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::SenderGroupSetL( TUint aIndex, const TContactItemId& aGroupId,
                            const TSaFilterItemSettings& aSettings )
    {
    iStorage->SenderGroupSetL(aIndex, aGroupId, aSettings);
    }


// ===========================================================================
// from MSaTelnumPatternDataStorage

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternCountL()
// Gets the number of tel.num.pattern filter items in the storage.
// ---------------------------------------------------------------------------
//
TUint
CDocument::TelnumPatternCountL()
const
    {
    return iTelnumPatternCache->Count();
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternResetL()
// Cleans the storage from stored tel.num.pattern filter items.
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternResetL()
    {
    iStorage->TelnumPatternResetL();
    iTelnumPatternCache->Reset();
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternAddL()
// Appends a tel.num.pattern filter item to the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternAddL( const TSaTelnumPatternCacheItem& aPatternHeader,
                              const TSaFilterItemSettings& aSettings )
    {
    iStorage->TelnumPatternAddL(aPatternHeader, aSettings);
    TSaTelnumPatternCacheItem patternCacheItem;
    patternCacheItem.iPatternName.Copy(aPatternHeader.iPatternName);
    patternCacheItem.iPatternMask.Copy(aPatternHeader.iPatternMask);
    iTelnumPatternCache->AppendL(patternCacheItem);
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternRemoveL()
// Removes a tel.num.pattern filter item from the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternRemoveL(TUint aIndex)
    {
    iStorage->TelnumPatternRemoveL(aIndex);
    iTelnumPatternCache->Delete(aIndex);
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternMoveUpDownL()
// Moves a tel.num.pattern filter item logically up or down in the list
// (within the storage).
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternMoveUpDownL(TUint aIndex, TBool aUp)
    {
    if (aUp && aIndex == 0  ||  !aUp && TInt(aIndex) == (iTelnumPatternCache->Count()-1) )
        {
        return;
        }

    iStorage->TelnumPatternMoveUpDownL(aIndex, aUp);
    TSaTelnumPatternCacheItem patternCacheItem = iTelnumPatternCache->At(aIndex);
    iTelnumPatternCache->At(aIndex) = iTelnumPatternCache->At(aUp ? aIndex-1 : aIndex+1);
    iTelnumPatternCache->At(aUp ? aIndex-1 : aIndex+1) = patternCacheItem;
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternGetHeaderL()
// Gets the tel.num.pattern filter item header at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternGetHeaderL( TUint aIndex,
                                    TSaTelnumPatternCacheItem& aPatternHeader )
const
    {
    aPatternHeader = iTelnumPatternCache->At(aIndex);
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternGetSettingsL()
// Gets the tel.num.pattern filter item settings at specified index
// within the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternGetSettingsL( TUint aIndex,
                                      TSaFilterItemSettings& aSettings )
const
    {
    iStorage->TelnumPatternGetSettingsL(aIndex, aSettings);
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternSetL()
// Sets the tel.num.pattern filter item (header and settings) at specified
// index within the storage.
// ---------------------------------------------------------------------------
//
void
CDocument::TelnumPatternSetL( TUint aIndex,
                              const TSaTelnumPatternCacheItem& aPatternHeader,
                              const TSaFilterItemSettings& aSettings )
    {
    iStorage->TelnumPatternSetL(aIndex, aPatternHeader, aSettings);
    TSaTelnumPatternCacheItem& patternCacheItem = iTelnumPatternCache->At(aIndex);
    patternCacheItem.iPatternName.Copy(aPatternHeader.iPatternName);
    patternCacheItem.iPatternMask.Copy(aPatternHeader.iPatternMask);
    }


// ===========================================================================
// from MSaSenderGroupDataCache

// ---------------------------------------------------------------------------
// CDocument::FirstMatchingSenderGroupL()
// Tries to lookup first item within sender group filter items cache
// which matches one of specified contact groups.
// ---------------------------------------------------------------------------
//
TBool
CDocument::FirstMatchingSenderGroupL( const CContactIdArray& aSenderGroupIds,
                                      TFilterItemHeader& aGroupLabel,
                                      TSaFilterItemSettings& aSettings )
const
    {
    TInt senderGroupIdsCount = aSenderGroupIds.Count();
    if (senderGroupIdsCount == 0)
        {
        return EFalse;
        }
    else
        {
        TInt cacheCount = iSenderGroupCache->Count();
        for (TInt indexCache = 0; indexCache < cacheCount; cacheCount++)
            {
            for (TInt indexGroups = 0; indexGroups < senderGroupIdsCount; indexGroups++)
                {
                if (aSenderGroupIds[indexGroups] == iSenderGroupCache->At(indexCache).iGroupId)
                    {
                    // matched
                    aGroupLabel = iSenderGroupCache->At(indexCache).iGroupLabel;
                    iStorage->SenderGroupGetSettingsL(TUint(indexCache), aSettings);
                    return ETrue;
                    }
                }
            }
        }
    return EFalse;

    }

// ---------------------------------------------------------------------------
// CDocument::SenderGroupCache()
// Returns (read-only) array of "cached data" for sender group filter items.
// ---------------------------------------------------------------------------
//
const CSaSenderGroupCache*
CDocument::SenderGroupCache()
const
    {
    return iSenderGroupCache;
    }


// ===========================================================================
// from MSaTelnumPatternDataCache

// ---------------------------------------------------------------------------
// CDocument::FirstMatchingTelnumPatternL()
// Tries to lookup (first) item within tel.num.pattern filter items cache
// which matches specified tel.number.
// ---------------------------------------------------------------------------
//
TBool
CDocument::FirstMatchingTelnumPatternL( const TDesC& aTelnum,
                                        TFilterItemHeader& aPatternName,
                                        TSaFilterItemSettings& aSettings )
const
    {
    TTelnumPattern telnumPattern;

    TInt cacheCount = iTelnumPatternCache->Count();

    // for all patterns within tel.num. pattern filter
    for (TInt indexCache = 0; indexCache < cacheCount; indexCache++)
        {
        telnumPattern = iTelnumPatternCache->At(indexCache).iPatternMask;
        TPtrC leftMost = aTelnum.Left(telnumPattern.Length());
        if (leftMost.Compare(telnumPattern) == 0)
            {
            // matched
            aPatternName = iTelnumPatternCache->At(indexCache).iPatternName;
            iStorage->TelnumPatternGetSettingsL(TUint(indexCache), aSettings);
            return ETrue;
            }
        }
    return EFalse;
    }

// ---------------------------------------------------------------------------
// CDocument::TelnumPatternCache()
// Returns (read-only) array of "cached data" for tel.num.pattern filter items.
// ---------------------------------------------------------------------------
//
const CSaTelnumPatternCache*
CDocument::TelnumPatternCache()
const
    {
    return iTelnumPatternCache;
    }

