/*                                                              coding: utf-8
 ============================================================================
 Name        : inc/UiListBase.h
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Declaration of base filter items list view and related container
               control (listbox).

 Notes:

 ============================================================================
*/


#ifndef UI_LIST_BASE_H_JP
#define UI_LIST_BASE_H_JP


// INCLUDES

#include <coecntrl.h>                  // CCoeControl | cone.lib
#include <aknview.h>                   // CAknView | avkon.lib eikcore.lib eiksrv.lib
#include <eiklbo.h>                    // MEikListBoxObserver

#include "SaDataModelTypes.h"          // TSaFilterItemSettings


// FORWARD DECLARATIONS

class MEikCommandObserver;             // <eikcmobs.h> | eikcore.lib
class CEikTextListBox;                 // <eiktxlbx.h> | avkon.lib eikcoctl.lib eikctl.lib
class CEikMenuPane;                    // <eikmenup.h> | eikcoctl.lib eikcore.lib
class CGulIcon;                        // <gulicon.h> | egul.lib
class TVwsViewId;                      // <vwsdef.h>

class CListBaseContainer;     // locally

// CLASS DECLARATIONS


// ---------------------------------------------------------------------------
// CListBaseView
// ---------------------------------------------------------------------------

/**
 * Base filter items list view that owns container control for filter items
 * listing (CListBaseContainer).
 */
class CListBaseView : public CAknView
    {
protected: // Constructors

    /**
     * C++ default constructor.
     *
     * @param aEditViewId  Identification of corresponding edit view.
     */
    CListBaseView(TUid aEditViewId);

    /**
     * Second-phase constructor called by the application framework.
     *
     * Initializes view from the resource.
     */
    void ConstructL();

public: // Destructor

    /**
     * Destructor.
     */
    virtual ~CListBaseView();

public: // From CAknView

    void HandleCommandL(TInt aCommand);
    void HandleStatusPaneSizeChange();

protected: // From CAknView

    void DoActivateL( const TVwsViewId& aPrevViewId, TUid aCustomMessageId,
                      const TDesC8& aCustomMessage );
    void DoDeactivate();

public: // From MEikMenuObserver

    void DynInitMenuPaneL(TInt aResourceId, CEikMenuPane* aMenuPane);

public: // New methods

    /**
     * Shows the popup note with information about the application.
     */
    static void RunNoteAboutL();

    /**
     * Handle the event of changing filter item.
     */
    void HandleMenuItemChangeL();

    /**
     * Handle the event of moving filter item up/down.
     */
    void HandleMenuItemMoveL(TBool aUp);

    /**
     * Handle the event of removing filter item.
     */
    void HandleMenuItemRemoveL();

    /**
     * Handle the event of adding new filter item.
     */
    void HandleMenuItemAddL();

    /**
     * Handle the event of selecting "About" menu item.
     */
    void HandleMenuItemAboutL();

    /**
     * Handle the event of selecting "Exit" menu item.
     */
    void HandleMenuItemExitL();

public: // Pure virtual methods to be implemented in derived classes

    /**
     * Called if the view needs to create container control, that contains
     * the listbox for filter items listing.
     *
     * @return Created container.
     */
    virtual CListBaseContainer* CreateContainerL()=0;

    /**
     * Called within "add item" command handling, before activating edit view.
     *
     * @return  ETrue to continue the command handling (activate edit view),
     *          EFalse to break the command handling.
     */
    virtual TBool OnItemAddBeforeEditViewL()=0;

    /**
     * Called within "add item" command handling, after return from edit view.
     *
     * @param aCommit  ETrue if edit view on its return indicated, that
     *                 the settings edit transaction should be commited
     *                 (i.e. new filter item with these settings added),
     *                 EFalse if it indicated that the settings edit
     *                 transaction should be cancelled.
     */
    virtual void OnItemAddAfterEditViewL(TBool aCommit)=0;

    /**
     * Called within "change item" command handling, before activating edit view.
     *
     * @param aIndex  Index of the item to change its settings.
     * @return  ETrue to continue the command handling (activate edit view),
     *          EFalse to break the command handling.
     */
    virtual TBool OnItemChangeBeforeEditViewL(TInt aIndex)=0;

    /**
     * Called within "change item" command handling, after return from edit view.
     *
     * @param aCommit  ETrue if edit view on its return indicated, that
     *                 the settings edit transaction should be commited
     *                 (i.e. filter item should use these new settings),
     *                 EFalse if it indicated that the settings edit
     *                 transaction should be cancelled.
     */
    virtual void OnItemChangeAfterEditViewL(TBool aCommit)=0;

    /**
     * Called within "move item" command handling.
     *
     * @param aIndex  Index of item to be moved.
     * @param aUp     ETrue if the direction is "up", EFalse if it is "down".
     */
    virtual void OnItemMoveL(TInt aIndex, TBool aUp)=0;

    /**
     * Called within "remove item" command handling.
     *
     * @param aIndex  Index of item to be removed.
     */
    virtual void OnItemRemoveL(TInt aIndex)=0;

private:

    /**
     * Sets up status pane.
     */
    void SetupStatusPaneL();

public: // Data members

    enum TEditViewReason
        {
        EEditViewAdd = 0,
        EEditViewChange
        };

    TSaFilterItemSettings iSettings;
    TEditViewReason iEditViewReason;

    TUid iEditViewId;

private: // Data members

    TInt iSelectedItem;
    CListBaseContainer* iContainer;

    }; // CListBaseView


// ---------------------------------------------------------------------------
// CListBaseContainer
// ---------------------------------------------------------------------------

/**
 * Base container control for filter items listing (in the form of a listbox).
 *
 * Class provides also methods for loading listbox items, removing single items
 * and moving single items up or down within the listbox.
 */
class CListBaseContainer : public CCoeControl
                         , public MEikListBoxObserver
    {
protected: // Constructor

    /**
     * Default C++ constructor.
     *
     * @param aViewUid          Identification of a view owning this container control.
     * @param aCommandObserver  Command observer.
     */
    CListBaseContainer(TUid aViewUid, MEikCommandObserver* aCommandObserver);

public: // Constructor and destructor

    /**
     * Destructor.
     */
    virtual ~CListBaseContainer();

    /**
     * Second-phase constructor (common for derived classes).
     *
     * @param aRect    Bounding rectangle.
     * @param aParent  Owning parent, or NULL.
     */
    void ConstructL(const TRect& aRect, const CCoeControl* aParent);

public: // From CCoeControl

    TInt CountComponentControls() const;
    CCoeControl* ComponentControl(TInt aIndex) const;
    TKeyResponse OfferKeyEventL(const TKeyEvent& aKeyEvent, TEventCode aType);
    void HandleResourceChange(TInt aType);

protected: // From CCoeControl

    void SizeChanged();

private: // From CCoeControl

    void Draw(const TRect& aRect) const;

public: // From MEikListBoxObserver

    void HandleListBoxEventL(CEikListBox* aListBox, TListBoxEvent aEventType);

protected: // Pure virtual methods to be implemented in derived classes

    /**
     * Creates listbox and saves its pointer to iListBox data member.
     */
    virtual void CreateListboxL()=0;

    /**
     * Returns number of items in associated filter items "data cache".
     *
     * @return Number of items in associated filter items "data cache".
     */
    virtual TInt CacheItemsCount()=0;

    /**
     * Returns formatted text to be used as item text within associated listbox
     * for item from "data cache" at specified index.

     * @param aFormatted  On return, formatted text to be used as item text
     *                    within associated listbox.
     * @param aIndex      Index to "data cache", for which to obtain formatted
     *                    text.
     */
    virtual void CacheItemL(TDes& aFormatted, TInt aIndex)=0;

public: // New functions

    /**
     * Static method for formatting item text to the form expected
     * by listbox of CAknSingleGraphicStyleListBox class.
     *
     * @param aBuffer     On return, the formatted text for the item.
     * @param aIconIndex  Index of icon to be used with the item in the listbox.
     * @param aLabel      Input text to be used as a label for the item
     *                    in the listbox.
     */
    static void FormatSingleGraphicStyleListBoxItemL( TDes& aBuffer, TInt aIconIndex,
                                                      const TDesC& aLabel );

    /**
     * Static method, which takes information about icons (icons filename and
     * array of indexes to this file) and returns array of such loaded icons.
     *
     * Indexes has to be in the order, where even position is for the
     * icon bitmap and following odd position for icon's mask. (i.e. "IMIM...")
     *
     * @param aFileName                Name of the icon file.
     * @param aArrayOfBitmapsAndMasks  Array of indexes to aFileName.
     * @return  Array of loaded icons.
     */
    static CArrayPtr<CGulIcon>*
        CreateIconsL(TFileName aFileName, CArrayFixFlat<TInt>* aArrayOfBitmapsAndMasks);

    /**
     * Loads items to the listbox using CacheItemsCount() and CacheItemL()
     * implemented in derived class.
     */
    void LoadItemsFromCacheL();

    /**
     * Moves single listbox item at specified index up or down.
     *
     * @param aIndex  Index of item to move up/down.
     * @param aUp     ETrue for "up" direction, EFalse for "down" direction.
     */
    void MoveUpDownItemL(TInt aIndex, TBool aUp);

    /**
     * Removes single item from the listbox at specified index.
     *
     * @param aIndex  Index of item to remove.
     */
    void RemoveItemL(TInt aIndex);

    /**
     * Gets the index number of the selected listbox item.
     *
     * @return  Index number of the selected listbox item.
     */
    TInt CurrentItemIndex();

    /**
     * Sets the index of a selected listbox item.
     *
     * @param aIndex  Index of item in the listbox to be selected.
     */
    void SetCurrentIndex(TInt aIndex);

    /**
     * Gets the number of the listbox items.
     *
     * @return  Number of listbox items.
     */
    TInt NumberOfItems();

private:

    void LayoutControls();

protected: // Data members

    TUid iViewUid;
    MEikCommandObserver* iCommandObserver;
    CEikTextListBox* iListBox;

    }; // CListBaseContainer


#endif // UI_LIST_BASE_H_JP
