/*                                                              coding: utf-8
 ============================================================================
 Name        : inc/UiEditBase.h
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Declaration of base filter items edit view and related container
               control (setting item list).

 Notes:

 ============================================================================
*/


#ifndef UI_EDIT_BASE_H_JP
#define UI_EDIT_BASE_H_JP


// INCLUDES

#include <aknsettingitemlist.h>        // CAknSettingItemList | avkon.lib
#include <aknview.h>                   // CAknView | avkon.lib eikcore.lib eiksrv.lib

#ifndef __SERIES60_30__
#include <mproengalerttoneseekerobserver.h>  // MProEngAlertToneSeekerObserver
#include <aknprogressdialog.h>               // MProgressDialogCallback
#endif

#include "SaDataModelTypes.h"          // TSaFilterItemSettings


// FORWARD DECLARATIONS

class MEikCommandObserver;             // <eikcmobs.h> | eikcore.lib
class CAknNavigationDecorator;         // <aknnavide.h> |  avkon.lib
class CEikMenuPane;                    // <eikmenup.h> | eikcoctl.lib eikcore.lib
class TVwsViewId;                      // <vwsdef.h>

#ifndef __SERIES60_30__
class MProEngAlertToneSeeker;          // <mproengalerttoneseeker.h>
                                       // profileengine.lib
class CAknWaitDialog;                  // <aknwaitdialog.h> | avkon.lib
                                       // eikcdlg.lib eikctl.lib
#endif

class CEditBaseContainer;     // locally


// CLASS DECLARATIONS

// ---------------------------------------------------------------------------
// CEditBaseView
// ---------------------------------------------------------------------------

/**
 * Base filter items edit view that owns container control for filter items
 * editing (in the form of a setting item list).
 */
class CEditBaseView : public CAknView
#ifndef __SERIES60_30__
                             , public MProEngAlertToneSeekerObserver
                             , public MProgressDialogCallback
#endif
    {
protected: // Constructors

    /**
     * C++ default constructor.
     * @param aResourceId  Resource identificator of AVKON_SETTING_ITEM_LIST
     *                     structure in the resource file.
     */
    CEditBaseView(TInt aResourceId);

    /**
     * Second-phase constructor called by the application framework.
     *
     * Initializes view from the resource.
     */
    void ConstructL();

public: // Destructor

    /**
     * Destructor.
     */
    virtual ~CEditBaseView();

public: // From CAknView

    void HandleCommandL( TInt aCommand );
    void HandleStatusPaneSizeChange();

protected: // From CAknView

    void DoActivateL( const TVwsViewId& aPrevViewId, TUid aCustomMessageId,
                      const TDesC8& aCustomMessage );
    void DoDeactivate();

public: // From MEikMenuObserver

    void DynInitMenuPaneL(TInt aResourceId, CEikMenuPane* aMenuPane);

#ifndef __SERIES60_30__
public: // From MProEngAlertToneSeekerObserver

    void HandleAlertToneListCompletedL(MDesCArray *aToneList);
    void HandleError(TInt aError);

public: // From MProgressDialogCallback

    void DialogDismissedL(TInt aButtonId);

#endif

public: // New methods

    /**
     * Shows popup dialog whether to save changes or not.
     *
     * @return  EAknSoftkeyYes (left soft key id) or 0.
     */
    static TInt RunQuerySaveL();

    /**
     * Handle the event of changing selected setting item.
     */
    void HandleMenuItemChangeL();

#ifndef __SERIES60_30__
    /**
     * Handle the event of alternatively changing message alert tone.
     */
    void HandleMenuItemChangeAlternativelyL();
#endif

    /**
     * Handle the event of clearing selected setting item.
     */
    void HandleMenuItemClearL();

    /**
     * Handle the event of showing selected setting item.
     */
    void HandleMenuItemShowL();


    /**
     * Handle the event of pressing right soft key.
     */
    void HandleRightSoftKeyPressedL();

protected: // Pure virtual methods to be defined in derived class

    /**
     * Creates the container (in the form of a setting item list)
     * for filter items editing.
     *
     * Such container is specialization of CEditBaseContainer.
     *
     * @return Created instance of the container.
     */
    virtual CEditBaseContainer* ItemSettingList()=0;

    /**
     * Return text to be displayed in the status pane area.
     *
     * @return Text to be displayed in the status pane area.
     */
    virtual HBufC* StatusPaneTextLC() const=0;

protected: // New methods

    /**
     * Sets up status pane.
     */
    void SetupStatusPaneL();

    /**
     * Cleans up status pane.
     */
    void CleanupStatusPane();

private: // Data members

    enum TViewActivationContext
        {
        EViewActivationContextUnknown,
        EViewActivationContextAdd,
        EViewActivationContextChange
        };

    /**
     * Resource identificator of AVKON_SETTING_ITEM_LIST structure
     * in the resource file.
     */
    TInt iResourceId;

    /** Owned container (setting item list). */
    CEditBaseContainer* iSettingItemList;

    CAknNavigationDecorator* iNaviDecorator;

    /** Remembers the local view, which called activation of this view. */
    TUid iViewUid;

    /**
     * Remembers the context, in which the edit view is activated (add/change)
     * item.
     */
    TViewActivationContext iViewActivationContext;

#ifndef __SERIES60_30__
    MProEngAlertToneSeeker* iToneSeeker;
    CAknWaitDialog*         iWaitDialog;
#endif

    }; // CEditBaseView


// ---------------------------------------------------------------------------
// CEditBaseContainer
// ---------------------------------------------------------------------------

/**
 * Base container control for filter items editing (in the form of a setting
 * item list).
 *
 * Class provides also methods for clearing filename of alert tone
 * and predicate, whether the user has already requested editing of some
 * setting item.
 */
class CEditBaseContainer : public CAknSettingItemList
    {
protected: // Constructor

    /**
     * C++ constructor.
     *
     * @param aViewUid          Identification of a view owning this container control.
     * @param aSettings         Data structure to be manipulated using UI.
     * @param aCommandObserver  Command observer.
     */
    CEditBaseContainer(TUid aViewUid, TSaFilterItemSettings& aSettings,
                       MEikCommandObserver* aCommandObserver);

public: // Destructor

    /**
     * Destructor.
     */
    virtual ~CEditBaseContainer();

public: // From CAknSettingItemList

    void EditItemL(TInt aIndex, TBool aCalledFromMenu);
    TKeyResponse OfferKeyEventL(const TKeyEvent& aKeyEvent, TEventCode aType);
    void HandleResourceChange(TInt aType);

protected: // From CCoeControl

    void SizeChanged();

protected: // From CAknSettingItemList

    CAknSettingItem* CreateSettingItemL(TInt id);

public: // New methods

    /**
     * Clears filename of alert tone within settings data.
     */
    inline void ClearAlertTone()
        {
        iSettings.iAlertTone = KNullDesC;
        iEdited = ETrue;
        };

    /**
     * Sets filename of alert tone within settings data.
     *
     * @param aFilename  Filename to be set.
     */
    inline void SetAlertTone(const TDesC& aFilename)
        {
        iEdited = ETrue;
        iSettings.iAlertTone.Copy(aFilename);
        };

    /**
     * Gets filename of alert tone within settings data.
     *
     * @return Filename of alert tone within settings data.
     */
    inline const TDesC& GetAlertTone()
        {
        return iSettings.iAlertTone;
        };

    /**
     * Returns whether the user has already requested editing of some setting
     * item.
     *
     * @return  ETrue if user has requested editing of some item,
     *          EFalse otherwise.
     */
    inline TBool IsEdited()
        {
        return iEdited;
        }

protected: // Data members

    TUid iViewUid;
    TSaFilterItemSettings& iSettings;
    MEikCommandObserver* iCommandObserver;

    TBool iEdited;

    }; // CEditBaseContainer


#endif // UI_EDIT_BASE_H_JP
