/*                                                              coding: utf-8
 ============================================================================
 Name        : inc/SaDataStorageDictionary.h
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Declaration of concrete implementation of data storage.

 Notes:

 ============================================================================
*/


#ifndef SA_DATA_STORAGE_DICTIONARY_H_JP
#define SA_DATA_STORAGE_DICTIONARY_H_JP


// INCLUDES

#include "SaDataModel.h"


// FORWARD DECLARATIONS

class RFs;                              // <f32file.h>
class CDictionaryFileStore;             // <s32file.h>  | estor.lib


// CONSTANTS

/** Maximum number of filter items per filter type. */
const TInt KMaxFilterItemsPerType = 0xFF;

/**
 * Base UID of the first filter type.
 *
 * Base UID of the steam within dictionary store contains number of stored
 * filter items of particular type. Sequentially following identifiers
 * of streams within store are used for filter items, one for each,
 * respecting order of the items.
 *
 * Therefore "KMaxFilterItemsPerType + 1" stores are allocated per filter type.
 */
#define STREAM_UID_FIRST_FILTER_TYPE  (0x0100)

/** Base UID of sender group filter items. @see STREAM_UID_FIRST_FILTER_TYPE */
const TInt KStreamUidSenderGroupBase = STREAM_UID_FIRST_FILTER_TYPE;
/** Base UID of tel.num.pattern filter items. @see STREAM_UID_FIRST_FILTER_TYPE */
const TInt KStreamUidTelnumPatternBase =
    1 * KMaxFilterItemsPerType + STREAM_UID_FIRST_FILTER_TYPE + 1;


// MACROS

/**
 * Returns UID for the steam of sender group filter item with given index.
 * @see STREAM_UID_FIRST_FILTER_TYPE
 */
#define STREAM_UID_SENDER_GROUP(INDEX) \
            (TUid::Uid(KStreamUidSenderGroupBase + (INDEX)))
/**
 * Returns UID for the steam of tel.num.pattern filter item with given index.
 * @see STREAM_UID_FIRST_FILTER_TYPE
 */
#define STREAM_UID_TELNUM_PATTERN(INDEX) \
            (TUid::Uid(KStreamUidTelnumPatternBase + (INDEX)))


// CLASS DECLARATIONS

// ---------------------------------------------------------------------------
// CSaDataStorageDictionary
// ---------------------------------------------------------------------------

/**
 * Derived class for handling data storage using "dictionary store".
 */
class CSaDataStorageDictionary : public CSaDataStorageBase
    {
public: // Constructors and destructor

    /**
     * Two-phased construction.
     *
     * @param aFs  Handle to a file server session.
     */
    static CSaDataStorageDictionary* NewL(RFs& aFs);

    /**
     * Two-phased construction.
     *
     * @param aFs  Handle to a file server session.
     */
    static CSaDataStorageDictionary* NewLC(RFs& aFs);

    /**
     * Destructor.
     */
    ~CSaDataStorageDictionary();

private: // Constructors

    /**
     * Default C++ constructor.
     *
     * @param aFs  Handle to a file server session.
     */
    CSaDataStorageDictionary(RFs& aFs);

    /**
     * Second-phase constructor.
     */
    void ConstructL();

public: // From CSaDataStorageBase

    void OpenL();
    inline TBool IsOpened() const;
    void Close();

public: // From MSaSenderGroupDataStorage

    TUint SenderGroupCountL() const;
    void  SenderGroupsResetL();
    void  SenderGroupAddL( TContactItemId aGroupId,
                           const TSaFilterItemSettings& aSettings );
    void  SenderGroupRemoveL(TUint aIndex);
    void  SenderGroupMoveUpDownL(TUint aIndex, TBool aUp);
    void  SenderGroupGetHeaderL(TUint aIndex, TContactItemId& aGroupId) const;
    void  SenderGroupGetSettingsL(TUint aIndex, TSaFilterItemSettings& aSettings) const;
    void  SenderGroupSetL( TUint aIndex, const TContactItemId& aGroupId,
                           const TSaFilterItemSettings& aSettings );

public: // From MSaTelnumPatternDataStorage

    TUint TelnumPatternCountL() const;
    void  TelnumPatternResetL();
    void  TelnumPatternAddL( const TSaTelnumPatternCacheItem& aPatternHeader,
                             const TSaFilterItemSettings& aSettings );
    void  TelnumPatternRemoveL(TUint aIndex);
    void  TelnumPatternMoveUpDownL(TUint aIndex, TBool aUp);
    void  TelnumPatternGetHeaderL( TUint aIndex,
                                   TSaTelnumPatternCacheItem& aPatternHeader) const;
    void  TelnumPatternGetSettingsL(TUint aIndex, TSaFilterItemSettings& aSettings) const;
    void  TelnumPatternSetL( TUint aIndex,
                             const TSaTelnumPatternCacheItem& aPatternHeader,
                             const TSaFilterItemSettings& aSettings );

private: // Data members

    TBool iOpened;
    RFs& iFs;
    TFileName iDictionaryFile;
    CDictionaryFileStore* iDictionaryFileStore;

    }; // CSaDataStorageDictionary


#endif // SA_DATA_STORAGE_DICTIONARY_H_JP
