/*                                                              coding: utf-8
 ============================================================================
 Name        : inc/SaDataModelTypes.h
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Declaration of data types related to application's data model.

 Notes:

 ============================================================================
*/


#ifndef SA_DATA_MODEL_TYPES_H_JP
#define SA_DATA_MODEL_TYPES_H_JP


// INCLUDES

#include <cntdef.h>                 // TContactItemId
#include "SmsAlerts.hrh"            // MAX_FILTER_ITEM_HEADER_LENGTH,
                                    // MAX_TELNUM_PATTERN_LENGTH,
                                    // MAX_FILE_NAME

// FORWARD DECLARATIONS

template <class T>
class CArrayFixFlat;                // <e32base.h> | euser.lib

class RReadStream;                  // <s32strm.h> | estor.lib
class RWriteStream;                 // <s32strm.h> | estor.lib

class TSaSenderGroupCacheItem;      // locally
class TSaTelnumPatternCacheItem;    // locally


// CONSTANTS

/** Maximum length of filter item's header. */
const TInt KMaxFilterItemHeaderLength = MAX_FILTER_ITEM_HEADER_LENGTH;
const TInt KMaxTelnumPatternLength = MAX_TELNUM_PATTERN_LENGTH;


// TYPEDEFS

typedef TBuf<MAX_FILE_NAME> TToneFileName;


// DATA TYPES

/**
 * Tel.num.pattern mask.
 *
 * Maximum length of the mask is given by system constant with probable
 * meaning of maximum phone number length used in
 * CContactDatabase::MatchPhoneNumberL.
 *
 * For SDK 3rd FP1, the constant has value 15.
 */
typedef TBuf<KMaxPhoneMatchLength> TTelnumPattern;

/** Filter item's header. */
typedef TBuf<KMaxFilterItemHeaderLength> TFilterItemHeader;

/**
 * Sender group filter items cache (array of particular data of the items).
 */
typedef CArrayFixFlat<TSaSenderGroupCacheItem> CSaSenderGroupCache;

/**
 * Tel.num. pattern filter items cache (array of particular data of the items).
 */
typedef CArrayFixFlat<TSaTelnumPatternCacheItem> CSaTelnumPatternCache;


// CLASS DECLARATIONS

// ---------------------------------------------------------------------------
// TSaFilterItemSettings
// ---------------------------------------------------------------------------

/**
 * Data unit of settings per filter item.
 *
 * Beside its public data members, it offers also method to externalize,
 * internalize and reset contained data.
 */
class TSaFilterItemSettings
    {
public: // Methods for using "<<" and ">>" operators with streams

    /**
     * Writes serialized settings data to stream.
     *
     * @param aStream  Stream where to externalize.
     */
    inline void ExternalizeL(RWriteStream& aStream) const
        {
        aStream.WriteInt32L(iAlertVolume);
        aStream.WriteInt8L(TInt8(iOverrideProfile));
        aStream.WriteInt8L(TInt8(iVibratingAlert));
        aStream << iAlertTone;
        };

    /**
     * Reads serialized setting data from stream.
     *
     * @param aStream  Stream from where to internalize.
     */
    inline void InternalizeL(RReadStream& aStream)
        {
        iAlertVolume = aStream.ReadInt32L();
        iOverrideProfile = TBool(aStream.ReadInt8L());
        iVibratingAlert = TBool(aStream.ReadInt8L());
        aStream >> iAlertTone;
        };

    /**
     * Brings settings data to default state.
     */
    inline void Reset()
        {
        iAlertVolume = 5;
        iOverrideProfile = EFalse;
        iVibratingAlert = EFalse;
        iAlertTone = KNullDesC;
        }

public: // Data members

    /**
     * Volume of the alert tone.
     */
    TInt iAlertVolume;

    /**
     * Flag whether to override disabled tones from profile.
     */
    TBool iOverrideProfile;

    /**
     * Flag whether to use vibrating alert or not.
     */
    TBool iVibratingAlert;

    /**
     * Filename of the alert tone.
     */
    TToneFileName iAlertTone;

    }; // TSaFilterItemSettings


// ---------------------------------------------------------------------------
// TSaSenderGroupCacheItem
// ---------------------------------------------------------------------------

/**
 * Data unit of "cached data" for sender group-based filtering.
 *
 * @note No internalize/externalize/reset method was necessary yet.
 */
class TSaSenderGroupCacheItem
    {
public: // Data members

    /** Unique contact group identifier from user's contact database. */
    TContactItemId iGroupId;

    /** Label of the group corresponding to iGroupId data member. */
    TFilterItemHeader iGroupLabel;

    }; // TSaSenderGroupCacheItem


// ---------------------------------------------------------------------------
// TSaTelnumPatternCacheItem
// ---------------------------------------------------------------------------

/**
 * Data unit of "cached data" for sender group-based filtering.
 *
 * Beside its public data members, it offers also method to externalize,
 * internalize and reset contained data.
 */
class TSaTelnumPatternCacheItem
    {
public: // Methods for using "<<" and ">>" operators with streams

    /**
     * Writes serialized unit of "cached data" for sender group-based filtering
     * to stream.
     *
     * @param aStream  Stream where to externalize.
     */
    inline void ExternalizeL(RWriteStream& aStream) const
        {
        aStream << iPatternName;
        aStream << iPatternMask;
        };

    /**
     * Reads serialized unit of "cached data" for sender group-based filtering
     * from stream.
     *
     * @param aStream  Stream from where to internalize.
     */
    inline void InternalizeL(RReadStream& aStream)
        {
        aStream >> iPatternName;
        aStream >> iPatternMask;
        };

    /**
     * Brings unit of "cached data" for sender group-based filtering
     * from stream to default state.
     */
    inline void Reset()
        {
        iPatternName = KNullDesC;
        iPatternMask = KNullDesC;
        };

public: // Data members

    TFilterItemHeader iPatternName;
    TTelnumPattern iPatternMask;

    }; // TSaTelnumPatternCacheItem


#endif // SA_DATA_MODEL_TYPES_H_JP
