/*                                                              coding: utf-8
 ============================================================================
 Name        : inc/SaDataModel.h
 Author      : Jan Pokorný <xpokor04@stud.fit.vutbr.cz>

 Description : Abstract classes related to application's data model.

 Notes:

 ============================================================================
*/


#ifndef SA_DATA_MODEL_H_JP
#define SA_DATA_MODEL_H_JP


// INCLUDES

#include "SaDataModelTypes.h"                   // related data types


// FORWARD DECLARATIONS

class CBase;                                    // <e32base.h> | euser.lib
class CContactIdArray;                          // <cntdef.h>  | cntmodel.lib


// CLASS DECLARATIONS

// ---------------------------------------------------------------------------
// MSaSenderGroupDataStorage
// ---------------------------------------------------------------------------

/**
 * Abstract mixin class providing data storage for group-based filtering.
 *
 * The interface consists of common accessors to abstract list ("storage")
 * of sender group filter items, for instance "items count", "get item at index"
 * or "add item".
 */
class MSaSenderGroupDataStorage
    {
public: // Declaration of the interface

    /**
     * Gets the number of sender group filter items in the storage.
     *
     * @return The number of sender group filter items in the storage.
     */
    virtual TUint SenderGroupCountL() const=0;

    /**
     * Cleans the storage from stored sender group filter items.
     */
    virtual void SenderGroupsResetL()=0;

    /**
     * Appends a sender group filter item to the storage.
     *
     * @param aGroupId   Unique contact group identifier from user's contact
     *                   database.
     * @param aSettings  Filter items's settings.
     */
    virtual void SenderGroupAddL( TContactItemId aGroupId,
                                  const TSaFilterItemSettings& aSettings )=0;

    /**
     * Removes a sender group filter item from the storage.
     *
     * @param aIndex  Index of item to be removed.
     */
    virtual void SenderGroupRemoveL(TUint aIndex)=0;

    /**
     * Moves a sender group filter item logically up or down in the list
     * (within the storage).
     *
     * @param aIndex  Index of item to be moved.
     * @param aUp     ETrue if the requested logical direction is "up"
     *                (i.e. to have a higher priority), EFalse for "down".
     */
    virtual void SenderGroupMoveUpDownL(TUint aIndex, TBool aUp)=0;

    /**
     * Gets the sender group filter item header at specified index
     * within the storage.
     *
     * More exactly, header means unique contact group identifier
     * from user's contact database -- considered as a group id that can
     * be matched to an SMS sender group id.
     *
     * @param aIndex    Index of item, where to get the header.
     * @param aGroupId  Filter item's header (group id).
     */
    virtual void SenderGroupGetHeaderL( TUint aIndex,
                                        TContactItemId& aGroupId ) const=0;

    /**
     * Gets the sender group filter item settings at specified index
     * within the storage.
     *
     * @param aIndex     Index of item, where to get the settings.
     * @param aSettings  Filter item's settings.
     */
    virtual void SenderGroupGetSettingsL( TUint aIndex,
                                          TSaFilterItemSettings& aSettings ) const=0;

    /**
     * Sets the sender group filter item (header and settings) at specified index
     * within the storage.
     *
     * More exactly, header means unique contact group identifier
     * from user's contact database -- considered as a group id that can
     * be matched to an SMS sender group id.
     *
     * Indexed position has to be already used.
     *
     * @param aIndex     Index of item, which to be overwritten with new data.
     * @param aGroupId   Filter item's header (group id).
     * @param aSettings  Filter item's settings.
     */
    virtual void SenderGroupSetL( TUint aIndex,
                                  const TContactItemId& aGroupId,
                                  const TSaFilterItemSettings& aSettings )=0;

    }; // MSaSenderGroupDataStorage


// ---------------------------------------------------------------------------
// MSaTelnumPatternDataStorage
// ---------------------------------------------------------------------------

/**
 * Abstract mixin class providing data storage for tel.num.pattern-based filtering.
 *
 * The interface consists of common accessors to abstract list ("storage")
 * of tel.num.pattern filter items, for instance "items count", "get item at index"
 * or "add item".
 */
class MSaTelnumPatternDataStorage
    {
public: // Declaration of the interface

    /**
     * Gets the number of tel.num.pattern filter items in the storage.
     *
     * @return The number of tel.num.pattern items in the storage.
     */
    virtual TUint TelnumPatternCountL() const=0;

    /**
     * Cleans the storage from stored tel.num.pattern filter items.
     */
    virtual void TelnumPatternResetL()=0;

    /**
     * Appends a tel.num.pattern filter item to the storage.
     *
     * @param aPatternHeader  Item's header (pattern name and pattern mask).
     * @param aSettings       Filter item's settings.
     */
    virtual void TelnumPatternAddL( const TSaTelnumPatternCacheItem& aPatternHeader,
                                    const TSaFilterItemSettings& aSettings )=0;

    /**
     * Removes a tel.num.pattern filter item from the storage.
     *
     * @param aIndex  Index of item to be removed.
     */
    virtual void TelnumPatternRemoveL(TUint aIndex)=0;

    /**
     * Moves a tel.num.pattern filter item logically up or down in the list
     * (within the storage).
     *
     * @param aIndex  Index of item to be moved.
     * @param aUp     ETrue if the requested logical direction is "up"
     *                (i.e. to have a higher priority), EFalse for "down".
     */
    virtual void TelnumPatternMoveUpDownL(TUint aIndex, TBool aUp)=0;

    /**
     * Gets the tel.num.pattern filter item header at specified index
     * within the storage.
     *
     * More exactly, header means structure consisting of pattern name
     * and pattern mask (as declared as TSaTelnumPatternCacheItem structure)
     * -- considered as a pattern that can be matched to an SMS sender's
     *    tel.number.
     *
     * @param aIndex          Index of item, where to get the header.
     * @param aPatternHeader  Filter item's header (pattern name and pattern mask).
     */
    virtual void TelnumPatternGetHeaderL( TUint aIndex,
                                          TSaTelnumPatternCacheItem& aPatternHeader ) const=0;

    /**
     * Gets the tel.num.pattern filter item settings at specified index
     * within the storage.
     *
     * @param aIndex     Index of item, where to get the settings.
     * @param aSettings  Filter item's settings.
     */
    virtual void TelnumPatternGetSettingsL( TUint aIndex,
                                            TSaFilterItemSettings& aSettings ) const=0;

    /**
     * Sets the tel.num.pattern filter item (header and settings) at specified
     * index within the storage.
     *
     * More exactly, header means structure consisting of pattern name
     * and pattern mask (as declared as TSaTelnumPatternCacheItem structure)
     * -- considered as a pattern that can be matched to an SMS sender's
     *    tel.number.
     *
     * Indexed position has to be already used.
     *
     * @param aIndex          Index of item, which to be overwritten with new data.
     * @param aPatternHeader  Filter item's header (pattern name and pattern mask).
     * @param aSettings       Filter item's settings.
     */
    virtual void TelnumPatternSetL( TUint aIndex,
                                    const TSaTelnumPatternCacheItem& aPatternHeader,
                                    const TSaFilterItemSettings& aSettings )=0;

    }; // MSaTelnumPatternDataStorage


// ---------------------------------------------------------------------------
// MSaSenderGroupDataCache
// ---------------------------------------------------------------------------

/**
 * Abstract mixin class providing non-persistent "data cache" for sender
 * group-based filtering.
 *
 * Such class is intended to serve as a wrapper for data storage interface
 * (MSaSenderGroupDataStorage), which itself is intended for data persistence,
 * by "caching" some particular data in volatile memory in order to provide
 * quick access to them (e.g. for filter items listing).
 *
 * More exactly, such particular data to cache are unique contact group
 * identifier from user's contact database and its label
 * (as declared as TSaSenderGroupCacheItem).
 *
 * Interface is extended with overloaded "item add" method, which
 * adds argument for group's label (version without it should not be used
 * in the context of "cache"), with read-only accessor to the cache array
 * and with the method looking up for the first matching sender group
 * within the cache.
 */
class MSaSenderGroupDataCache : public MSaSenderGroupDataStorage
    {
public: // Declaration of the interface

    /**
     * Appends a sender group filter item to the cache/storage, specifying also
     * group's label.
     *
     * @param aGroupId     Unique contact group identifier from user's contact
     *                     database.
     * @param aSettings    Filter items's settings.
     * @param aGroupLabel  Label of the group specified with aGroupId.
     */
    virtual void SenderGroupAddL( TContactItemId aGroupId,
                                  const TSaFilterItemSettings& aSettings,
                                  const TFilterItemHeader& aGroupLabel )=0;

    /**
     * Tries to lookup first item within sender group filter items cache
     * which matches one of specified contact groups.
     *
     * It is supposed that specified contact groups are those connected with
     * one particular tel.number -- SMS sender's tel.number.
     *
     * @param aSenderGroupIds  Contact groups to lookup within sender group
     *                         filter items cache.
     * @param aGroupLabel      Label of the first possibly matched sender group.
     * @param aSettings        Settings of the first possibly matched sender group.
     * @return ETrue and the group label + settings if the lookup was successful,
     *         EFalse otherwise.
     */
    virtual TBool FirstMatchingSenderGroupL( const CContactIdArray& aSenderGroupIds,
                                             TFilterItemHeader& aGroupLabel,
                                             TSaFilterItemSettings& aSettings) const=0;

    /**
     * Returns (read-only) array of "cached data" for sender group filter items.
     *
     * Such "cached data" item consists of unique contact group identifier
     * from user's contact database and its label
     * (as declared as TSaSenderGroupCacheItem).
     *
     * @return Array of "cached data" for sender group filter items.
     */
    virtual const CSaSenderGroupCache* SenderGroupCache() const=0;

    }; // MSaSenderGroupDataCache


// ---------------------------------------------------------------------------
// MSaTelnumPatternDataCache
// ---------------------------------------------------------------------------

/**
 * Abstract mixin class providing non-persistent "data cache" for
 * tel.num.pattern-based filtering.
 *
 * Such class is intended to serve as a wrapper for data storage interface
 * (MSaTelnumPatternDataStorage), which itself is intended for data persistence,
 * by "caching" some particular data in volatile memory in order to provide
 * quick access to them (e.g. for filter items listing).
 *
 * More exactly, such particular data to cache are pattern name and pattern mask
 * (as declared as TSaTelnumPatternCacheItem).
 *
 * Interface is extended with with read-only accessor to the cache array
 * and with the method looking up for the first matchïng tel.num.pattern
 * within the cache.
 */
class MSaTelnumPatternDataCache : public MSaTelnumPatternDataStorage
    {
public: // Declaration of the interface

    /**
     * Tries to lookup (first) item within tel.num.pattern filter items cache
     * which matches specified tel.number.
     *
     * It is supposed that specified tel.number is SMS sender's tel.number.
     *
     * @param aTelnum       Tel.number to look up within tel.num.pattern filter
     *                      items cache.
     * @param aPatternName  Name of the first possibly matched tel.num.pattern.
     * @param aSettings     Settings of the first possibly matched tel.num.pattern.
     * @return ETrue and the pattern name + settings if the lookup was successful,
     *         EFalse otherwise.
     */
    virtual TBool FirstMatchingTelnumPatternL( const TDesC& aTelnum,
                                               TFilterItemHeader& aPatternName,
                                               TSaFilterItemSettings& aSettings ) const=0;

    /**
     * Returns (read-only) array of "cached data" for tel.num.pattern filter
     * items.
     *
     * Such "cached data" item consists of pattern name and pattern mask
     * (as declared as TSaTelnumPatternCacheItem).
     *  more exactly the array
     * of pairs "name of the tel.num.pattern filter item, pattern itself".
     *
     * @return Array of pairs "cached data" for tel.num.pattern filter items.
     */
    virtual const CSaTelnumPatternCache* TelnumPatternCache() const=0;

    }; // MSaTelnumPatternDataCache


// ---------------------------------------------------------------------------
// MSaDataStorage
// ---------------------------------------------------------------------------

/**
 * Abstract mixin class combining all particular data storage providers.
 */
class MSaDataStorage : public MSaSenderGroupDataStorage
                     , public MSaTelnumPatternDataStorage
    {
    // No additional interface methods.
    }; // MSaDataStorage


// ---------------------------------------------------------------------------
// MSaDataCache
// ---------------------------------------------------------------------------

/**
 * Abstract mixin class combining all particular "data cache" providers.
 */
class MSaDataCache : public MSaSenderGroupDataCache
                   , public MSaTelnumPatternDataCache
    {
    // No additional interface methods.
    }; // MSaDataCache


// ---------------------------------------------------------------------------
// CSaDataStorageBase
// ---------------------------------------------------------------------------

/**
 * Abstract base class for handling data storage.
 */
class CSaDataStorageBase : public CBase,
                           public MSaDataStorage
    {
public:

    /**
     * Destructor.
     */
    virtual ~CSaDataStorageBase() { };

public: // Pure virtual methods to be implemented in derived class

    /**
     * Called if the owner of the class is about to begin working
     * with the storage.
     */
    virtual void OpenL()=0;

    /**
     * Called if the owner of the class wants to know, whether the storage
     * is accessible.
     *
     * @return Boolean value, whether the storage is already accessible.
     */
    virtual TBool IsOpened() const=0;

    /**
     * Called if the owner of the class is about to stop working with the storage.
     */
    virtual void Close()=0;

    }; // CSaDataStorageBase


#endif // SA_DATA_MODEL_H_JP
