//==============================================================================
/*! \file
 *
 * Author:  Premysl Krsek, krsek@fit.vutbr.cz   \n
 * File:    SmoothingModule.cpp                 \n
 * Date:    2008/10                             \n
 *
 * $Id:$
 */

#include "SmoothingModule.h"
#include "CSmoothing.h"

// MDSTk
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Image/mdsDensityVolume.h>

//#include <MDSTk/Image/mdsPixelTraits.h>
//#include <MDSTk/Image/mdsImageFunctions.h>

// STL
//#include <sstream>


//==============================================================================
/*
* Global constants.
*/

//! Module description
const std::string MODULE_DESCRIPTION    = "MDSTk based Smoothing module, implements G. Taubin smoothing algorithm.";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS      = "r:l:k";

//! Additional arguments
const std::string MODULE_ARGUMENT_REPEAT   = "r";
const std::string MODULE_ARGUMENT_LAMBDA   = "l";
const std::string MODULE_ARGUMENT_KPB      = "k";

//! Default parameters values
const int DEFAULT_REPEAT      = 10;
const double DEFAULT_LAMBDA   = 0.6307;
const double DEFAULT_KPB      = 0.1;

//==============================================================================
/*
 * Implementation of the class CSmoothingModule.
 */
CSmoothingModule::CSmoothingModule(const std::string& sDescription)
    : mds::mod::CModule(sDescription)
{
    allowArguments(MODULE_ARGUMENTS);
}

//==============================================================================

CSmoothingModule::~CSmoothingModule()
{
}

//==============================================================================

bool CSmoothingModule::startup()
{
    // Note
    MDS_LOG_NOTE("Module startup");

    // Test existence of input and output channel
    if( getNumOfInputs() != 1 || getNumOfOutputs() != 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input and output channels" << std::endl);
        return false;
    }

    // Parameters 'r'
    m_iRepeat = DEFAULT_REPEAT;
    m_Arguments.value(MODULE_ARGUMENT_REPEAT, m_iRepeat);
    if( m_iRepeat <= 0 )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'r' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // Parameter 'l'
    m_dLambda = DEFAULT_LAMBDA;
    m_Arguments.value(MODULE_ARGUMENT_LAMBDA, m_dLambda);
    if( m_dLambda < 0 )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'l' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // Parameter 'k'
    m_dKpb = DEFAULT_KPB;
    m_Arguments.value(MODULE_ARGUMENT_KPB, m_dKpb);
    if( m_dKpb < 0 )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'k' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // O.K.
    return true;
}

//==============================================================================

bool CSmoothingModule::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // I/O channels
    mds::mod::CChannel *pIChannel = getInput(0);
    mds::mod::CChannel *pOChannel = getOutput(0);

    // Is any input?
    if( !pIChannel->isConnected() )
    {
        return false;
    }

    // Create a new triangle mesh
    vctl::MCTriS                tri_mesh;

    // Wait for data
    if( pIChannel->wait(1000) )
    {
        // Read input triangle mesh from the input channel
        if( readInput(pIChannel, &tri_mesh) )
        {
            // create smoothing algorithm class
            CSmoothing              smoothing;   

            // smoothing of input tri mesh
            smoothing.registerProgressFunc(mds::mod::CProgress::tProgressFunc(this, &CSmoothingModule::ProgressFunction));

            // start smoothing of triangle meshes
            if ( smoothing.Smooth(tri_mesh, m_iRepeat, m_dLambda, m_dKpb) )
            {
                // Write smoothed triangle mesh to the output channel
                if( !writeOutput(pOChannel, &tri_mesh) )
                {
                    MDS_CERR('<' << m_sFilename << "> Failed to write the output smoothed MDSTk::VectorEntity triangle mesh" << std::endl);
                }
            }
            else
            {
                MDS_CERR(" Smoothing process failed! " << std::endl);
            }

        }
        else
        {
            MDS_CERR('<' << m_sFilename << "> Failed to read input MDSTk::VectorEntity Triangle mesh" << std::endl);
            return false;
        }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Returning true means to continue processing the input channel
    return true;
}

//==============================================================================

void CSmoothingModule::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}

//==============================================================================

void CSmoothingModule::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR(std::endl);
    MDS_CERR("Extended usage: [-r iValue] [-l fValue] [-k fValue]" << std::endl);
    MDS_CERR("Options:" << std::endl);
    MDS_CERR("  -r    Number of smoothing iterations." << std::endl);
    MDS_CERR("        Default value is " << DEFAULT_REPEAT << std::endl);
    MDS_CERR(std::endl);
    MDS_CERR("  -l    Positive scale factor, by G. Taubin smoothing algorithm." << std::endl);
    MDS_CERR("        Default value is " << DEFAULT_LAMBDA << std::endl);
    MDS_CERR("        Scale factor value have to be positive." << std::endl);
    MDS_CERR(std::endl);
    MDS_CERR("  -k    Filter pass-band frequency, by G. Taubin smoothing algorithm." << std::endl);
    MDS_CERR("        Default value is " << DEFAULT_KPB << std::endl);
    MDS_CERR("        Pass-band frequency value have to be positive." << std::endl);
}

//==============================================================================
/*
 * Function main() which creates and executes the console application.
 */
int main(int argc, char *argv[])
{
    // Creation of the module using smart pointer
    CSmoothingModulePtr spModule(new CSmoothingModule(MODULE_DESCRIPTION));

    // Initialize and execute the module
    if( spModule->init(argc, argv) )
    {
        spModule->run();
    }

    // Console application finished
    return 0;
}

