//==============================================================================
/*! \file
 *
 * Author:  Premysl Krsek, krsek@fit.vutbr.cz   \n
 * File:    MCModule.cpp                        \n
 * Date:    2008/9                              \n
 *
 * $Id:$
 * - Sample module.
 */

#include "MarchingCubesModule.h"
#include "CMarchingCubes.h"

#include <time.h>

// MDSTk
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Image/mdsDensityVolume.h>

//#include <MDSTk/Image/mdsPixelTraits.h>
//#include <MDSTk/Image/mdsImageFunctions.h>

// STL
//#include <sstream>


//==============================================================================
/*
 * Global constants.
 */

//! Module description
const std::string MODULE_DESCRIPTION    = "MDSTk based Marching cubes module.";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS      = "lo:hi:reduce";

//! Additional arguments
const std::string MODULE_ARGUMENT_LOW       = "lo";
const std::string MODULE_ARGUMENT_HIGH      = "hi";
const std::string MODULE_ARGUMENT_REDUCE    = "reduce";

const int ALLOWED_MIN   = mds::img::CPixelTraits<mds::img::tDensityPixel>::getPixelMin();
const int ALLOWED_MAX   = mds::img::CPixelTraits<mds::img::tDensityPixel>::getPixelMax();

//! Default parameters values
const int DEFAULT_LOW   = 1000;
const int DEFAULT_HIGH  = ALLOWED_MAX;

//==============================================================================
/*
 * Implementation of the class CMarchinCubesModule.
 */
CMarchinCubesModule::CMarchinCubesModule(const std::string& sDescription)
    : mds::mod::CModule(sDescription)
{
    allowArguments(MODULE_ARGUMENTS);
}

//==============================================================================

CMarchinCubesModule::~CMarchinCubesModule()
{
}

//==============================================================================

bool CMarchinCubesModule::startup()
{
    // Note
    MDS_LOG_NOTE("Module startup");

    // Test existence of input and output channel
    if( getNumOfInputs() != 1 || getNumOfOutputs() != 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input and output channels" << std::endl);
        return false;
    }

    // Parameter 'lo'
    m_iLow = DEFAULT_LOW;
    m_Arguments.value(MODULE_ARGUMENT_LOW, m_iLow);
    if ( (m_iLow < ALLOWED_MIN) || (m_iLow > ALLOWED_MAX) )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'l' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // Parameter 'hi'
    m_iHigh = DEFAULT_HIGH;
    m_Arguments.value(MODULE_ARGUMENT_HIGH, m_iHigh);
    if ( (m_iHigh < ALLOWED_MIN) || (m_iHigh > ALLOWED_MAX) )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'h' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }
    if (m_iHigh < m_iLow)
    {
        MDS_CERR('<' << m_sFilename << "> Parameter h < l : type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // Parametr "reduce"
    if ( m_Arguments.exists(MODULE_ARGUMENT_REDUCE) )
    {
        m_bReduce = true;
    }
    else
    {
        m_bReduce = false;
    }

    // O.K.
    return true;
}

//==============================================================================

bool CMarchinCubesModule::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // I/O channels
    mds::mod::CChannel *pIChannel = getInput(0);
    mds::mod::CChannel *pOChannel = getOutput(0);

    // Is any input?
    if( !pIChannel->isConnected() )
    {
        return false;
    }

    // Create a new density volume
    mds::img::CDensityVolumePtr spVolume(new mds::img::CDensityVolume());
    // Create a new triangle mesh
    vctl::MCTriS                tri_mesh;

    // Wait for data
    if( pIChannel->wait(1000) )
    {
        // Read density volume from the input channel
        if( readInput(pIChannel, (mds::img::CDensityVolume *) spVolume) )
        {
            // create marching cubes algorithm class
            CMarchingCubes          marching_cubes;   
            // create volume thresholding functor
            CThresholdFunctor<mds::img::CDensityVolume, mds::img::tDensityPixel>       thresholding_functor(m_iLow, m_iHigh);

            marching_cubes.registerProgressFunc(mds::mod::CProgress::tProgressFunc(this, &CMarchinCubesModule::ProgressFunction));

            // measure processing time
            time_t timer_1, timer_2;
            time(&timer_1);

            // start marching cubes generation of triangle meshes
            if ( marching_cubes.GenerateMesh(*spVolume, tri_mesh, thresholding_functor, false) )
            {
                // reduce output mesh on flat regions
                if (m_bReduce)
                {
                    if ( ! marching_cubes.ReduceFlatAreas(tri_mesh))
                    {
                        MDS_CERR("Failed reduction of vertices on flat regions!" << std::endl);
                        return false;
                    }
                }

                // Print output mesh parameters
                MDS_CERR( std::endl << "Output mesh has " << tri_mesh.GetNumber() << "Tris and " << tri_mesh.GetVerticeS()->GetNumber() << " Vertices." << std::endl );

                // calculate processing time
                time(&timer_2);
                MDS_CERR("\nProcessing time is: " << (timer_2 - timer_1) << "\n");

                // Write created triangle mesh to the output channel
                if( !writeOutput(pOChannel, &tri_mesh) )
                {
                    MDS_CERR('<' << m_sFilename << "> Failed to write the output triangle mesh" << std::endl);
                }
            }
            else
            {
                MDS_CERR(" Marching Cubes process failed, created MDSTk::VectorEntity tri mesh is not regular! " << std::endl);
            }

        }
        else
        {
            MDS_CERR('<' << m_sFilename << "> Failed to read input Density Volume data" << std::endl);
            return false;
        }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Returning true means to continue processing the input channel
    return true;
}

//==============================================================================

void CMarchinCubesModule::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}

//==============================================================================

void CMarchinCubesModule::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR(std::endl);
    MDS_CERR("Extended usage: [-lo iValue] [-hi iValue] -reduce" << std::endl);
    MDS_CERR("Options:" << std::endl);
    MDS_CERR("  -lo    Low threshold for Marching Cubes voxels segmentation." << std::endl);
    MDS_CERR("        Default value is " << DEFAULT_LOW << std::endl);
    MDS_CERR(std::endl);
    MDS_CERR("  -hi    High threshold for Marching Cubes voxels segmentation." << std::endl);
    MDS_CERR("        Default value is " << DEFAULT_HIGH << std::endl);
    MDS_CERR(std::endl);
    MDS_CERR("        Threshold range is from " << ALLOWED_MIN << " to " << ALLOWED_MAX << std::endl);
    MDS_CERR(std::endl);
    MDS_CERR("  -reduce  Reduce output mesh vertices on straight edges and on flat regions." << std::endl);
}


//==============================================================================
/*
 * Function main() which creates and executes the console application.
 */
int main(int argc, char *argv[])
{
    // Creation of the module using smart pointer
    CMarchinCubesModulePtr spModule(new CMarchinCubesModule(MODULE_DESCRIPTION));

    // Initialize and execute the module
    if( spModule->init(argc, argv) )
    {
        spModule->run();
    }

    // Console application finished
    return 0;
}

