
#include <stdio.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#include <opencv/cv.h>
#include "vplcore.h"
#include "vplgllf2dgridhist.h"


int vplGllf2DGridHistClear( struct VplGllf2DGridHist * gh );


struct VplGllf2DGridHist * vplGllf2DGridHistCreate( void )
{
	struct VplGllf2DGridHist * gh = NULL;

	CV_FUNCNAME( "vplGllf2DGridHistCreate" );
	__BEGIN__

	gh = ( VplGllf2DGridHist * )malloc( sizeof( VplGllf2DGridHist ) );

	gh->dims = 0;
	gh->grid = cvSize( 0, 0 );
	gh->gridOverlap = (float)1.15;
	gh->hist = NULL;
	gh->sizes = NULL;
	gh->featureSize = 0;
	gh->histSize = 0;
	gh->planes = NULL;

	/* optimized for parallel processing */ 
	gh->nt = 1;

	#ifdef _OPENMP
	#pragma omp parallel 
		gh->nt = omp_get_num_threads();
	#endif

	__END__
	return gh;
}


struct VplGllf2DGridHist * vplGllf2DGridHistInit( struct VplGllf2DGridHist * gh, CvSize grid, float gridOverlap, int dims, int * sizes, float ** ranges )
{
	CV_FUNCNAME( "vplGllf2DGridHistInit" );
	__BEGIN__

	int i;

	if( !sizes || !ranges ) EXIT;
	if( grid.width <= 0 || grid.height <= 0 ) EXIT;
	if( dims <= 0 ) EXIT;

	if( !gh ) gh = vplGllf2DGridHistCreate();
	else           vplGllf2DGridHistClear( gh );

	gh->hist = ( CvHistogram ** )malloc( sizeof( CvHistogram * ) * grid.width * grid.height );
	if( !gh->hist ) EXIT;

	for( i = 0; i < grid.width * grid.height; ++i )
		gh->hist[i] = cvCreateHist( dims, sizes, CV_HIST_ARRAY, ranges, 1 );

	gh->dims = dims;
	gh->grid = grid;
	gh->gridOverlap = gridOverlap;
	gh->sizes = (int *)malloc( sizeof(int) * dims );
	memcpy( gh->sizes, sizes, sizeof(int) * dims );

	gh->histSize = 1;
	for( i = 0; i < dims; ++i ) gh->histSize *= sizes[i];

	gh->planes = (CvMat **)malloc( sizeof(CvMat *) * gh->dims * gh->nt );	
	for( i = 0; i < dims*gh->nt; ++i ) gh->planes[i] = cvCreateMatHeader( 1, 1, CV_8UC1 );

	gh->featureSize = gh->histSize * grid.width * grid.height;

	return gh;

	__END__

	vplGllf2DGridHistRelease( &gh );
	return NULL;
}



int	vplGllf2DGridHistExtract( struct VplGllf2DGridHist * gh, CvMat ** images, CvMat * feature )
{
	CV_FUNCNAME( "vplGllf2DGridHistExtract" );
	__BEGIN__

	int k, iHist;
	CvSize wnd;
	CvRect w;
	CvMat hm;

	if( !gh || !gh->hist || !images || !*images ) EXIT;

	wnd = cvSize( cvRound( gh->gridOverlap * images[0]->cols / gh->grid.width ), cvRound( gh->gridOverlap * images[0]->rows / gh->grid.height ) );

	#ifdef _OPENMP
	#pragma omp parallel default(shared) private(k,iHist,w,hm)
	#endif
	{
		CvMat ** planes = gh->planes;
		#ifdef _OPENMP
		planes = gh->planes + gh->nt*(int)omp_get_thread_num();
		#pragma omp for schedule(dynamic) 
		#endif
		for( iHist = 0; iHist < gh->grid.height*gh->grid.width; ++iHist )
		{
			w = cvRect( (iHist % gh->grid.width) * wnd.width, (iHist / gh->grid.width) * wnd.height, wnd.width, wnd.height );

			if( w.x+w.width  >= images[0]->cols ) w.width  = images[0]->cols - w.x;
			if( w.y+w.height >= images[0]->rows ) w.height = images[0]->rows - w.y;

			for( k = 0; k < gh->dims; ++k ) 
				cvGetSubRect( images[k], planes[k], w );

			cvCalcHist( (IplImage **)planes, gh->hist[iHist], 0, 0 );
			
			cvGetMat( gh->hist[iHist]->bins, &hm, 0, 1 );
			cvConvertScale( &hm, &hm, 1./(w.width*w.height), 0.0 );

			if( feature && feature->cols == gh->featureSize )
			{
				CvMat hmr, hf;
				hmr = cvMat( 1, hm.cols*hm.rows, hm.type, hm.data.ptr );
				cvGetSubRect( feature, &hf, cvRect( iHist * gh->histSize, 0, gh->histSize, 1 ) );
				cvCopy( &hmr, &hf, 0 );
			}

		}
	}

	return 1;
	__END__
	return -1;
}


int	vplGllf2DGridHistGetSize( struct VplGllf2DGridHist * gh )
{
	if( !gh ) return 0;
	return gh->featureSize;
}


int	vplGllf2DGridHistGetFeature( struct VplGllf2DGridHist * gh, CvMat * feature )
{
	CV_FUNCNAME( "vplGllf2DGridHistExtract" );
	__BEGIN__

	int iHist;
	CvMat hm, hmr, hf;

	if( !gh || !feature ) EXIT;
	if( feature->cols != gh->featureSize ) EXIT;

	for( iHist = 0; iHist < gh->grid.height*gh->grid.width; ++iHist )
	{
		cvGetMat( gh->hist[iHist]->bins, &hm, 0, 1 );
		hmr = cvMat( 1, hm.cols*hm.rows, hm.type, hm.data.ptr );
		cvGetSubRect( feature, &hf, cvRect( iHist * gh->histSize, 0, gh->histSize, 1 ) );
		cvCopy( &hmr, &hf, 0 );
	}

	return 1;
	__END__
	return -1;
}



int	vplGllf2DGridHistRender( struct VplGllf2DGridHist * gh, CvArr * canvas )
{
	CV_FUNCNAME( "vplGllf2DGridHistRender" );
	__BEGIN__

    CvMat srcstub, *src = (CvMat*)canvas;
	int i, j, iHist;
	double xs, ys;
	CvSize wnd;
	CvRect w;

	if( !gh || !canvas || gh->dims != 2 ) return -1;

	src = cvGetMat( canvas, &srcstub, &i, 0 );
	cvZero( src );
	wnd = cvSize( cvRound( src->cols / gh->grid.width ), cvRound( src->rows / gh->grid.height ) );

	for( j = 0; j < gh->grid.height; ++j )
	{
		for( i = 0; i < gh->grid.width; ++i )
		{
			int h, s;
			CvMat m;
			float max_value;

			w = cvRect( i * wnd.width, j * wnd.height, wnd.width, wnd.height );
			iHist = i + j * gh->grid.width;
			w = cvRect( i * wnd.width, j * wnd.height, wnd.width, wnd.height );
			iHist = i + j * gh->grid.width;

			if( w.x+w.width  >= src->cols ) w.width  = src->cols - w.x;
			if( w.y+w.height >= src->rows ) w.height = src->rows - w.y;

			cvGetSubRect( src, &m, w );
			
			cvGetMinMaxHistValue( gh->hist[iHist], 0, &max_value, 0, 0 );

			xs = 1.0 * w.width  / gh->sizes[0];
			ys = 1.0 * w.height / gh->sizes[1];
			for( s = 0; s < gh->sizes[1]; s++ )
			{
				for( h = 0; h < gh->sizes[0]; h++ )
				{
					float bin_val = cvQueryHistValue_2D( gh->hist[iHist], h, s );
					//int val = cvRound( bin_val * 2.55 * max_value );
					int val = (int)( 255. * bin_val/max_value );
					cvRectangle( &m, cvPoint( (int)(h*xs), (int)(s*ys) ), cvPoint( (int)((h+1)*xs), (int)((s+1)*ys)), CV_RGB(val,val,val), CV_FILLED, 8, 0 );
				}
			}
			cvRectangle( &m, cvPoint( 0, 0 ), cvPoint( w.width-1, w.height-1 ), CV_RGB(100,100,200), 1, 8, 0 );
		}
	}
	cvRectangle( src, cvPoint( 0, 0 ), cvPoint( src->cols-1, src->rows-1 ), CV_RGB(100,100,200), 3, 8, 0 );

	__END__
	return 1;
}



int vplGllf2DGridHistClear( struct VplGllf2DGridHist * gh )
{
	CV_FUNCNAME( "vplGllf2DGridHistClear" );
	__BEGIN__

	int i;
	if( !gh ) return -1;

	if( gh->hist )
	{
		for( i = 0; i < gh->grid.width * gh->grid.height; ++i )
			cvReleaseHist( &gh->hist[i] );
		free( gh->hist );
		gh->hist = NULL;
	}

	if( gh->planes )
	{
		for( i = 0; i < gh->dims*gh->nt; ++i ) cvReleaseMat( &gh->planes[i] );
		free( gh->planes );
	}

	if( gh->sizes ) free( gh->sizes );
	gh->sizes = NULL;
	
	__END__
	return 1;
}


int vplGllf2DGridHistRelease( struct VplGllf2DGridHist ** gh )
{
	CV_FUNCNAME( "vplGllf2DGridHistRelease" );
	__BEGIN__

	struct VplGllf2DGridHist * _gh;

	if( !gh || !*gh ) return 1;

	_gh = *gh;

	vplGllf2DGridHistClear( _gh );

	free( _gh );
	*gh = NULL;

	__END__
	return 1;
}
