/**
 * OpenCV library extension of FFMPEG
 *
 * Author: Petr Chmelar (c) 2008 
 * License: GNU/GPL and Intel CV
 */

#include "cvffmpeg.h"


/**
 * Constructor
 */
ffMedia* newMedia() {
    ffMedia* media = alloc(ffMedia, 1);
    if (media == null) return null;
    
    media->formatCtx = null;
    media->videoStream = -1;
    media->codecCtx = null;
    media->codec = null;
    
    media->ffEof = true;

    media->ffFrame = null;
    
#ifdef FFMPEG_SWSCALE_H
    media->sws = null;
#endif // SWS
    
    return media;
}


/**
 * Closes a media file
 */
void ffClose(ffMedia* media) {
    if (media != null) {
        media->ffEof = true;

        #ifdef FFMPEG_SWSCALE_H
            ffSWSClose(media->sws);
            free(media->sws);
            media->sws = null;
        #endif
        
        // Free the YUV frame
        av_free(media->ffFrame);
        media->ffFrame = null;

        // Close the codec
        if (media->codecCtx != null) {
            avcodec_close(media->codecCtx);
            media->codecCtx = null;
        }

        // Close the video file
        if (media->formatCtx != null) {
            av_close_input_file(media->formatCtx);
            media->formatCtx = null;
        }

        // Cancel the video stream
        media->videoStream = -1;
    }
};


/**
 * Opens a media file and returns true if succeed
 */
boolean ffOpenFile(ffMedia* media, const char* filename) {
    int i;      // just i
    
    // Check the media structure
    if (media == null) {
        return false;
    }
    
    // Register all formats and codecs
    av_register_all();
    
    // Open video file
    if(av_open_input_file(&(media->formatCtx), filename, NULL, 0, NULL) != 0)
        return false; // Couldn't open file
    
    // Retrieve stream information
    if(av_find_stream_info(media->formatCtx) < 0)
        return false; // Couldn't find stream information
    
    // Dump information about file onto standard error
    dump_format(media->formatCtx, 0, filename, false);
    
    // Find the first video stream
    for (i=0; i < media->formatCtx->nb_streams; i++)
        if(media->formatCtx->streams[i]->codec->codec_type == CODEC_TYPE_VIDEO) {
            media->videoStream = i;
            break;
        }
    if (media->videoStream == -1) return false; // Didn't find a video stream
    
    // Get a pointer to the codec context for the video stream
    media->codecCtx = media->formatCtx->streams[media->videoStream]->codec;
    
    // Find the decoder for the video stream
    media->codec = avcodec_find_decoder(media->codecCtx->codec_id);
    if(media->codec == NULL) return false; // Codec not found
    
    // Open codec
    if(avcodec_open(media->codecCtx, media->codec) < 0) return false; // Could not open codec
    
    // Hack to correct wrong frame rates that seem to be generated by some codecs
    if(media->codecCtx->time_base.den > 1000 && media->codecCtx->time_base.num == 1)
        media->codecCtx->time_base.num = 1000;
    
    // Allocate video frame to ffFrame
    media->ffFrame = avcodec_alloc_frame();
    if(media->ffFrame == null) return false;
    
#ifdef FFMPEG_SWSCALE_H
    media->sws = ffSWSInit(media->codecCtx->width, media->codecCtx->height, PIX_FMT_RGBA32, SWS_FAST_BILINEAR);
#endif // SWS
    
    return true;
};


/*
Frame Grabbing (Video4Linux and IEEE1394)
Toru Tamaki sent me some sample code that demonstrates how to grab frames from a Video4Linux or IEEE1394 video source using libavformat / libavcodec. For Video4Linux, the call to av_open_input_file() should be modified as follows:

    // TODO:
    boolean ffOpenCapture();
    boolean ffOpenIEEE1349();


AVFormatParameters formatParams;
AVInputFormat *iformat;

formatParams.device = "/dev/video0";
formatParams.channel = 0;
formatParams.standard = "ntsc";
formatParams.width = 640;
formatParams.height = 480;
formatParams.frame_rate = 29;
formatParams.frame_rate_base = 1;
filename = "";
iformat = av_find_input_format("video4linux");

av_open_input_file(&ffmpegFormatContext,
                 filename, iformat, 0, &formatParams);

For IEEE1394, call av_open_input_file() like this:

AVFormatParameters formatParams;
AVInputFormat *iformat;

formatParams.device = "/dev/dv1394";
filename = "";
iformat = av_find_input_format("dv1394");

av_open_input_file(&ffmpegFormatContext,
                 filename, iformat, 0, &formatParams);

*/



/**
 * Return the next frame or null if finished
 */
AVFrame* ffAvFrame(ffMedia* media) {
    AVPacket    packet;
    boolean     frameFinished;
    
    media->ffEof = true;
    
    // Read frames
    while(av_read_frame(media->formatCtx, &packet) >= 0) {
        // Is this a packet from the video stream?
        if(packet.stream_index == media->videoStream) {
            // Decode video frame
            avcodec_decode_video(media->codecCtx, media->ffFrame, &frameFinished, packet.data, packet.size);
            // Free the packet that was allocated by av_read_frame
            av_free_packet(&packet);
            
            // Did we get a video frame?
            if(frameFinished) {
                if(media->ffFrame->data == null) return null;
                
                media->ffEof = false;
                return media->ffFrame;
            }
        }
        // Process audio etc...
        else {
            // Free the packet that was allocated by av_read_frame
            av_free_packet(&packet);
        }
        
    }
    
    return null;
};


/**
 * Returns the actual position
 */
unsigned long ffPosition(ffMedia* media) {
    return media->codecCtx->frame_number;
    // return (unsigned long)ROUND(((double)media->formatCtx->timestamp / (double)AV_TIME_BASE) / (double)av_q2d(media->codecCtx->time_base));
}


/**
 * Returns the ESTIMATED number of frames in the video (NOT EXACT!)
 */
unsigned long ffLength(ffMedia* media) {

    return (unsigned long)ROUND(((double)media->formatCtx->duration / (double)AV_TIME_BASE) / (double)av_q2d(media->codecCtx->time_base));
}

/**
 * Seek to a specified position and return the frame
 */
AVFrame* ffAvFrameSeek(ffMedia* media, unsigned long position) {
    int64_t     timestamp;
    int64_t     MyPts;
    
    AVPacket    packet;
    boolean     frameFinished;
    
    
    // Conversion of int64_t position value into unsigned long timestamp for seeking using av_seek_frame        
    timestamp = (int64_t)ROUND((double)position * (double)AV_TIME_BASE * (double)av_q2d(media->codecCtx->time_base));

    // check the frame number
    printf("%-15d /%-15d", position, timestamp);


    if (av_seek_frame(media->formatCtx, -1, timestamp , AVSEEK_FLAG_ANY) < 0) return null;
    else {
        // the loop here was useless and it was deleted
        avcodec_flush_buffers(media->codecCtx);
        media->codecCtx->frame_number = position;        
        return ffAvFrame(media);
    }

};

/**
 * Seek to a specified position and return the frame
 */
AVFrame* ffAvFramePreciseSeek(ffMedia* media, unsigned long position) {
    // if actual position
    if (position == ffPosition(media)) return media->ffFrame;
    // if previous
    if (position < ffPosition(media)) {
        if (av_seek_frame(media->formatCtx, -1, 1 , AVSEEK_FLAG_BACKWARD) < 0) return null;
        else {
            avcodec_flush_buffers(media->codecCtx);
            media->codecCtx->frame_number = 0;            
        }
    }
    
    // go to the specified position (next)
    media->codecCtx->hurry_up = 1;
    while(ffPosition(media) < position && !media->ffEof) {
        ffAvFrame(media);
    };
    media->codecCtx->hurry_up = 0;
    
    return ffAvFrame(media); 
}


#ifdef FFMPEG_SWSCALE_H

// SWS-init function
ffSWS* ffSWSInit(int w, int h, int f, int t) {
    
    ffSWS* sws = alloc(ffSWS, 1);
    if (sws == null) return null;
    
    sws->width = w;
    sws->height = h; 
    sws->format = f;
    sws->flag = t;
    
    sws->imgConvertCtx = null;
    
    // Allocate video frame to swsFrame
    sws->swsFrame = avcodec_alloc_frame();
    if(sws->swsFrame == null) return false;
    
    // Determine required buffer size and allocate buffer
    sws->numBytes = avpicture_get_size(f, w, h);
    sws->buffer = alloc(uint8_t, sws->numBytes);
    
    // Assign appropriate parts of buffer to image planes in media->swsFrame
    avpicture_fill((AVPicture *)sws->swsFrame, sws->buffer, f, w, h);

#ifdef _CV_H_    
    // You can use OpenCV frame only in case of PIX_FMT_RGBA32
    if (f == PIX_FMT_RGBA32) {
        // create cvimage header
        sws->cvFrame = cvCreateImageHeader( cvSize(w, h), 8, 4 );
    }
    else sws->cvFrame = null;
#endif // CV
    
    return sws;

}

// SWS-delete function
void ffSWSClose(ffSWS* sws) {

    sws->width = 0;
    sws->height = 0;
    sws->format = 0;
    sws->flag = 0;

    sws->numBytes = 0;

    #ifdef _CV_H_
    // Release cvFrame header
    if (sws->cvFrame != null) {
        cvReleaseImageHeader(&sws->cvFrame);
        sws->cvFrame = null;
    }
    #endif // CV

    // Free the SWS frame            
    if (sws->swsFrame != null) {
        av_free(sws->swsFrame);
        sws->swsFrame = null;
    }

    if (sws->buffer != null) {
        free(sws->buffer);
        sws->buffer = null;
    }

    if (sws->imgConvertCtx != null) {
        sws_freeContext(sws->imgConvertCtx);
        sws->imgConvertCtx = null;                    
    }

}


/**
 * Convert the actual frame to the IplImage or return null if none
 */
AVFrame* ffConvert(ffMedia* media, ffSWS* sws) {
    // check the frame
    if(media->ffFrame == null || media->ffFrame->data == null) {
        return null;
    }
    
    if (sws == null) sws = media->sws;
/*
    if (width != media->codecCtx->width || height != media->codecCtx->width) {
        if (width == 0 || height == 0) {
            width = media->codecCtx->width;
            height = media->codecCtx->height;
        } 
        else { // fuck off the old SWS
            ffSWSInit(media, width, height, format);
        }
    }
*/    
    // free the convert context
    if (sws->imgConvertCtx != null) sws_freeContext(sws->imgConvertCtx);

    // create it
    sws->imgConvertCtx = sws_getContext(media->codecCtx->width,
            media->codecCtx->height, media->codecCtx->pix_fmt,
            sws->width, sws->height, sws->format, sws->flag, NULL, NULL, NULL);   // SWS_BICUBIC

    // transform the image
    sws_scale(sws->imgConvertCtx, media->ffFrame->data, media->ffFrame->linesize, 0,
            media->codecCtx->height, sws->swsFrame->data, sws->swsFrame->linesize);

#ifdef _CV_H_
    if (sws->cvFrame != null && sws->format == PIX_FMT_RGBA32) {
        // set data to the cvFrame
        cvSetData(sws->cvFrame, sws->swsFrame->data[0], sws->swsFrame->linesize[0]);
    }
#endif
    
    return media->sws->swsFrame;
}




#ifdef _CV_H_
/**
 * Return the next frame or null if finished
 */
    IplImage* ffCvFrame(ffMedia* media) {
    // get next frame in YUV
    ffAvFrame(media);
    // return the IplImage
    return ffCvConvert(media, null);
};

/**
 * Seek to a specified position and return the frame
 */
IplImage* ffCvFrameSeek(ffMedia* media, unsigned long position) {
    // get next frame in YUV
    ffAvFrameSeek(media, position);
    // return the IplImage
    return ffCvConvert(media, null);
}

/**
 * Seek to a specified position (precisely but slowly) and return the frame
 */
IplImage* ffCvFramePreciseSeek(ffMedia* media, unsigned long position) {
    // get next frame in YUV
    ffAvFramePreciseSeek(media, position);
    // return the IplImage
    return ffCvConvert(media, null);
}


/**
 * Convert the actual frame to the IplImage or return null if none
 */
IplImage* ffCvConvert(ffMedia* media, ffSWS* sws) {
    // convert the YUV frame
    ffConvert(media, sws);
    // and return the IplImage
    return media->sws->cvFrame;
}
#endif // CV
#endif // SWS





