// Clustering.cpp : Defines the entry point for the console application.

#include <vector>
#include <fstream>
#include <iostream>
#include <sstream>
#include <cmath>

#include <cv.h>
#include <ml.h>
#include <highgui.h>

const double PI = 3.141592653589793238462643383279502884197169399375;

using namespace std;

void getProbabilities( CvMat* labels, long double &log_likelihood, CvMat * &probabilities, const CvMat * dataVectors,const CvMat* means, const CvMat** covs, const CvMat* weights);

double computeCoherence( const CvMat * labels);

int maxIteration = 20;
int maxEMIteration = 15;
int clusterLength = 25;
int PCADimensionsUsed = 8;
vector< int> mixtureRestrictions( 3, 256);

void getParams(ifstream &in, int *vectorLength, int *frameCount)
{
	*vectorLength = 0;
	*frameCount=1;
	string str;
	getline(in, str);
	int start = 1;
	while( (start=str.find(',', start)) != string::npos )
	{
		(*vectorLength)++;
		start++;
	}
	while( !getline(in,str).eof() )
		(*frameCount)++;
	//cout << "debug: vectorLength=" << *vectorLength << " frameCount=" << *frameCount << endl;
	in.clear();
	in.seekg (0, ios::beg);
}

int main(int argc, char* argv[])
{
	mixtureRestrictions[0] = CvEM::COV_MAT_SPHERICAL;

	cvNamedWindow( "Result");

	vector< string> featureFileNames;
	string unwantedFramesFileName;
	string outputFileName;

	// Read TrainTool command line parameters
	for( int i = 1; i < argc; i++) {
	if( string( argv[ i]) == "-f") {
			featureFileNames.push_back( argv[++i]);
	} else if( string( argv[ i]) == "-o") {
		outputFileName = argv[ ++i];
	} else if( string( argv[ i]) == "-u") {
		unwantedFramesFileName = argv[ ++i];
	} else if( string( argv[ i]) == "-maxIteration") {
			stringstream str( argv[ ++i]);
			str >> maxIteration;
	} else if( string( argv[ i]) == "-PCADimensions") {
			stringstream str( argv[ ++i]);
			str >> PCADimensionsUsed;
	} else if( string( argv[ i]) == "-clusterLength") {
			stringstream str( argv[ ++i]);
			str >> clusterLength;
		} else if( string( argv[ i]) == "-maxEMIteration") {
			stringstream str( argv[ ++i]);
			str >> maxEMIteration;
	} else if( string( argv[ i]) == "-firstGauss") {
			string str( argv[ ++i]);
			if( str == "COV_MAT_SPHERICAL") {
				mixtureRestrictions[ 0] = CvEM::COV_MAT_SPHERICAL;
			} else if( str == "COV_MAT_DIAGONAL") {
				mixtureRestrictions[ 0] = CvEM::COV_MAT_DIAGONAL;
			} else if( str == "COV_MAT_GENERIC") {
				mixtureRestrictions[ 0] = CvEM::COV_MAT_GENERIC;
			} else {
				cerr << "Warning: Gaussian mixture restriction not recognized \"" << str << "\"." << endl;
			}
		} else if( string( argv[ i]) == "-secondGauss") {
			string str( argv[ ++i]);
			if( str == "COV_MAT_SPHERICAL") {
				mixtureRestrictions[ 1] = CvEM::COV_MAT_SPHERICAL;
			} else if( str == "COV_MAT_DIAGONAL") {
				mixtureRestrictions[ 1] = CvEM::COV_MAT_DIAGONAL;
			} else if( str == "COV_MAT_GENERIC") {
				mixtureRestrictions[ 1] = CvEM::COV_MAT_GENERIC;
			} else {
				cerr << "Warning: Gaussian mixture restriction not recognized \"" << str << "\"." << endl;
			}
		} else if( string( argv[ i]) == "-thirdGauss") {
			string str( argv[ ++i]);
			if( str == "COV_MAT_SPHERICAL") {
				mixtureRestrictions[ 2] = CvEM::COV_MAT_SPHERICAL;
			} else if( str == "COV_MAT_DIAGONAL") {
				mixtureRestrictions[ 2] = CvEM::COV_MAT_DIAGONAL;
			} else if( str == "COV_MAT_GENERIC") {
				mixtureRestrictions[ 2] = CvEM::COV_MAT_GENERIC;
			} else {
				cerr << "Warning: Gaussian mixture restriction not recognized \"" << str << "\"." << endl;
			}
	} else if( strcmp(argv[i],"-h") == 0) {
			cout << "clustering.exe [options]" << endl;
			cout << "	options:" << endl;
			cout << "	-h				this help" << endl;
			cout << "	-f file			input feature file. This option can be specified multiple times - all files are then used." << endl;
			cout << "	-o file			output file - here the final results will be stored." << endl;
			cout << "	-u file			file with specification of unwanted frames" << endl;
			cout << "	-maxIteration	The number of clustering attemps. - The best is selected as result." << endl;
			cout << "	-PCADimensions	The number of dimensions after PCA transformation. " << endl;
			cout << "	-clusterLength  The number of frames in one cluster. This is used only to choose" << endl;
			cout << "					the number of clusters that will be generated. " << endl;
			cout << "					clusterNo = frame_count * 0.02 /clusterLength " << endl;
			cout << "	-firstGauss		Restriction on gaussin mixture in the frist iterations of ME alg. " << endl;
			cout << "					Possible values are COV_MAT_SPHERICAL, COV_MAT_DIAGONAL, COV_MAT_GENERIC." << endl;
			cout << "	-secondGauss	Restriction on gaussin mixture in the second iterations of ME alg. " << endl;
			cout << "					Possible values are COV_MAT_SPHERICAL, COV_MAT_DIAGONAL, COV_MAT_GENERIC." << endl;
			cout << "	-thirdGauss		Restriction on gaussin mixture in the third iterations of ME alg.  " << endl;
			cout << "					Possible values are COV_MAT_SPHERICAL, COV_MAT_DIAGONAL, COV_MAT_GENERIC." << endl;
			return 0;
	} else {
		cerr << "Warning: parameter \"" << argv[i] << "\" was not recognized." << endl;
	}
	}

	cout << " maxIteration " << maxIteration << endl;
	cout << " maxEMIteration " << maxEMIteration << endl;
	cout << " clusterLength " << clusterLength << endl;
	cout << " PCADimensionsUsed " << PCADimensionsUsed << endl;
	cout << " mixtureRestrictions[ 0] " << mixtureRestrictions[ 0] << endl;
	cout << " mixtureRestrictions[ 1] " << mixtureRestrictions[ 1] << endl;
	cout << " mixtureRestrictions[ 2] " << mixtureRestrictions[ 2] << endl;

	if( featureFileNames.empty()) {
		cerr << "Error: No input file with features specified. Use -h for more information." << endl;
		exit( 0);
	}
	if( outputFileName.empty()) {
		cerr << "Error: Output file not specified. Use -h for more information." << endl;
		exit( 0);
	}
	if( unwantedFramesFileName.empty()) {
		cerr << "Warning: Input file with \"Unwanted frames\" not specified." << endl;
	}

	int inputVectorLength = 0;
	int frameCount = 0;

	{
		vector< string>::iterator fileName = featureFileNames.begin();
		while( fileName != featureFileNames.end()) {

			ifstream in( fileName->data(), ios_base::binary | ios_base::in);

			if( !in.good()) {
				cerr << "Warning: Unable to open file \"" << *fileName << "\"." << endl;
				fileName = featureFileNames.erase( fileName);
				continue;
			}

			int vectorLength;
			int tempFrameCount;

			getParams(in, &vectorLength, &tempFrameCount);

			const int temp = vectorLength;
			vectorLength = min( vectorLength, tempFrameCount);
			tempFrameCount = max( temp, tempFrameCount);

			inputVectorLength += vectorLength;

			if( frameCount == 0) {
				frameCount = tempFrameCount;
			} else if( frameCount != tempFrameCount ) {

				cerr << "Warning: File \"" << *fileName << "\" contains vectors of length" << tempFrameCount << ", but should contain vectors of length" << frameCount << ". This file will not be used." << endl;

				fileName = featureFileNames.erase( fileName);
				continue;
			}

			fileName++;
		}
	}

	const int clusterCount = int( frameCount * 0.02 / clusterLength);
	cout << "The number of generated clusters will be: " << clusterCount << endl;

	vector< bool> wantedFrames( frameCount, true);

	if( !unwantedFramesFileName.empty()) {

		ifstream in( unwantedFramesFileName.data());

		if( !in.good()) {
			cerr << "Warning: Unable to open file \"" << unwantedFramesFileName << "\"." << endl;
		}
		
		const int bufferSize = 10000;
		char buf[ bufferSize];
		while( in.good()) {

			in.getline( buf, bufferSize);
			stringstream str( buf);

			if( buf[ 0] != '#' && buf[ 0] != 0){

				int frameID;
				str >> frameID;

				if( str.fail()) {
					cerr << "Warning: While reading unwanted frames. The string \"" << buf << "\" is not a number. " << endl;
				} else {

					if( frameID < wantedFrames.size() && wantedFrames[ frameID]) {
						frameCount--;
						wantedFrames[ frameID] = false;
					} else {
						cerr << "Warning: Unwanted frame \"" << frameID << "\"outside the video size \"" << wantedFrames.size() << "\"." << endl;
					}

				}
			}
		}
	}

	vector< int> framePositions( frameCount, 0);

	CvMat * inputVectors = cvCreateMat( frameCount, inputVectorLength, CV_32F);

	{
		int currentVectorPos = 0;

		vector< string>::iterator fileName = featureFileNames.begin();
		while( fileName != featureFileNames.end()) {

			ifstream in( fileName->data(), ios_base::binary);

			if( !in.good()) {
				cerr << "Warning: Unable to open file \"" << *fileName << "\"." << endl;
				fileName = featureFileNames.erase( fileName);
				continue;
			}

			int tempVectorLength;
			int tempFrameCount;

			getParams(in, &tempVectorLength, &tempFrameCount);

			const int temp = tempVectorLength;
			tempVectorLength = min( tempVectorLength, tempFrameCount);
			tempFrameCount = max( temp, tempFrameCount);

			float features[tempVectorLength];

			int currentSample = 0;
			for( int i = 0; i < tempFrameCount; i++) {

			string str;
			// TODO: test na eof
			getline(in, str);
			int j = 0;
			int start = 1;
			while( (start=str.find(',', start)) != string::npos )
			{
				int tmp;
				sscanf(&str.c_str()[start]+2, "%i", &tmp);
				features[j] = tmp;
				start++;
				j++;
			}

				if( wantedFrames[ i]) {
					memcpy( (void *) cvPtr2D( inputVectors, currentSample, currentVectorPos), features, tempVectorLength * sizeof( float));
					framePositions[ currentSample] = i;
					currentSample++;
				}
			}

			currentVectorPos += tempVectorLength;
			fileName++;
		}
	}

	cvNormalize( inputVectors, inputVectors, -4.0, 4.0, CV_MINMAX);

/*
///////////////////////////////////////////////////////////////////////////////////////
// normalizace vstupnich dat
	CvMat *vectorIn = cvCreateMatHeader( 1, inputVectors->cols, CV_32F);
	CvMat *doubleVectorIn = cvCreateMat( 1, inputVectors->cols, CV_64F);
	CvMat *sum = cvCreateMat( 1, inputVectors->cols, CV_64F);
	CvMat *sum2 = cvCreateMat( 1, inputVectors->cols, CV_64F);

	for( int sample = 0; sample < inputVectors->rows; sample++){
		cvGetRow( inputVectors, vectorIn, sample);
		cvConvertScale( vectorIn, doubleVectorIn);
		cvAdd( doubleVectorIn, sum, sum);
		cvMul( doubleVectorIn, doubleVectorIn, doubleVectorIn);
		cvAdd( doubleVectorIn, sum2, sum2);
	}

	CvMat *stddev = cvCreateMat( 1, inputVectors->cols, CV_64F);
	CvMat *temp1 = cvCreateMat( 1, inputVectors->cols, CV_64F);
	CvMat *temp2 = cvCreateMat( 1, inputVectors->cols, CV_64F);
	CvMat *temp3 = cvCreateMat( 1, inputVectors->cols, CV_64F);

	cvConvertScale( sum2, temp1, 1.0 / (double) inputVectors->rows); 
	cvConvertScale( sum, sum, 1.0 / (double) inputVectors->rows); 
	cvMul( sum, sum, temp2);

	cvSub( temp1, temp2, stddev);
	cvPow( stddev, stddev, 0.5);

	for( int sample = 0; sample < inputVectors->rows; sample++){
		cvGetRow( inputVectors, vectorIn, sample);
		cvConvertScale( vectorIn, doubleVectorIn);

		cvSub( doubleVectorIn, sum, doubleVectorIn);
		cvDiv( doubleVectorIn, stddev, doubleVectorIn);

		cvConvertScale( doubleVectorIn, vectorIn);
	}

	cvReleaseMat( &doubleVectorIn);
	cvReleaseMat( &sum);
	cvReleaseMat( &sum2);
	cvReleaseMat( &stddev);
	cvReleaseMat( &temp1);
	cvReleaseMat( &temp2);
	cvReleaseMat( &temp3);
	cvReleaseMatHeader( &vectorIn);
///////////////////////////////////////////////////////////////////////////////////////
*/

	CvMat * projectedVectors = cvCreateMat( frameCount, PCADimensionsUsed, CV_32F);

	CvMat * avgVector = cvCreateMat( 1, inputVectors->cols, CV_32F);
	CvMat * eigenValues = cvCreateMat( 1, inputVectors->cols, CV_32F);
	CvMat * eigenVectors = cvCreateMat( inputVectors->cols, inputVectors->cols, CV_32F);

	cvCalcPCA( inputVectors, avgVector, eigenValues, eigenVectors, CV_PCA_DATA_AS_ROW);

	cvSetIdentity( eigenVectors);

	cvProjectPCA( inputVectors, avgVector, eigenVectors, projectedVectors);

	cvReleaseMat( &inputVectors);

	const int resolutionX = 700;
	const int resolutionY = 700;

	CvMat * initialMeans = cvCreateMat( clusterCount, PCADimensionsUsed, CV_32F);

	CvMat* labels = cvCreateMat( projectedVectors->rows, 1, CV_32SC1);
	CvMat* labels2 = cvCreateMat( projectedVectors->rows, 1, CV_32SC1);

	vector< CvScalar> colors;
	colors.push_back( cvScalar( 255, 0, 0));
	colors.push_back( cvScalar( 0, 255, 0));
	colors.push_back( cvScalar( 0, 0, 255));
	colors.push_back( cvScalar( 255, 255, 0));
	colors.push_back( cvScalar( 255, 0, 255));
	colors.push_back( cvScalar( 0, 255, 255));
	colors.push_back( cvScalar( 255, 255, 255));
	colors.push_back( cvScalar( 128, 0, 0));
	colors.push_back( cvScalar( 0, 128, 0));
	colors.push_back( cvScalar( 0, 0, 128));
	colors.push_back( cvScalar( 128, 128, 0));
	colors.push_back( cvScalar( 128, 0, 128));
	colors.push_back( cvScalar( 0, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));
	colors.push_back( cvScalar( 128, 128, 128));

	vector< CvMat *> computedProbabilities;
	vector< CvMat *> computedMeans;
	vector< double> computedLogLikelihoods;
	vector< double> computedCoherences;
	vector< IplImage *> computedImages;

////////////////////////////////////////////////////////////////////////////////////////////////
	for( int iteration = 0; iteration < maxIteration; iteration++){

		// select random initial positions
		for( int i = 0; i < clusterCount; i++) {

			int selected = rand() % projectedVectors->rows;

			for( int feature = 0; feature < projectedVectors->cols; feature++){
				cvSet2D( initialMeans, i, feature, cvGet2D( projectedVectors, selected, feature) );
			}
		}

		CvEM *em_model_final = NULL;
		CvEM em_model1;

		CvEMParams params(
				clusterCount,
				mixtureRestrictions[ 0],
				CvEM::START_E_STEP,
				cvTermCriteria(CV_TERMCRIT_ITER+CV_TERMCRIT_EPS, 30, FLT_EPSILON),
				NULL,
				NULL, //weights
				initialMeans, //means
				NULL // covariance matrixes
				);
		em_model1.train( projectedVectors, NULL, params, labels);
		em_model_final = &em_model1;

		CvEM em_model2;
		if( mixtureRestrictions[ 1] != 256) {
			params.cov_mat_type = mixtureRestrictions[ 1];
			params.start_step = CvEM::START_E_STEP;
			params.means = em_model1.get_means();
			params.covs = (const CvMat**)em_model1.get_covs();
			params.weights = em_model1.get_weights();
			em_model2.train( projectedVectors, NULL, params, labels);
			em_model_final = &em_model2;
		}

		CvEM em_model3;
		if( mixtureRestrictions[ 2] != 256) {
			params.cov_mat_type = mixtureRestrictions[ 2];
			params.start_step = CvEM::START_E_STEP;
			params.means = em_model2.get_means();
			params.covs = (const CvMat**)em_model2.get_covs();
			params.weights = em_model2.get_weights();
			em_model3.train( projectedVectors, NULL, params, labels);
			em_model_final = &em_model3;
		}

		IplImage * image = cvCreateImage( cvSize( resolutionX, resolutionY), IPL_DEPTH_8U, 3);
		cvSet( image, cvScalarAll( 0));

		float minX = 1e10;
		float maxX = -1e10;
		float minY = 1e10;
		float maxY = -1e10;
		for( int i = 0; i < projectedVectors->rows; i++) {
			float x = (float) cvGetReal2D( projectedVectors, i, 0);
			float y = (float) cvGetReal2D( projectedVectors, i, 1);

			maxX = max( x, maxX);
			maxY = max( y, maxY);

			minX = min( x, minX);
			minY = min( y, minY);
		}

		// get log likelihood
//		cout << log_likelihood << endl;
		cout << "Coherence: " << computeCoherence( labels) << endl;

		CvMat * probabilities = NULL;
		long double log_likelihood = 0.0;

		getProbabilities( labels2, log_likelihood, probabilities, projectedVectors, em_model_final->get_means(), em_model_final->get_covs(), em_model_final->get_weights());
		for( int i = 0; i < projectedVectors->rows; i++) {
			int x = cvRound( ( cvGetReal2D( projectedVectors, i, 0) - minX)  / ( maxX - minX) * (resolutionX - 1));
			int y = cvRound( ( cvGetReal2D( projectedVectors, i, 1) - minY)  / ( maxY - minY) * (resolutionY - 1));

			int j = (int)cvGetReal2D( labels2, i, 0);
			cvSet2D( image, x, y, colors[ j] );
		}
		cout << "Coherence 2: " << computeCoherence( labels2) << endl;

		int same = 0;
		for( int i = 0; i < projectedVectors->rows; i++) {
			if( cvGet1D( labels, i).val[0] == cvGet1D( labels2, i).val[0]){
				same++;
			}
		}

		cout << "Fraction of equal labels: " << same / (double) projectedVectors->rows << endl;

		cout << " Log likelihood: " << log_likelihood << endl;

		computedProbabilities.push_back( probabilities);

		computedMeans.push_back( cvCreateMat( em_model_final->get_means()->rows, em_model_final->get_means()->cols, em_model_final->get_means()->type));
		cvCopy( em_model_final->get_means(), computedMeans.back());

		computedLogLikelihoods.push_back( log_likelihood);//log_likelihood);
		computedCoherences.push_back( computeCoherence( labels2));

		computedImages.push_back( image);

		cout << endl;
		cvShowImage( "Result", image);
		cvWaitKey( 100);
	}
////////////////////////////////////////////////////////////////////////////////////////////////

	// get the best gaussian mixture
	vector< int> coherenceRanks( computedCoherences.size(), 0);
	vector< int> likelihoodRanks( computedLogLikelihoods.size(), 0);

	for( unsigned result = 0; result < computedCoherences.size(); result++) {

		for( unsigned i = 0; i< computedCoherences.size(); i++) {
			if( computedCoherences[ result] > computedCoherences[ i]) {
				coherenceRanks[ result]++;
			}
			if( computedLogLikelihoods[ result] > computedLogLikelihoods[ i]) {
				likelihoodRanks[ result]++;
			}
		}
	}

	double bestScore = 0;
	int bestResult = 0;
	for( unsigned result = 0; result < computedCoherences.size(); result++) {
		if( coherenceRanks[ result] + likelihoodRanks[ result] > bestScore) {
			bestScore = coherenceRanks[ result] + likelihoodRanks[ result];
			bestResult = result;
		}
	}

	cout << "============================================================" << endl;
	cout << "Selected best result: " << endl;
	cout << endl;
	cout << "Coherence: " << coherenceRanks[ bestResult] << '\t' << computedCoherences[ bestResult] << endl;
	cout << "Log-likelihood: " << likelihoodRanks[ bestResult] << '\t' << computedLogLikelihoods[ bestResult] << endl;
	cout << "Score: " << coherenceRanks[ bestResult] + likelihoodRanks[ bestResult] << " out of " << coherenceRanks.size() * 2 - 2 << endl;

	cvShowImage( "Result", computedImages[ bestResult]);

	// save the result
	ofstream outFile( outputFileName.data(), ios_base::binary);

	int originalFrameCout = wantedFrames.size();
	outFile.write( (char*) &originalFrameCout, sizeof( int));

	outFile.write( (char*) &clusterCount, sizeof( int));
	outFile.write( (char*) &PCADimensionsUsed, sizeof( int));

	float *zeroVector = new float[ clusterCount + PCADimensionsUsed];
	for( int i = 0; i < clusterCount + PCADimensionsUsed; i++) {
		zeroVector[ i] = 0.0;
	}

	CvMat * index = cvCreateMatHeader( 1, clusterCount, computedProbabilities[ bestResult]->type);
	CvMat * floatVector = cvCreateMat( 1, clusterCount, CV_32F);

	int currentPosition = 0;
	for( int i = 0; i < originalFrameCout; i++) {

		if( i < framePositions.size() && framePositions[ currentPosition] == i){
			
			cvGetRow( computedProbabilities[ bestResult], index, currentPosition);
			cvConvertScale( index, floatVector);

			outFile.write( (char*) cvPtr2D( floatVector, 0, 0), clusterCount * 4);
			outFile.write( (char*) cvPtr2D( projectedVectors, currentPosition, 0), PCADimensionsUsed * 4);
			currentPosition++;
		} else {
			outFile.write( (char*) zeroVector, (clusterCount + PCADimensionsUsed) * 4);
		}
	}

	// write the distances

	for( int y = 0; y < clusterCount; y++) {
		for( int x = 0; x < clusterCount; x++) {
			double distance = 0;

			for( int i = 0; i < PCADimensionsUsed; i++) {
				distance += pow( cvGetReal2D( computedMeans[ bestResult], x, i) - cvGetReal2D( computedMeans[ bestResult], y, i), 2);
			}
			float d = (float) sqrt( distance);
			outFile.write( (char*) &d, 4);
		}
	}

	// print the probabilities
/*	for( int i = 0; i < computedProbabilities[ bestResult]->rows; i++){
		for( int d = 0; d < computedProbabilities[ bestResult]->cols; d++){
			cout << cvGetReal2D( computedProbabilities[ bestResult], i, d) << " ";
		}
		cout << endl;
	}*/

	cvWaitKey( 100);

	return 0;
}

double computeCoherence( const CvMat * labels)
{
	int same = 0;
	int change = 0;

	for( int i = 1; i < labels->rows; i++) {
		if( cvGet1D( labels, i).val[0] == cvGet1D( labels, i-1).val[0]) {
			same++;
		} else {
			change++;
		}
	}

	return ((double) same) / (double)( same + change);
}


void getProbabilities( CvMat* labels, long double &log_likelihood, CvMat * &probabilities, const CvMat * dataVectors,const CvMat* means, const CvMat** covs, const CvMat* weights)
{

	CvMat *doubleData = cvCreateMat( dataVectors->rows, dataVectors->cols, CV_64F);
	cvConvertScale( dataVectors, doubleData);

	log_likelihood = 0.0;
	probabilities =  cvCreateMat( doubleData->rows, means->rows, CV_64F);

	CvMat *inData = cvCreateMatHeader( 1, doubleData->cols, CV_64F);
	CvMat *mean = cvCreateMatHeader( 1, doubleData->cols, CV_64F);

	CvMat *temp1 =  cvCreateMat( 1, doubleData->cols, CV_64F);
	CvMat *temp2 =  cvCreateMat( 1, doubleData->cols, CV_64F);
	CvMat *likelihood =  cvCreateMat( 1, 1, CV_64F);

	CvMat *tempCov = cvCreateMat( doubleData->cols, doubleData->cols, CV_64F);

	vector< double> determinants;
	vector< double> normFactors;
	vector< CvMat *> inverseCovs;

	cout << "Determinants: ";
	for( int i = 0; i < means->rows; i++){

		cvConvertScale( covs[ i], tempCov, 1);

		inverseCovs.push_back( cvCreateMat( covs[ 0]->rows, covs[ 0]->cols, CV_64F));

		determinants.push_back( cvDet( tempCov));
		cout << determinants.back() << " ";
		normFactors.push_back( cvGetReal1D( weights, i) / ( pow( 2.0 * PI, means->cols / 2.0) * pow( determinants.back(), 0.5) * dataVectors->rows));

		cvInvert( tempCov, inverseCovs.back(), CV_SVD_SYM);
	}

	cvReleaseMat( &tempCov);
	cout << endl;

	for( int sample = 0; sample < doubleData->rows; sample++) {

		vector< double> responses( means->rows, 0.0f);

		inData = cvGetRow( doubleData, inData, sample);

		double bestResponse = 0.0;
		double bestResponsePos = 0;

		for( unsigned mixtureID = 0; mixtureID < responses.size(); mixtureID++) {
			
			if( determinants[ mixtureID] != 0) {
				// subtact mean
				mean = cvGetRow( means, mean, mixtureID);
				cvSub( inData, mean, temp1);

				cvGEMM( temp1, inverseCovs[ mixtureID], 1.0, NULL, 1.0, temp2);
				cvGEMM( temp2, temp1, 1.0, NULL, 1.0, likelihood, CV_GEMM_B_T);

				responses[ mixtureID] = normFactors[ mixtureID] * exp( -0.5 * cvGetReal1D( likelihood, 0));

				if( responses[ mixtureID] > 1.0) {
					double x= responses[ mixtureID];
					x = x;
				}

				if( responses[ mixtureID] > bestResponse) {
					bestResponse = responses[ mixtureID];
					bestResponsePos = mixtureID;
				}
			}
		}

		cvSet1D( labels, sample, cvScalarAll( bestResponsePos));

		long double likelihoodTemp = 0;
		for( unsigned mixtureID = 0; mixtureID < responses.size(); mixtureID++) {
			likelihoodTemp += responses[ mixtureID];
		}

		if( log( likelihoodTemp) > 0) {
			double x= log( likelihoodTemp);
			x = x;
		}

		log_likelihood += log( likelihoodTemp);

		for( unsigned mixtureID = 0; mixtureID < responses.size(); mixtureID++) {
			cvSetReal2D( probabilities, sample, mixtureID, (double)(responses[ mixtureID]) );
		}
	}

	cvReleaseMat( &doubleData);

	cvReleaseMatHeader( &inData);
	cvReleaseMatHeader( &mean);

	cvReleaseMat( &temp1);
	cvReleaseMat( &temp2);
	cvReleaseMat( &likelihood);

//	(data - means)T * covariance * (data - means)
}
