/*!
\file tracker.hpp

\author Juraj Blaho
\modified by Michal Hradis 17.11.2008

\brief Tracker.
*/

#ifndef TRACKER_INCLUDED
#define TRACKER_INCLUDED

#include <vector>

#include <cxcore.h>
#include <cv.h>

using namespace std;

/*!
\brief State of a single tracked object
*/
struct TrackedState{

		double x;
		double y;
		double rx;
		double ry;

		double dx, dy, drx, dry;

		int prevID;

		/*< Previous state */
	
		/*!
		*/
		TrackedState( const CvRect &rectangle, TrackedState *_previous):
			x( rectangle.x + rectangle.width / 2.0), 
			y( rectangle.y + rectangle.height / 2.0), 
			rx( rectangle.width / 2.0),
			ry( rectangle.height / 2.0),
			dx( x - _previous->x),
			dy( y - _previous->y), 
			drx( rx - _previous->rx),
			dry( ry - _previous->ry),
			prevID( _previous->prevID)
		{
		}
		TrackedState( const CvRect &rectangle):
			x( rectangle.x + rectangle.width / 2.0), 
			y( rectangle.y + rectangle.height / 2.0), 
			rx( rectangle.width / 2.0),
			ry( rectangle.height / 2.0),
			dx( 0),
			dy( 0), 
			drx( 0),
			dry( 0),
			prevID( -1)
		{
		}
};

/*!
\brief Blob tracker

This tracker is able to track several objects (blobs) detected in the image.
*/
class Tracker{

	std::vector< TrackedState> lastBlobs;

	int _current;
	int _previous;
	
	bool _use_morphology;
	int _value_threshold;
	int _area_threshold;
		
	public:
		/*!
		\brief The constructor.
		*/
		Tracker( int area_threshold=400, int value_threshold=128, bool use_morphology=true):
			_use_morphology(use_morphology),
			_value_threshold(value_threshold),
			_area_threshold(area_threshold)
		{}
		
		
		/*!
		\brief Evaluate new frame
		
		\param image The source probability image to be evaluated.
		*/
		std::vector< TrackedState> NewFrame(IplImage *image);
		
};

#endif
