/******************************************************************************
	\brief Http header simple reader.
******************************************************************************/
#include "curlTools/header.h"

using namespace comm;


#define SEPNUM 5

#define WHITENUM 3
const char whites[WHITENUM] = {' ', '\n', '\r'};

#define SPECNUM 3
const char specials[SPECNUM] = {',', ':', ';'};

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CHeaderReader::CHeaderReader( std::istream *input, long maxlength )
///
/// \brief	Constructor. 
///
/// \param [in]	input	If non-null, the input. 
/// \param	maxlength	The maximal stream length. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CHeaderReader::CHeaderReader( std::istream *input, long maxlength ):
	tokeniser(input),
	in(input),
	maxCount(maxlength),
	remains(maxlength)
{
	// get remaining data length
	if(maxlength != -1)
		remains = maxlength;

	// initialize tokeniser
	initTokeniser();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CHeaderReader::~CHeaderReader()
///
/// \brief	Destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CHeaderReader::~CHeaderReader()
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CHeaderReader::initTokeniser()
///
/// \brief	Initialises the tokeniser. 
///
/// \author	Wik
/// \date	15.12.2009
////////////////////////////////////////////////////////////////////////////////////////////////////

void CHeaderReader::initTokeniser()
{
	tokeniser.addSpecialChars(specials, SPECNUM);
	tokeniser.addWhiteChars(whites, WHITENUM);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::readLine( tHeaderMap & map )
///
/// \brief	Reads a line - header and parameters. 
///
/// \param [out]	map	the map of headers. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::readLine( tHeaderMap & map )
{

	// string buffer
	std::string header;

	// parameters vector buffer
	tParametersVector values;

	// Read header string
	if( !readHeader( header ) )
		return false;

	// Nex must be :
	if( !compareNextToken( std::string( ":" ) ) )
		return false;

	// Try to read values
	if( ! readValues( values ) )
		return false;

	// Set header parameters to the map
	map[ header ] = values;

	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::readHeader( std::string &attribute )
///
/// \brief	Reads a header. 
///
/// \param [out]	attribute	the attribute - header name. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::readHeader( std::string &attribute )
{

	stringTools::strToken a;
    long readed;
	bool allOK = tokeniser.getNextToken(a, readed, remains);

	if(allOK){

		// less characters remain
		if(remains != -1)
			remains -= readed;

		// return value
		attribute = a.data;

	}

	return allOK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::compareNextToken(const std::string &str)
///
/// \brief	Compare next readed token with given string. 
///
/// \param	str	The string to compare. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::compareNextToken(const std::string &str){

	stringTools::strToken token;
	long readed;

	// read token
	bool allOK = tokeniser.getNextToken(token, readed, remains);

	if(allOK){
		// compare token with given string
		return token.data == str;
	}

	return false;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::readValues( tParametersVector & values )
///
/// \brief	Reads the values. 
///
/// \param [out]	values	the values. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::readValues( tParametersVector & values )
{

	long readed;
	stringTools::strToken old, current;

	// Parse parameters
	while( tokeniser.getNextToken( current, readed, remains ) )
	{
		if(remains != -1)
			remains -= readed;

		if( ( current.data == ";" ) && ( old.data != "" ) )
		{
			values.push_back( old.data );
		}

		old.data = current.data;
	}

	// If only one parameter was included, insert it
	if( current.data != "" )
		values.push_back( current.data );

	return true;
}


