/******************************************************************************
	\brief Curl multi interface encapsulation
******************************************************************************/
#include "curlTools/curlMulti.h"

using namespace comm;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CCurlMulti::CCurlMulti()
///
/// \brief	Default constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CCurlMulti::CCurlMulti():
CCurl(easy)
{


}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurlMulti::perform()
///
/// \brief	Performs curl operation. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurlMulti::perform(){

	// Curl is not properly initialized -> failed
	if(handle == NULL){
		serror = "Curl was not properly initialized.";
		writee();
		return false;
	}

	// return code
	CURLMcode mcode;

	// operation is still running
	int still_runing;

	while((mcode = curl_multi_perform(handle, &still_runing)) == CURLM_CALL_MULTI_PERFORM){

		// some failure?
		if(mcode != 0){

			//some error occurred, so get the string
			serror = curl_multi_strerror(mcode);
			writee();
			return false;
		} // if mcode != 0

	} // while ...

	return true;
} // perform

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlMulti::stop()
///
/// \brief	Stops all operations. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlMulti::stop(){

	ccurlMap::iterator i;

	// cleanup all included connections
	for(i = connections.begin(); i != connections.end(); ++i){
		i->second->stop();
		delete i->second;
	}
	// clear map
	connections.erase(connections.begin(), connections.end());

	curl_multi_cleanup(handle);
} // stop

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurlMulti::reset()
///
/// \brief	Resets interface. 
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurlMulti::reset(){
	// do nothing
	return true;
} // reset

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	long CCurlMulti::addCE()
///
/// \brief	Adds new curl easy interface to the multi interface.
///
/// \return	The calculated ce. 
////////////////////////////////////////////////////////////////////////////////////////////////////

long CCurlMulti::addCE(){

	CCurlEasy *curl = new CCurlEasy;

	// something is wrong
	if(curl == NULL){

		serror = "Cannot create new interface.";
		writee();
		return -1;
	}

	if(!curl->init(dataout, headerout, errout))
		return -1;

	// Add handle to multi interface
	CURLMcode mcode;
	if((mcode = curl_multi_add_handle(handle, curl->handle)) != 0){

		//some error occurred, so get the string
		serror = curl_multi_strerror(mcode);
		writee();
		return -1;
	}

	connections.insert(std::make_pair(idCounter, curl));

	return idCounter++;
} // addCE

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CCurlEasy * CCurlMulti::getCE(long id)
///
/// \brief	Gets a curl easy interface by id. 
///
/// \param	id	The identifier. 
///
/// \return	null if it fails, else the ce. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CCurlEasy * CCurlMulti::getCE(long id){

	ccurlMap::iterator i = connections.find(id);

	if(i != connections.end())
		return i->second;

	return NULL;
} // getCE

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlMulti::removeCE(long id)
///
/// \brief	Removes the curl easy interface described by id. 
///
/// \param	id	The identifier. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlMulti::removeCE(long id){

	ccurlMap::iterator i;

	// find and remove handle
	if((i = connections.find(id)) != connections.end()){
		curl_multi_remove_handle(handle, i->second->handle);
		delete i->second;
		connections.erase(id);
	} // if

} // removeCE

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlMulti::setCurlCallback(CURLoption cbType, tCurlCallback callback )
///
/// \brief	Callback. Not implemented yet...
///
/// \param	cbType		Type of the cb. 
/// \param	callback	The callback. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlMulti::setCurlCallback(CURLoption cbType, tCurlCallback callback /* = NULL */){


	
}
