/******************************************************************************
	\brief Curl easy interface encapsulation
******************************************************************************/
#include <curlTools/curlEasy.h>
#include <cstring>

using namespace comm;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CCurlEasy::CCurlEasy()
///
/// \brief	Default constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CCurlEasy::CCurlEasy():
CCurl( easy ),
transferStage(TRANSFER_STAGE_UNKNOWN)
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurlEasy::setUrl(const std::string & url)
///
/// \brief	Sets a url address. 
///
/// \param	url	The url. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurlEasy::setUrl(const std::string & url){

		// Length of string
		int length = static_cast<int>(url.length());


		std::cout << url.c_str() << std::endl;

		// set option
		if(setopt(CURLOPT_URL, url.c_str())){
			sUrl = url;
			return true;
		}

		return false;
} // setUrl

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	std::string CCurlEasy::getUrl()
///
/// \brief	Gets the url currently used. 
///
/// \return	The url. 
////////////////////////////////////////////////////////////////////////////////////////////////////

std::string CCurlEasy::getUrl()
{

	char * url;

	if( curl_easy_getinfo( handle, CURLINFO_EFFECTIVE_URL, &url ) == CURLE_OK )
	{
		return std::string( url );
	}

	return std::string("");
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurlEasy::perform()
///
/// \brief	Performs curl operation. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurlEasy::perform(){

	// Curl is not properly initialized -> failed
	if(handle == NULL){
		serror = "Curl was not properly initialized.";
		writee();
		return false;
	}

	// return code
	CURLcode code;

	// lock interface
	lock();

	// set current stage to the transfer
	transferStage = TRANSFER_STAGE_STARTED;

	// perform operation
	code = curl_easy_perform(handle);

	// unlock interface
	unlock();

	// handle return code
	if(code != 0){
		
		char buffer[256];
		setopt(CURLOPT_ERRORBUFFER, buffer ); 

		//some error occurred, so get the string
		serror = curl_easy_strerror(code);

		// write error
		writee();

		// set transfer stage to the unknown (interface should be reinitialized by user)
		transferStage = TRANSFER_STAGE_UNKNOWN;

		return false;
	} // if(code != 0)

	// all is ok, stage is == completed
	transferStage = TRANSFER_STAGE_COMPLETED;
	return true;
} // perform

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlEasy::stop()
///
/// \brief	Stops all operations. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlEasy::stop(){

	curl_easy_cleanup(handle);

	// all is ok, stage is == completed
	transferStage = TRANSFER_STAGE_COMPLETED;

} // stop

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurlEasy::reset()
///
/// \brief	Resets curl interface. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurlEasy::reset(){

	if(handle != NULL)
		curl_easy_reset(handle);

	if(handle == NULL){
		serror = "Unknown error.";
		writee();
		return false;
	} // if handle == NULL

	return true;
} // reset

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurlEasy::setNamePasswd(const std::string &name, const std::string &passwd)
///
/// \brief	Sets a name and password. 
///
/// \param	name	The name. 
/// \param	passwd	The passwd. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurlEasy::setNamePasswd(const std::string &name, const std::string &passwd){

	sNP = name + ":" + passwd;

	setopt(CURLOPT_HTTPAUTH, CURLAUTH_DIGEST);
	setopt(CURLOPT_SSL_VERIFYPEER, 0);

	
	if(setopt(CURLOPT_USERPWD, sNP.c_str()) ){

		return true;
	}

	return false;
} // setNamePasswd

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlEasy::modeUpload()
///
/// \brief	Mode upload. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlEasy::modeUpload(){

	setopt(CURLOPT_UPLOAD, true);
	
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlEasy::modeDownload()
///
/// \brief	Mode download. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlEasy::modeDownload(){

	setopt(CURLOPT_UPLOAD, false);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlEasy::setForm(const curlForm * form)
///
/// \brief	Prepares a form. 
///
/// \param	form	The form. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlEasy::setForm(const curlForm * form){

	if(form == NULL)
		return;

	curl_httppost *f = form->getForm();
	
	if(f == NULL)
		return;

	setopt(CURLOPT_HTTPPOST, f);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlEasy::setCookie(const std::string & cookie)
///
/// \brief	Sets a cookie. 
///
/// \param	cookie	The cookie. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlEasy::setCookie(const std::string & cookie)
{
    setopt(CURLOPT_COOKIE, cookie.c_str()+'\0');
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurlEasy::setCookieFile(const std::string & cookieFileName )
///
/// \brief	Sets a cookie file. 
///
/// \param	cookieFileName	Filename of the cookie file. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurlEasy::setCookieFile(const std::string & cookieFileName /* = std::string */)
{
    setopt(CURLOPT_COOKIEFILE, cookieFileName.c_str());
}

