/******************************************************************************
	\brief Base of all user interfaces
******************************************************************************/
#include <curlTools/CCurl.h>
#include <stdio.h>

using namespace comm;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CCurl::CCurl(interfaceType type)
///
/// \brief	Constructor. 
///
/// \param	type	The type of the interface. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CCurl::CCurl(interfaceType type):
handle(NULL),
serror(""), 
errout(NULL),
dataout(NULL),
headerout(NULL),
usedInterface(type),
slist(NULL)
{

	init(NULL, NULL, NULL);

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CCurl::~CCurl()
///
/// \brief	Destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CCurl::~CCurl(){

	// Do cleanup
	switch( usedInterface )
	{
	case easy:
		curl_easy_cleanup( handle );
		break;

	case multi:
		curl_multi_cleanup( handle );
		break;
	}

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurl::basicInit(std::ostream * dstream, std::ostream * hstream,
/// 	std::ostream * estream)
///
/// \brief	Basic initialise. 
///
/// \param [out] dstream	If non-null, the output data stream. 
/// \param [out] hstream	If non-null, the header stream. 
/// \param [out] estream	If non-null, the output error stream. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurl::basicInit(std::ostream * dstream, std::ostream * hstream, std::ostream * estream){

	if(handle == NULL){
		serror = "Unknown error.";
		writee();
		return false;
	}

    // EXPERIMENTAL !!!
    setopt(CURLOPT_TCP_NODELAY, long(1));
    setopt(CURLOPT_MAXCONNECTS, long(50));

    if(!setopt(CURLOPT_ERRORBUFFER, buffer)){

		serror = "Cannot set error buffer.";
		writee();
		return false;
	}

	if(!setopt(CURLOPT_WRITEFUNCTION, &write_data)){

		serror = "Cannot set write data function.";
		writee();
		return false;
	}

	if(!setopt(CURLOPT_HEADERFUNCTION, &header_callback)){

		serror = "Cannot set write header function.";
		writee();
		return false;
	}

	setDataOutput(dstream);
	setHeaderOutput(hstream);
	setErrOutput(estream);

	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurl::init(std::ostream * dstream, std::ostream * hstream, std::ostream * estream)
///
/// \brief	Initialises this object. 
///
/// \param [out]	dstream	If non-null, the data output stream. 
/// \param [out]	hstream	If non-null, the header output stream. 
/// \param [out]	estream	If non-null, the error output stream. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurl::init(std::ostream * dstream, std::ostream * hstream, std::ostream * estream){
	
	switch(usedInterface){
		case easy:
			handle = curl_easy_init();
			break;
		case multi:
			handle = curl_multi_init();
			break;
	}

	return basicInit(dstream, hstream, estream);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::writee()
///
/// \brief	Writes error to the output stream. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::writee(){

	if(errout != NULL){

		if(*errout){

			*errout << error() << std::endl;
		} // if *out
	} // if out != NULL
}	// writee

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::setErrOutput(std::ostream *stream)
///
/// \brief	Sets an error output. 
///
/// \param [out]	stream	If non-null, the output stream. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::setErrOutput(std::ostream *stream){

	errout = stream;
	setopt(CURLOPT_ERRORBUFFER, buffer);
} // setErrOutput

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::setHeaderOutput(std::ostream *stream)
///
/// \brief	Sets a header output. 
///
/// \param [out]	stream	If non-null, the output stream. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::setHeaderOutput(std::ostream *stream){

	headerout = stream;
	setopt(CURLOPT_WRITEHEADER, stream);
} // setErrOutput

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::setDataOutput(std::ostream *stream)
///
/// \brief	Sets a data output. 
///
/// \param [out]	stream	If non-null, the output stream. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::setDataOutput(std::ostream *stream){

	dataout = stream;
	setopt(CURLOPT_WRITEDATA, stream);
} // setErrOutput

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	size_t CCurl::write_data(void *ptr, size_t size, size_t nmemb, void *stream)
///
/// \brief	Writes a data to the data stream. 
///
/// \param [in]	ptr		If non-null, the data pointer. 
/// \param	size			The size of element. 
/// \param	nmemb			The number of elements (size in bytes is size*nmemb). 
/// \param [out]	stream	If non-null, the output stream. 
///
/// \return	. 
////////////////////////////////////////////////////////////////////////////////////////////////////

size_t CCurl::write_data(void *ptr, size_t size, size_t nmemb, void *stream)
{
	std::ostream *os = static_cast<std::ostream *>(stream);
	
	// output stream is valid
	if(stream != NULL){

		if(*os){
			// write data to the output stream
			os->write(static_cast<char *>(ptr), static_cast<std::streamsize>(size*nmemb));
			if(*os)
				return size*nmemb;
		}else
			return 0;
	}else
		return size*nmemb;

	return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	size_t CCurl::header_callback(void *ptr, size_t size, size_t nmemb, void *stream)
///
/// \brief	Callback, called when the header is readed. 
///
/// \param [in]	ptr		If non-null, the pointer to the header data. 
/// \param	size			The size of element. 
/// \param	nmemb			The number of elements. 
/// \param [out]	stream	If non-null, the output stream. 
///
/// \return	. 
////////////////////////////////////////////////////////////////////////////////////////////////////

size_t CCurl::header_callback(void *ptr, size_t size, size_t nmemb, void *stream)
{
	// 
	std::ostream *os = static_cast<std::ostream *>(stream);

	// is stream valid?
	if(stream != NULL){

		if(*os){
			// write data to the stream
			os->write(static_cast<char *>(ptr), static_cast<std::streamsize>(size*nmemb));
			if(*os)
				return size*nmemb;
		}else
			return 0;
	}else
		return size*nmemb;

    return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::setCurlCallback(CURLoption cbType, tCurlCallback callback)
///
/// \brief Sets callback function
///
/// \param	cbType		Type of the cb. 
/// \param	callback	The callback. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::setCurlCallback(CURLoption cbType, tCurlCallback callback){

	switch(cbType)
	{
	case CURLOPT_READFUNCTION:
	case CURLOPT_WRITEFUNCTION:
	case CURLOPT_HEADERFUNCTION:
	case CURLOPT_DEBUGFUNCTION:
	case CURLOPT_PROGRESSFUNCTION:
	case CURLOPT_IOCTLFUNCTION:
	case CURLOPT_SSL_CTX_FUNCTION:
		if(!setopt(cbType, callback)){

			serror = "Cannot set callback function.";
			writee();
		}
	} // switch cbType
} // setCallback

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	char * CCurl::encode(const char * uri)
///
/// \brief	Encodes uri to string.  
///
/// \param	uri	The uri string. 
///
/// \return	null if it fails, else. 
////////////////////////////////////////////////////////////////////////////////////////////////////

char * CCurl::encode(const char * uri){
	
	return curl_escape(uri, 0);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	char * CCurl::decode(const char * uri, int * outputlength)
///
/// \brief	Decodes uri. 
///
/// \param	uri						The uri. 
/// \param [out]	outputlength	If non-null, the output length. 
///
/// \return	null if it fails, else. 
////////////////////////////////////////////////////////////////////////////////////////////////////

char * CCurl::decode(const char * uri, int * outputlength){
	return curl_unescape(uri, 0);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CCurl::setShare( CCurlShared & shared )
///
/// \brief	Sets a sharing pool. 
///
/// \param [in,out]	shared	the shared interface. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CCurl::setShare( CCurlShared & shared )
{
	return setopt( CURLOPT_SHARE, shared.handle );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::setSessionTimeout( long duration )
///
/// \brief	Sets a session timeout in ms. 
///
/// \param	duration	The duration. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::setSessionTimeout( long duration )
{
	setopt( CURLOPT_TIMEOUT, duration );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CCurl::setOperationTimeout( long duration )
///
/// \brief	Sets an operation timeout in seconds. 
///
/// \param	duration	The duration. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CCurl::setOperationTimeout( long duration )
{
	setopt( CURLOPT_CONNECTTIMEOUT, duration );
}
