/******************************************************************************
	\brief Http header creation/parsing tools
******************************************************************************/
#include "curlTools/header.h"
#include <algorithm>
#include <cctype>

using namespace comm;


#define SEPNUM 5


#define WHITENUM 3
const char whites[WHITENUM] = {' ', '\n', '\r'};

#define SPECNUM 3
const char specials[SPECNUM] = {',', ':', ';'};

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CHeaderReader::CHeaderReader( std::istream *input, long maxlength )
///
/// \brief	Constructor. 
///
/// \param [in]	input	If non-null, the input stream. 
/// \param	maxlength	The length of the input. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CHeaderReader::CHeaderReader( std::istream *input, long maxlength ):
	tokeniser(input),
	in(input),
	maxCount(maxlength),
	remains(maxlength)
{

	if(maxlength != -1)
		remains = maxlength;

	initTokeniser();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CHeaderReader::~CHeaderReader()
///
/// \brief	Destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CHeaderReader::~CHeaderReader()
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CHeaderReader::initTokeniser()
///
/// \brief	Initialises the tokeniser. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CHeaderReader::initTokeniser()
{
	tokeniser.addSpecialChars(specials, SPECNUM);
	tokeniser.addWhiteChars(whites, WHITENUM);

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::readLine( tHeaderMap & map )
///
/// \brief	Reads a line (header name and its parameters). 
///
/// \param [out]	map	the headers map. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::readLine( tHeaderMap & map )
{

	std::string header;

	tParametersVector values;

	// Read header string
	if( !readHeader( header ) )
		return false;

	// Nex must be :
	if( !compareNextToken( std::string( ":" ) ) )
		return false;


	if( ! readValues( values ) )
		return false;

	// Set header to lower
	// explicit cast needed to resolve ambiguity
	std::transform(header.begin(), header.end(), header.begin(), (int(*)(int)) std::tolower);

	map[ header ] = values;

	return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::readHeader( std::string &attribute )
///
/// \brief	Reads a header. 
///
/// \param [out]	attribute	the attribute name and parameters. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::readHeader( std::string &attribute )
{

	stringTools::strToken a;
    long readed;
	bool allOK = tokeniser.getNextToken(a, readed, remains);

	if(allOK){

		// less characters remain
		if(remains != -1)
			remains -= readed;

		// return value
		attribute = a.data;

	}

	return allOK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::compareNextToken(const std::string &str)
///
/// \brief	Compare readed token with the given string. 
///
/// \param	str	The string. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::compareNextToken(const std::string &str){

	stringTools::strToken token;
	long readed;

	// read token
	bool allOK = tokeniser.getNextToken(token, readed, remains);

	if(allOK){

		return token.data == str;
	}

	return false;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	bool CHeaderReader::readValues( tParametersVector & values )
///
/// \brief	Reads the header parameters values. 
///
/// \param [in,out]	values	the values. 
///
/// \return	true if it succeeds, false if it fails. 
////////////////////////////////////////////////////////////////////////////////////////////////////

bool CHeaderReader::readValues( tParametersVector & values )
{

	long readed;
	stringTools::strToken old, current;

	// Parse parameters
	while( tokeniser.getNextToken( current, readed, remains ) )
	{
		if(remains != -1)
			remains -= readed;

		if( ( current.data == ";" ) && ( old.data != "" ) )
		{
			values.push_back( old.data );
		}

		old.data = current.data;
	}

	// If only one parameter was included, insert it
	if( current.data != "" )
		values.push_back( current.data );

	return true;
}


