/******************************************************************************
	\brief Error handling routines
******************************************************************************/
#include <comm/errorHandler.h>
#include <MDSTk/Base/mdsLogChannel.h>
#include <MDSTk/System/mdsTimestamp.h>

using namespace comm;

/********************************************
		CLASS CErrorInfo
*********************************************/

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CErrorInfo::CErrorInfo( eErrorType type, eProcessingRoutine processingType, long code,
/// 	std::string description, std::string caller )
///
/// \brief	Constructor. 
///
/// \param	type			The type of error. 
/// \param	processingType	Type of the processing. 
/// \param	code			The code of error. 
/// \param	description		The string description. 
/// \param	caller			The callers name. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CErrorInfo::CErrorInfo( eErrorType type, eProcessingRoutine processingType, long code, std::string description, std::string caller ) :
	m_type( type ),
	m_process( processingType ),
	m_code( code ),
	m_caller( caller ),
	m_dataString( description )
{

}

/********************************************
		CLASS CErrorHandler
*********************************************/

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CErrorHandler::CErrorHandler()
///
/// \brief	Default constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CErrorHandler::CErrorHandler()
: m_useLog( true ),
  m_channel( NULL )
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CErrorHandler::setLogName( const std::string & name )
///
/// \brief	Sets a log file name. 
///
/// \param	name	The name. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CErrorHandler::setLogName( const std::string & name )
{
	if( m_channel.get() != NULL )
		m_log.unregisterChannel( m_channel );

	m_channel = new mds::base::CLogChannelFStream( name );
	m_log.registerChannel( m_channel.get() );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CErrorHandler::cbError(comm::CErrorInfo * error)
///
/// \brief	Error callback. 
///
/// \param [in,out]	error	If non-null, the error. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CErrorHandler::cbError(comm::CErrorInfo * error)
{
	comm::CErrorInfoPtr ptrError( error );

	if( error == NULL )
	{
		exit( -13 );
	}

	// switch error processing type
	switch( error->m_process )
	{
	case CErrorInfo::EP_ONLY_LOG:
		log( error );
		break;

	case CErrorInfo::EP_LOG_AND_CALL:
		log( error );
		call( error );
		break;

	case CErrorInfo::EP_LOG_AND_EXIT:
		log( error );
		exit( -13 );
		break;	

	default:
		break;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CErrorHandler::log(comm::CErrorInfo * error)
///
/// \brief	Logs error. 
///
/// \param [in]	error	If non-null, the error. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CErrorHandler::log(comm::CErrorInfo * error)
{
	if( ! m_useLog )
		return;		// No log please...

	comm::CErrorInfoPtr ptrError( error );
	// Log time
	
	m_log.lock();
	m_log.getStream() << MDS_TIMESTAMP( error->m_caller ) <<  ": " << error->m_dataString << std::endl;
	m_log.unlock();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CErrorHandler::call(comm::CErrorInfo *error)
///
/// \brief	Calls error callback. 
///
/// \param [in]	error	If non-null, the error. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CErrorHandler::call(comm::CErrorInfo *error)
{
	comm::CErrorInfoPtr ptrError( error );

	switch( error->m_type )
	{
	// Connection error - call if too much errors
	case CErrorInfo::ET_CONNECTION:

		m_sigError.invoke( error );
		break;

	// Other error type - just call
	case CErrorInfo::ET_OTHER:
		m_sigError.invoke( error );
		break;

	}
}

