/******************************************************************************
	\brief Connection callbacks implementation
******************************************************************************/
#include <comm/connection.h>
#include <algorithm>
#include <fstream> 


//=================================================================
// CALLBACKS
//=================================================================
namespace comm{

#ifdef LOG_TRAFFIC
	std::ofstream log( "_curlin.log" );
	std::ofstream sendlog( "_curlout.log" );
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	size_t dispatchCallback(void *ptr, size_t size, size_t nmemb, void *userData)
///
/// \brief	Callback, called when the dispatching data to send. 
///
/// \param [in]	ptr			If non-null, the pointer to data. 
/// \param	size			The size of one element. 
/// \param	nmemb			The number of elements. 
/// \param [in]	userData	If non-null, pointer to user defined data. 
///
/// \return	number of elements dispatched. 
////////////////////////////////////////////////////////////////////////////////////////////////////

size_t dispatchCallback(void *ptr, size_t size, size_t nmemb, void *userData)
{

#ifdef LOG_TRAFFIC
	sendlog.write( (char *)ptr, nmemb*size );
#endif

	// Try to convert transfer info
	if(userData == NULL)
	{
		// Compose error info
		std::string strError("HeaderReceiverCallback: User data (CDTInfo) not set." );
		// Error - unknown content type
		MDS_SINGLETON( CErrorHandler ).cbError( new CErrorInfo( CErrorInfo::ET_OTHER, CErrorInfo::EP_LOG_AND_CALL, 0, strError, "vceConnection" ) );

		return 0;
	}

	CDTInfo * info = (CDTInfo *) userData;

	// Cancelled?
	if( info->m_cancelEvent->isSet() )
		return CURL_READFUNC_ABORT;

	// call receiver
	return info->m_dispatcher->dispatch((char *) ptr, size*nmemb);
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	size_t receiveCallback(void *ptr, size_t size, size_t nmemb, void *userData)
///
/// \brief	Callback, called when the receiveing data. 
///
/// \param [in]	ptr			If non-null, the pointer to data to receive. 
/// \param	size			The size of data element. 
/// \param	nmem			The num of data elements. 
/// \param [in]	userData	If non-null, pointer to user defined data.  
///
/// \return	number of parsed elements. 
////////////////////////////////////////////////////////////////////////////////////////////////////

size_t receiveCallback(void *ptr, size_t size, size_t nmemb, void *userData)
{

#ifdef LOG_TRAFFIC
	log.write( (char *)ptr, nmemb*size );
#endif

	// Try to convert stream to transfer
	if(userData == NULL)
	{
		// Compose error info
		std::string strError("receiveCallback: User data (CDTInfo) not set." );
		// Error - unknown content type
		MDS_SINGLETON( CErrorHandler ).cbError( new CErrorInfo( CErrorInfo::ET_OTHER, CErrorInfo::EP_LOG_AND_CALL, 0, strError, "vceConnection" ) );

		return 0;
	}

	CDTInfo * info = (CDTInfo *) userData;
	
	if( info->m_cancelEvent->isSet() )
		return 0;	// Transfer is cancelled...

	// Test incomming header
	if( info->m_stage == CDTInfo::STAGE_HEADER )
	{
		if( ! info->testHeader() )
		{
			// SEND ERROR - WRONG DATA TYPE
			// Compose error info
			std::string strError(" receiverCallback: Http header problem. Cannot use receiver." );
			// Error - unknown content type
			MDS_SINGLETON( CErrorHandler ).cbError( new CErrorInfo( CErrorInfo::ET_OTHER, CErrorInfo::EP_LOG_AND_CALL, 0, strError, "vceConnection" ) );

			return 0;
		}

		info->m_stage = CDTInfo::STAGE_DATA;
	}

	// call receiver
	return info->m_receiver->receive((char *) ptr, size*nmemb);

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	size_t headerReceiveCallback(void *ptr, size_t size, size_t nmemb, void *userData)
///
/// \brief	Callback, called when receiveing the header. 
///
/// \param [in,out]	ptr			If non-null, the pointer to header data. 
/// \param	size				The size of data element. 
/// \param	nmemb				The number of elemets. 
/// \param [in,out]	userData	If non-null, user defined data. 
///
/// \return	number of received elements. 
////////////////////////////////////////////////////////////////////////////////////////////////////

size_t headerReceiveCallback(void *ptr, size_t size, size_t nmemb, void *userData)
{

#ifdef LOG_TRAFFIC
	log.write( (char *)ptr, nmemb*size );
#endif

	// Try to convert to transfer info
	if(userData == NULL)
	{
		// Compose error info
		std::string strError("headerReceiverCallback: User data (CDTInfo) not set." );
		// Error - unknown content type
		MDS_SINGLETON( CErrorHandler ).cbError( new CErrorInfo( CErrorInfo::ET_OTHER, CErrorInfo::EP_LOG_AND_CALL, 0, strError, "vceConnection" ) );
		
		return 0;
	}

	CDTInfo * info = (CDTInfo *) userData;

	
	// If is all parsed, then this is not the header but a trailer (see curl documentation)
	// So discard it
	if( info->m_stage > CDTInfo::STAGE_HEADER )
		return  size*nmemb;

    // call receiver
	size_t retval = info->m_headerParser->parse((char *) ptr, size*nmemb);

    return retval;
}

} // namespace comm


using namespace comm;

//=================================================================
// CLASS CConnectionParams
//=================================================================

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CConnectionParams::CConnectionParams( const CConnectionParams & params )
///
/// \brief	Copy constructor. 
///
/// \param	params	original paramteres object. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CConnectionParams::CConnectionParams( const CConnectionParams & params )
{
	tParametersMap::const_iterator i;

	for( i = params.m_options.begin(); i != params.m_options.end(); ++i )
	{
		m_options.insert( *i );
	}

	m_sessionID = params.m_sessionID;

	//! Client id
	m_clientID = params.m_clientID;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CConnectionParams CConnectionParams::operator= ( const CConnectionParams & params )
///
/// \brief	Copy operator. 
///
/// \param	params	Original data.
///
/// \return	A shallow copy of this object. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CConnectionParams CConnectionParams::operator = ( const CConnectionParams & params )
{
	m_options.clear();

	tParametersMap::const_iterator i;

	for( i = params.m_options.begin(); i != params.m_options.end(); ++i )
	{
		m_options.insert( *i );
	}

	m_sessionID = params.m_sessionID;

	//! Client id
	m_clientID = params.m_clientID;

	return params;
}



