/******************************************************************************
	\brief Buffer reciever and dispatcher
******************************************************************************/
#include <comm/bufferdt.h>

using namespace comm;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferReceiver::CBufferReceiver()
///
/// \brief	Default constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferReceiver::CBufferReceiver()
	: buffer(NULL), buflength(0), position(0), bDeleteBuffer(true)
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferReceiver::CBufferReceiver(size_t length)
///
/// \brief	Constructor. 
///
/// \param	length	The length of the buffer. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferReceiver::CBufferReceiver(size_t length)
{
	tBRLock Lock( * this );

	buffer = new char[length];

	assert(buffer != NULL);

	buflength = length;

	position = 0;

	bDeleteBuffer = true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferReceiver::CBufferReceiver(char * _buffer, std::size_t _length,
/// 	bool _bDeleteBuffer )
///
/// \brief	Constructor. 
///
/// \param [in]	_buffer	If non-null, the buffer to use. 
/// \param	_length			The length of buffer. 
/// \param	_bDeleteBuffer	Buffer should be deleted in the destructor flag. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferReceiver::CBufferReceiver(char * _buffer, std::size_t _length, bool _bDeleteBuffer /*= true*/)
	: buffer(_buffer), buflength(_length), position(0), bDeleteBuffer(_bDeleteBuffer)
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferReceiver::~CBufferReceiver()
///
/// \brief	Destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferReceiver::~CBufferReceiver()
{
	tBRLock Lock( *this );
	deleteBuffer();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CBufferReceiver::setBuffer(char * _buffer, size_t _length, bool _bDeleteBuffer )
///
/// \brief	Sets a buffer. 
///
/// \param [in]	_buffer	If non-null, the buffer. 
/// \param	_length			The length of the buffer. 
/// \param	_bDeleteBuffer	Buffer should be deleted in the destructor flag. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CBufferReceiver::setBuffer(char * _buffer, size_t _length, bool _bDeleteBuffer /*= true*/)
{
	tBRLock Lock( * this );

	// delete old buffer
	deleteBuffer();

	bDeleteBuffer = _bDeleteBuffer;

	buffer = _buffer;

	buflength = _length;

	position = 0;
}


#define MIN(a, b) ((a) < (b) ? (a) : (b))

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	std::size_t CBufferReceiver::receive(char * data, std::size_t length)
///
/// \brief	Receives data. 
///
/// \param [in]	data	If non-null, the data pointer. 
/// \param	length		The length of data buffer. 
///
/// \return	received data length. 
////////////////////////////////////////////////////////////////////////////////////////////////////

std::size_t CBufferReceiver::receive(char * data, std::size_t length)
{
	tBRLock Lock( * this );

	if(buffer == NULL || buflength == 0 || position >= buflength)
		return 0;

	// how many bytes can we write?
	size_t maxwriteable = MIN(buflength - position, length);

	// write data
	memcpy(buffer + position, data, maxwriteable);

	position += maxwriteable;

	return maxwriteable;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CBufferReceiver::deleteBuffer()
///
/// \brief	Deletes a buffer (if it should be done). 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CBufferReceiver::deleteBuffer()
{
	if(!bDeleteBuffer)
		return;

	if(buffer == NULL)
		return;

	delete [] buffer;
	buffer = NULL;

	// zero length
	buflength = 0;

	// zero position
	position = 0;
}

// ------------ DISPATCHER -----------------------

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferDispatcher::CBufferDispatcher()
///
/// \brief	Default constructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferDispatcher::CBufferDispatcher()
	: buffer(NULL), buflength(0), position(0), bDeleteBuffer(true)
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferDispatcher::CBufferDispatcher(size_t length)
///
/// \brief	Constructor. 
///
/// \param	length	The length of the internal buffer. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferDispatcher::CBufferDispatcher(size_t length)
{
	tBDLock Lock( * this );

	buffer = new char[length];

	assert(buffer != NULL);

	buflength = length;

	position = 0;

	bDeleteBuffer = true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferDispatcher::CBufferDispatcher(char * _buffer, std::size_t _length,
/// 	bool _bDeleteBuffer )
///
/// \brief	Constructor. 
///
/// \param [in]	_buffer	If non-null, the buffer to use. 
/// \param	_length			The length of the buffer. 
/// \param	_bDeleteBuffer	Buffer should be deleted in the destructor flag. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferDispatcher::CBufferDispatcher(char * _buffer, std::size_t _length, bool _bDeleteBuffer /*= true*/)
	: buffer(_buffer), buflength(_length), position(0), bDeleteBuffer(_bDeleteBuffer)
{

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	CBufferDispatcher::~CBufferDispatcher()
///
/// \brief	Destructor. 
////////////////////////////////////////////////////////////////////////////////////////////////////

CBufferDispatcher::~CBufferDispatcher()
{
	tBDLock Lock( * this );

	deleteBuffer();
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CBufferDispatcher::setBuffer(char * _buffer, size_t _length, bool _bDeleteBuffer )
///
/// \brief	Sets a buffer. 
///
/// \param [in]	_buffer		If non-null, the output data buffer. 
/// \param	_length			The length of the buffer. 
/// \param	_bDeleteBuffer	Buffer should be deleted in the destructor flag. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CBufferDispatcher::setBuffer(char * _buffer, size_t _length, bool _bDeleteBuffer /*= false*/)
{
	tBDLock Lock( * this );

	// delete old buffer
	deleteBuffer();

	buffer = _buffer;

	bDeleteBuffer = _bDeleteBuffer;

	buflength = _length;

	position = 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	std::size_t CBufferDispatcher::dispatch(char * data, std::size_t length)
///
/// \brief	Dispatches data from the internal buffer. 
///
/// \param [out]	data	If non-null, the data pointer. 
/// \param	length			The length of data to write. 
///
/// \return	size of the dispatched data. 
////////////////////////////////////////////////////////////////////////////////////////////////////

std::size_t CBufferDispatcher::dispatch(char * data, std::size_t length)
{
	tBDLock Lock( * this );

	if(buffer == NULL || buflength == 0 || position >= buflength)
		return 0;

	// how many bytes can we write?
	size_t maxwriteable = MIN(buflength - position, length);

	// write data
	memcpy(data, buffer + position, maxwriteable);

	position += maxwriteable;

	return maxwriteable;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn	void CBufferDispatcher::deleteBuffer()
///
/// \brief	Deletes a buffer. 
////////////////////////////////////////////////////////////////////////////////////////////////////

void CBufferDispatcher::deleteBuffer()
{
	if(!bDeleteBuffer)
		return;

	if(buffer == NULL)
		return;

	delete [] buffer;
	buffer = NULL;

	// no length
	buflength = 0;

	// no position
	position = 0;
}

