/******************************************************************************
	\brief Base curl interface encapsulation
******************************************************************************/
#ifndef CCURL_H_INCLUDED
#define CCURL_H_INCLUDED

#define CURL_STATICLIB

#include <curl.h>
#include <curlTools/CCurlShared.h>
#include <string>
#include <map>
#include <iostream>
#include <MDSTk/Base/mdsLock.h>

namespace comm{

	//! CURL callback function type
	typedef size_t (*tCurlCallback)(void *ptr, size_t size, size_t nmemb, void *stream);

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CCurl
	///
	/// \brief	Abstract class of curl interface objects. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CCurl : public mds::base::CLockableObject<CCurl>
	{
	public:
		//! Type of CURL interface
		enum interfaceType {easy, multi};

		//! Constructor
		CCurl(interfaceType type);

		//! Destructor
		virtual ~CCurl();

		//! Initialize engine.
		virtual bool init(std::ostream * dstream = NULL, std::ostream * hstream = NULL, std::ostream * estream = NULL);

		//! Reset curl
		virtual bool reset() = 0;

		//! Stop engine.
		//virtual void stop() = 0;

		//! Perform operation
		virtual bool perform() = 0;

		//! Get error string.
		virtual std::string error(){return serror;}

		//! Set option template
		template<class optiondata> bool setopt(CURLoption option, optiondata data);

		//! Set option
		virtual bool setopt(CURLoption option, long data){
			return setopt<long>(option, data);
		}
		
		//! Set option
		virtual bool setopt(CURLoption option, void *data){
			return setopt<void *>(option, data);
		}

		//! Set char * option
		virtual bool  setopt(CURLoption option, const char *data){
			return setopt<const char *>(option, data);
		}

		//! Set error output stream
		virtual void  setErrOutput(std::ostream * stream);

		//! Get error output stream
		virtual std::ostream * getErrOutput(){
			return errout;
		}

		//! Set header output stream
		virtual void  setHeaderOutput(std::ostream * stream);

		//! Get header output stream
		virtual std::ostream * getHeaderOutput(){
			return headerout;
		}

		//! Set data output stream
		virtual void  setDataOutput(std::ostream * stream);

		//! Get data output stream
		virtual std::ostream * getDataOutput(){
			return dataout;
		}

		//! Set session timeout in seconnds
		void setSessionTimeout( long duration );

		//! Set operation timeout in seconds
		void setOperationTimeout( long duration );

		// Encode string to URL
		char * encode(const char * uri);

		// Decode URL to string
		char * decode(const char * uri, int * outputlength = NULL);

		//! Set callback function
		virtual void setCurlCallback(CURLoption cbType, tCurlCallback callback = NULL);

		//! Get curl easy handle
		CURL * getHandle()
		{
			return handle;
		}

		//! Set debug info on/off
		void setDebug(bool isOn = true)
		{
			setopt(CURLOPT_VERBOSE, isOn ? 1 : 0);
		}

		//! Set share 
		bool setShare( CCurlShared & shared );

	protected:
		//! CURL handle
		CURL * handle;

		//! Error string
		std::string serror;

		//! string buffer
		char *buffer[CURL_ERROR_SIZE];

		//! Type of used interface.
		interfaceType usedInterface;

		//! Http header
		struct curl_slist *slist;

		//! If not NULL, errors are written here.
		std::ostream * errout,
		//! If not NULL, data are written here.
					 * dataout,
	    //! If not NULL, header is written here.
					 * headerout;

	protected:
		//! Write error to out stream (if possible).
		void writee();

		//! Write handler
		static size_t write_data(void *ptr, size_t size, size_t nmemb, void *stream);

		//! Header handler
		static size_t header_callback(void *ptr, size_t size, size_t nmemb, void *stream = NULL);

		//! Do basic initialization.
		virtual bool basicInit(std::ostream * dstream, std::ostream * hstream, std::ostream * estream);

	}; // class CCurl


	//! Set option template
	template<class optiondata> bool CCurl::setopt(CURLoption option, optiondata data){

			if(handle == NULL){
				serror = "Curl was not properly initialized.";
				writee();
				return false;
			}
			
			// set option
			CURLcode code = curl_easy_setopt(handle, option, data);

			if(code != CURLE_OK){

				//some error occured, so get the string
				serror = curl_easy_strerror(code);
				writee();
				return false;
			} //if code != 0

			return true;
		} // setopt



} // namespace comm


#endif // CCURL_H_INCLUDED
