/******************************************************************************
	\brief Threads pool
******************************************************************************/
#ifndef VCE_THREADPOOL_H
#define VCE_THREADPOOL_H

#include <deque>

#include <MDSTk/System/mdsEvent.h>
#include <MDSTk/Base/mdsLock.h>

#include "vceManager.h"
#include "vceWorker.h"
#include "vceConnection.h"
#include "vceConnections.h"


namespace comm
{

//==============================================================================
/*!
 * Thread pool... 
 */ 
template <class Local, class Thread, int N>
class CThreadPool
    : public CManager
    , public mds::base::CLockableObject<CThreadPool<Local, Thread, N> >
{
public:
    //! Basic compilation time tests
    enum { WRONG_TEMPLATE_PARAMETER_LOCAL = Local::CLASS_LOCAL };
    enum { WRONG_TEMPLATE_PARAMETER_THREAD = Thread::CLASS_WORKER };

    //! Scoped lock    
    typedef typename mds::base::CLockableObject< CThreadPool<Local, Thread, N> >::CLock tLock;

    //! Default waiting timeout
    enum { TIMEOUT = 500 };

    //! Parameters
    typedef typename Local::tParams tParams;

public:
    //! Default constructor
    CThreadPool() : m_NotEmpty(false, false)
    {
        for( int i = 0; i < N; ++i )
        {
            m_pThreads[i] = new Thread(*this);
        }
    }

    //! Destructor
    virtual ~CThreadPool()
    {
        reset();
        for( int i = 0; i < N; ++i )
        {
            m_pThreads[i]->terminate(true, 5 * TIMEOUT);

            delete m_pThreads[i];
        }
    }

    //! Adds a job
    bool addJob(CJob& Job)
    {
        tLock Lock(*this);
        
        // do not add the same jobs
        typename tQueue::iterator it = m_Queue.begin();
        typename tQueue::iterator itEnd = m_Queue.end();
        for( ; it != itEnd; ++it )
        {
            if( Job.checkIdentity(*it) )
            {
                return true;
            }
        }
        
        // add a job copy 
        CJob *pCopy = Job.clone();
        if( pCopy )
        {
            m_Queue.push_back(pCopy);
        }
        
        // try to wake up a thread
        m_NotEmpty.set();
        
        return true;
    }
    
    //! Get job from the queue
    virtual CJob* getJob()
    {
        if( !m_NotEmpty.wait(TIMEOUT) )
        {
            return 0;
        }
        
        tLock Lock(*this);
        
        if( m_Queue.empty() )
        {
            return 0;
        }
        
        CJob *pJob = m_Queue.front();
        m_Queue.pop_front();
        
        // Ve fronte jsou stale joby, tak pustime dalsi thread
        if( !m_Queue.empty() )
        {
            m_NotEmpty.set();
        }
        
        return pJob;
    }
    
	//! Returns input queue length
	unsigned getInputQueueSize()
	{
		tLock Lock(*this);

		return m_Queue.size();
	}

    //! Reset threads pool - stop all jobs
    bool reset()
    {
        tLock Lock(*this);
        
        // Nic nepobezi, ale thready zustanou aktivni...
        m_NotEmpty.reset();
        
        // Ukoncit a vycistit vsechna connections
        for( int i = 0; i < N; ++i )
        {
            m_pThreads[i]->getLocalData().cancel();
        }
        
        // Projit frontu a resetovat GELDy
        typename tQueue::iterator it = m_Queue.begin();
        typename tQueue::iterator itEnd = m_Queue.end();
        for( ; it != itEnd; ++it )
        {
            (*it)->done();
            delete *it;
        }
        
        // Vycistit frontu
        m_Queue.clear();
        
        // Pockat na vsechny thready
        for( int i = 0; i < N; ++i )
        {
            m_pThreads[i]->waitForFinish();
            m_pThreads[i]->reinit();
        }
        
        return true;
    }
    
    //! Set default parameters
    bool setParams(const tParams& Params)
    {
		m_globalParams = Params;
        // Nastavit parametry vsem threadum...
        for( int i = 0; i < N; ++i )
        {
            m_pThreads[i]->getLocalData().setParams(Params);
        }
		
		return true;
    }
    
protected:
    //! Jobs queue type
    typedef std::deque<CJob *> tQueue;

protected:
    //! Jobs queue
    tQueue m_Queue;
    
    //! Threads pool
    Thread *m_pThreads[N];
    
    //! Job is in the queue event
    mds::sys::CEvent m_NotEmpty;
    
    //! Local parameters
	tParams m_globalParams;
};


typedef CWorker<CBinaryGeldConnection> CDefaultWorker;

typedef CThreadPool<CBinaryGeldConnection, CDefaultWorker, 10> CDefaultThreadPool;


} // namespace comm

#endif // VCE_THREADPOOL_H
