/******************************************************************************
	\brief Header data transfer module
******************************************************************************/
#ifndef HMODULE_H_INCLUDED_
#define HMODULE_H_INCLUDED_

// Header parsing and storing
#include <curlTools/header.h>
// Input stream
#include <istream>
#include <cstring>
// Curl slist handler
#include <curlTools/slist.h>

// Smart pointers
#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Base/mdsLock.h>

namespace comm
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CHeaderParser
	///
	/// \brief	Header parser. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CHeaderParser :
		public mds::base::CObject,
		public mds::base::CLockableObject< CHeaderParser >
	
	{

	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CHeaderParser);

		//! Lock type
		typedef mds::base::CLockableObject< CHeaderParser >::CLock tHPLock;

		//! Constructor
		CHeaderParser():
			parser( new CHeaderReader )
		{

		}

        //! Virtual destructor.
        virtual ~CHeaderParser() {}

		//! Parse header
		virtual size_t parse(char * data, std::size_t length);

		//! Find information
		virtual tParametersVector * findValue( std::string parameter )
		{
			tHPLock Lock( *this );

			std::map< std::string, tParametersVector >::iterator i =  header_map.find(parameter);

			if( i != header_map.end() )
			{
				return & ( i->second );
			}

			return NULL;
		}

		

		//! Reset - clean all
		void reset()
		{
			tHPLock Lock( *this );
			header_map.clear();
		}

	protected:
		//! Lines of header
		tHeaderMap header_map;

		//! Header reader object
		CHeaderReaderPtr parser;


	}; // class headerParser

	//! Smart pointer
	typedef CHeaderParser::tSmartPtr CHeaderParserPtr;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CHeaderCreator
	///
	/// \brief	Header creator. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CHeaderCreator :
		public mds::base::CObject
	
	{

	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CHeaderCreator);

        //! Virtual destructor.
        virtual ~CHeaderCreator() {}

        //! Set (or add) header line.
		void setValue(const std::string & parameter, const std::string & value, bool bModeAppend = false);

		//! Create header from obtained parameters.
		CSlist * generateHeader();

	protected:
		//! Type of parameters map
		typedef std::map<std::string, std::string> tParameterMap;
		
		//! Map of parameters
		tParameterMap parameter_map;

	}; // class headerCreator

	//! Smart pointer
	typedef CHeaderCreator::tSmartPtr CHeaderCreatorPtr;


} // namespace comm


// HMODULE_H_INCLUDED_
#endif
