/******************************************************************************
	\brief Data transfering modules and helper classes
******************************************************************************/
#ifndef DTMODULE_H_INCLUDED_
#define DTMODULE_H_INCLUDED_

#include <cstring>
#include <assert.h>
#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Base/mdsFactory.h>
#include <comm/hmodule.h>

namespace comm
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CReceiver
	///
	/// \brief	Receives data. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CReceiver :
		public mds::base::CObject
	{
	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CReceiver);

        //! Declare virtual destructor!
        virtual ~CReceiver() {}

		//! Create function used in the factory
		static CReceiver * create()
		{
			return new CReceiver;
		}


		//! Receive data
		virtual std::size_t receive(char * data, std::size_t length)
		{
			return 0;
		}

		//! Do on init (after constructor)
		virtual void onInit(void * data = NULL)
		{
		}

		//! Called before transfer start
		virtual void onTransferStart()
		{

		}

		//! Called after transfer end
		virtual void onTransferEnd()
		{

		}

		//! Do on exit (before destructor)
		virtual void onExit()
		{
		}


		//! Get type
		long getType() { return m_type; }

	protected:
		//! Receiver type
		long m_type;


	}; // class CReceiver

	//! Smart pointer type
	typedef CReceiver::tSmartPtr CReceiverPtr;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CReceiverInfo
	///
	/// \brief	Information about the receiver. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CReceiverInfo :
		public mds::base::CObject
	{
	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CReceiverInfo);

		//! Constructor
		CReceiverInfo();

		//! Constructor
		CReceiverInfo( long typeID, CReceiver * receiver );

		//! Constructor
		CReceiverInfo( CReceiver * receiver );

		
	public:
		//! Id of the receiver type
		long m_receiverTypeID;

		//! Receiver pointer
		CReceiverPtr m_receiver;

	private:
		//! Disable copy constructor
		CReceiverInfo( const CReceiverInfo & );

		//! Disable assignement operator
		CReceiverInfo & operator = ( const CReceiverInfo & );

	}; // class CreceiverInfo

	typedef CReceiverInfo::tSmartPtr CReceiverInfoPtr;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CDispatcher
	///
	/// \brief	Dispatcher dispatches data. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CDispatcher :
		public mds::base::CObject
	{
	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CDispatcher);

        //! Declare virtual destructor!
        virtual ~CDispatcher() {}

		//! Dispatch data, length is maximum ammount of data, that can be dispatched.
		virtual std::size_t dispatch(char * data, std::size_t length)
		{
			return 0;
		}

		//! Do on init (after constructor)
		virtual void onInit(void * data = NULL)
		{
		}

		//! Called before transfer start
		virtual void onTransferStart()
		{

		}

		//! Called after transfer end
		virtual void onTransferEnd()
		{

		}

		//! Do on exit (before destructor)
		virtual void onExit()
		{
		}

		//! Create function used in the factory
		static CDispatcher * create()
		{

		}

	protected:


	}; // class CDispatcher

	//! Smart pointers type
	typedef CDispatcher::tSmartPtr CDispatcherPtr;

	//! Factory used to generate dispatchers
	typedef mds::base::CFactory< CDispatcher, long > tDispatcherFactory;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CNullReceiver
	///
	/// \brief	Null receiver - receiver example. Just sends all receiving data to \dev\null. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CNullReceiver : public CReceiver
	{
	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CNullReceiver);

		//! Create function used in the factory
		virtual CReceiver * create()
		{
			return new CNullReceiver;
		}

		//! Receive data
		virtual std::size_t receive(char * data, std::size_t length)
		{
			return length;
		}

	};

	typedef CNullReceiver::tSmartPtr CNullReceiverPtr;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CNullDispatcher
	///
	/// \brief	Null dispatcher - dispatcher example. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CNullDispatcher : public CDispatcher
	{
	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CNullDispatcher);

		//! Constructor
		CNullDispatcher(bool _assertWhenCalled = false)
			: assertWhenCalled(_assertWhenCalled)
		{
		}

		//! Create function used in the factory
		virtual CDispatcher * create()
		{
			return new CNullDispatcher();
		}


		//! Dispatch data
		virtual std::size_t dispatch(char * data, std::size_t length)
		{
			assert(assertWhenCalled);
			return length;
		}

	protected:
		//! Call assert, when trying to dispatch some data.
		bool assertWhenCalled;

	};

	typedef CNullDispatcher::tSmartPtr CNullDispatcherPtr;


} // namespace comm


// DTMODULE_H_INCLUDED_
#endif 
