/******************************************************************************
	\brief Curl connection - processing thread specialization
******************************************************************************/

#ifndef VCE_CONNECTION_H
#define VCE_CONNECTION_H

#include <MDSTk/System/mdsEvent.h>
#include <curlTools/curlEasy.h>
#include <curlTools/curlOption.h>
#include <comm/curlDTInfo.h>
#include <comm/local.h>
#include <comm/errorHandler.h>

// STL
#include <string>


namespace comm
{
////////////////////////////////////////////////////////////////////////////////////////////////////
/// \class	CConnectionParams
///
/// \brief	Curl connection parameters. 
////////////////////////////////////////////////////////////////////////////////////////////////////

class CConnectionParams
{
public:
	

public:
	
	//! Simple constructor
	CConnectionParams() : m_sessionID(0), m_clientID("") { }

	//! Copy constructor
	CConnectionParams( const CConnectionParams & params );

	
	//! Settings map type
	typedef std::map< CURLoption, CCurlOptionBasePtr > tParametersMap;

    // Nastaveni parametru
    // - Melo by fungovat spPool->setParams(CConnectionParams().setURL("a").setLogin("b").setPasswd("c"));
    CConnectionParams& setURL(const std::string& ssURL)
    {
        // Set url as a string option
		m_options[ CURLOPT_URL ] = new CCurlOptionString( CURLOPT_URL, ssURL );
        return *this;
    }

	//! Set login name and password.
	CConnectionParams& setLoginPassword(const std::string& ssLogin, const std::string & ssPassword)
    {
        // Set login and password as a string option
		m_options[ CURLOPT_USERPWD ] = new CCurlOptionString( CURLOPT_USERPWD, ssLogin + ":" + ssPassword + '\0' );
        return *this;
    }

	//! Set long-type parameter
	CConnectionParams& setLong( const CURLoption & option, const long & value )
	{
		m_options[ option ] = new CCurlOptionLong( option, value );
	}

	//! Set string-type parameter
	CConnectionParams& setLong( const CURLoption & option, const std::string & value )
	{
		m_options[ option ] = new CCurlOptionString( option, value );
	}

	//! Set void pointer-type parameter
	CConnectionParams& setLong( const CURLoption & option, void * value )
	{
		m_options[ option ] = new CCurlOptionPtrVoid( option, value );
	}

	//! Set session id
	CConnectionParams& setSessionID( int id ) { m_sessionID = id; return *this; }

	//! Set client id
	CConnectionParams& setClientID( const std::string & id ){ m_clientID = id; return *this; }
    
	//! Options map
	tParametersMap m_options;

	//! Assignement operator
	CConnectionParams operator = ( const CConnectionParams & params );

	//! Session id
	int m_sessionID;

	//! Client id
	std::string m_clientID;

};

//==============================================================================
/*
 * Forward declaration of CConnection callbacks.
 */
//! Data sending callback - calls dispatcher.
size_t dispatchCallback(void *ptr, size_t size, size_t nmemb, void *userData);
//! Data receiving callback - calls receiver.
size_t receiveCallback(void *ptr, size_t size, size_t nmemb, void *userData);
//! Header receiving callback
size_t headerReceiveCallback(void *ptr, size_t size, size_t nmemb, void *userData);

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \property	template < class tpDispatcher,
/// 			class tpReceiver > class CConnection : public CLocal
///
/// \brief	 Per thread connection data. 
////////////////////////////////////////////////////////////////////////////////////////////////////

template
< 
	class tpDispatcher,
	class tpReceiver
>
class CConnection : public CLocal
{
public:
    //! Data send timeout
    enum { TIMEOUT = 20000 };

    //! Parameters
    typedef CConnectionParams tParams;

public:
    //! Default constructor.
    CConnection() : m_Cancel(false, false) 
	{
		// Create receiver
		m_dtinfo.m_receiver = new tpReceiver;
		// Create dispatcher
		m_dtinfo.m_dispatcher = new tpDispatcher;
		// Create header parser
		m_dtinfo.m_headerParser = new CHeaderParser;
		// Set cancel event ptr
		m_dtinfo.m_cancelEvent = & m_Cancel;
		// Set stage
		m_dtinfo.m_stage = CDTInfo::STAGE_WAITING;
	}

	//! Constructor - set dispatcher, receiver and header parser. Set NULL if you want to use defaults. All objects are deleted in destructor!
	CConnection( CDispatcher * dispatcher, CReceiver * receiver, CHeaderParser * header_parser )
		: m_Cancel(false, false) 
	{
		if( receiver == 0 )
		{
			// Create receiver
			m_dtinfo.m_receiver = new tpReceiver;
		}
		else
		{
			m_dtinfo.m_receiver = receiver;
		}

		if( dispatcher == 0 )
		{
			// Create dispatcher
			m_dtinfo.m_dispatcher = new tpDispatcher;
		}
		else
		{
			m_dtinfo.m_dispatcher =  dispatcher;
		}

		if( header_parser == 0 )
		{
			// Create header parser
			m_dtinfo.m_headerParser = new CHeaderParser;
		}
		else
		{
			m_dtinfo.m_headerParser = header_parser;
		}

		// Do check
		assert( m_dtinfo.m_receiver != 0 && m_dtinfo.m_dispatcher != 0 && m_dtinfo.m_headerParser != 0 );

		// Set cancel event ptr
		m_dtinfo.m_cancelEvent = & m_Cancel;
		// Set stage
		m_dtinfo.m_stage = CDTInfo::STAGE_WAITING;
	}
    
    //! Virtual destructor.
    virtual ~CConnection() 
	{
		//! Clear dt info
		delete m_dtinfo.m_receiver;
		delete m_dtinfo.m_dispatcher;
		delete m_dtinfo.m_headerParser;
	}
   

    //! Re-initializaes all the internal data
    virtual int reinit()
    {
        m_Cancel.reset();
        
        // Init curl
		m_curl.reset();

		// Set stage
		m_dtinfo.m_stage = CDTInfo::STAGE_WAITING;

		return 0;
    }

    //! Cancel current operation
    virtual bool cancel()
    {
        return m_Cancel.set();
    }
    
    //! Is operation cancelled?
    virtual bool isCanceled()
    {
        return m_Cancel.isSet();
    }


	//! Set content type string. Set "" if you don't care...
	void setExpectedContentType( const std::string & content_type )
	{
		m_dtinfo.m_contentType = content_type;
	}

	//! Get data transfer info
	CDTInfo & getDTInfo()
	{
		return m_dtinfo;
	}

	//! Set parameters
	bool setParams(const tParams& Params)
	{
		m_params = Params;
		return true;
	}


protected:
	//! Perform operation
	bool perform()
	{
		// Apply parameters
		applyParams( m_params );

		// Prepare all...
		m_dtinfo.onTransferStart();		
		// Do perform
		bool retval = m_curl.perform();

		if( !retval )
		{
			// Compose error info
			std::string strError("perform: " + m_curl.error() );
			// Error - unknown content type
			MDS_SINGLETON( CErrorHandler ).cbError( new CErrorInfo( CErrorInfo::ET_CONNECTION, CErrorInfo::EP_LOG_AND_CALL, 0, strError, "vceConnection" ) );
		}

		// Transfer completed...
		m_dtinfo.onTransferEnd();

		// Reset curl
		m_curl.reset();

		return retval;
	}

	//! Set curl callbacks
	void setCallbacks()
	{
		// - set callbacks
		m_curl.setCurlCallback(CURLOPT_READFUNCTION, &dispatchCallback );
		m_curl.setCurlCallback(CURLOPT_WRITEFUNCTION, &receiveCallback );
		m_curl.setCurlCallback(CURLOPT_HEADERFUNCTION, &headerReceiveCallback );

		// - set user data for callbacks
		m_curl.setopt(CURLOPT_WRITEDATA, & m_dtinfo );
		m_curl.setopt(CURLOPT_READDATA, & m_dtinfo );
		m_curl.setopt(CURLOPT_WRITEHEADER, & m_dtinfo );
	}

	// Nastaveni parametru
    bool applyParams(const CConnectionParams& Params)
    {
		// Apply given params
		typename CConnectionParams::tParametersMap::const_iterator i;
		for( i = Params.m_options.begin(); i != Params.m_options.end(); ++i )
			i->second->apply( m_curl );

		// Apply default params - callbacks
		setCallbacks();

		return true;
    }
    
protected:
    //! Event signalled on exit.
    mds::sys::CEvent m_Cancel;

	//! Curl interface
	CCurlEasy m_curl;

	//! Data transfer classes info
	CDTInfo m_dtinfo;

	//! Parameters
	tParams m_params;

	// FRIEND FUNCTIONS
	//! Data sending callback - calls dispatcher.
	friend size_t dispatchCallback(void *ptr, size_t size, size_t nmemb, void *userData);
    //! Data receiving callback - calls receiver.
	friend size_t receiveCallback(void *ptr, size_t size, size_t nmemb, void *userData);
	//! Header receiving callback
	friend size_t headerReceiveCallback(void *ptr, size_t size, size_t nmemb, void *userData);
};

} // namespace comm

#endif // VCE_CONNECTION_H
