/******************************************************************************
	\brief Data transfer modules implementation - buffered data
******************************************************************************/
#ifndef BUFFERDT_H_INCLUDED_
#define BUFFERDT_H_INCLUDED_

#include "dtmodule.h"
#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsLock.h>
#include <MDSTk/Base/mdsSharedPtr.h>

namespace comm
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CBufferReceiver
	///
	/// \brief	To the buffer receiver. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CBufferReceiver : 
		public CReceiver,
		public mds::base::CLockableObject< CBufferReceiver >
	{
	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CBufferReceiver);

		//! Lock type
		typedef mds::base::CLockableObject< CBufferReceiver >::CLock tBRLock;

		//! Constructor
		CBufferReceiver();

		//! Constructor
		CBufferReceiver(char * _buffer, size_t _length, bool _bDeleteBuffer = false);

		//! Constructor
		CBufferReceiver(size_t length);

		//! Destructor
		virtual ~CBufferReceiver();

		//! Set new buffer.
		void setBuffer(char * _buffer, size_t _length, bool _bDeleteBuffer = false);

		//! Receive data
		virtual std::size_t receive(char * data, std::size_t length);

		//! Get written data length
		size_t getLength()
		{
			return position;
		}

		//! Get buffer pointer.
		char * getBuffer()
		{
			return buffer;
		}

		//! Create instance
		static CReceiver * create()
		{
			return new CBufferReceiver;
		}

	protected:
		//! Delete buffer (if we have to).
		void deleteBuffer();

	protected:
		//! Buffer
		char * buffer;

		//! Length of buffer
		size_t buflength;

		//! Current position
		size_t position;

		//! Delete buffer on exit?
		bool bDeleteBuffer;
	}; // class CBufferReceiver

	typedef CBufferReceiver::tSmartPtr CBufferReceiverPtr;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class	CBufferDispatcher
	///
	/// \brief	From the buffer dispatcher. 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CBufferDispatcher : 
		public CDispatcher,
		public mds::base::CLockableObject< CBufferDispatcher >
	{

	public:
		//! Smart pointer type
		//! - declares type tPointer
		MDS_SHAREDPTR(CBufferDispatcher);

		//! lock type
		typedef mds::base::CLockableObject< CBufferDispatcher >::CLock tBDLock;

		//! Constructor
		CBufferDispatcher();

		//! Constructor
		CBufferDispatcher(char * _buffer, size_t _length, bool _bDeleteBuffer = true);

		//! Constructor
		CBufferDispatcher(size_t length);

		//! Destructor
		virtual ~CBufferDispatcher();

		//! Set new buffer.
		void setBuffer(char * _buffer, size_t _length, bool _bDeleteBuffer = true);

		//! Receive data
		virtual std::size_t dispatch(char * data, std::size_t length);

		//! Get written data length
		size_t getLength()
		{
			return position;
		}

		//! Get buffer pointer.
		char * getBuffer()
		{
			return buffer;
		}

		//! Create instance
		static CDispatcher * create()
		{
			return new CBufferDispatcher;
		}

	protected:
		//! Delete buffer (if we have to).
		void deleteBuffer();

	protected:
		//! Buffer
		char * buffer;

		//! Length of buffer
		size_t buflength;

		//! Current position
		size_t position;

		//! Delete buffer on exit?
		bool bDeleteBuffer;


	}; // class CBufferDispatcher

	typedef CBufferDispatcher::tSmartPtr CBufferDispatcherPtr;

} // namespace comm

// BUFFERDT_H_INCLUDED_
#endif
