
#include <stdio.h>

#ifdef _OPENMP1
#include <omp.h>
#endif

#include <opencv/cv.h>
#include "vplcore.h"
#include "vplgllfgraddist.h"
#include "vpltiming.h"


int vplGllfGradDistClear( struct VplGllfGradDist * gd );


struct VplGllfGradDist * vplGllfGradDistCreate( void )
{
	struct VplGllfGradDist * gd = NULL;

	CV_FUNCNAME( "vplGllfGradDistCreate" );
	__BEGIN__

	gd = ( VplGllfGradDist * )malloc( sizeof( VplGllfGradDist ) );
	if( !gd ) return NULL;

	gd->hist = NULL;
	gd->frame	= NULL;
	gd->subframe = NULL;
	gd->dx  = NULL;
	gd->dy  = NULL;
	gd->mag = NULL;
	gd->ori = NULL;

	gd->pyrs  = 0;
	gd->bins  = 0;
	gd->cells = 0;
	gd->aperture_size = 3;

	gd->gridOverlap = 1.0;
	gd->scale_0 = 0;

	//vtm = vplTimeMeasuresCreate( 20 );

	__END__
	return gd;
}


struct VplGllfGradDist * vplGllfGradDistInit( struct VplGllfGradDist * gd, int pyrs, int bins, float gridOverlap, int aperture_size )
{
	CV_FUNCNAME( "vplGllfGradDistInit" );
	__BEGIN__
	int i = 0;

	if( !gd ) gd = vplGllfGradDistCreate();
	else           vplGllfGradDistClear( gd );

	gd->gridOverlap = gridOverlap;
	gd->pyrs  = pyrs;
	gd->bins  = bins;
	gd->cells = 0;
	i = gd->pyrs;
	while( --i >= 0 ) 
		gd->cells += ( 1 << i )*( 1 << i );

	gd->hist = cvCreateMat( gd->cells, bins, VPL_DATA_MAT_TYPE );
	cvZero( gd->hist );

	gd->aperture_size = aperture_size;

	__END__
	return gd;
}


int	vplGllfGradDistGetFeature( struct VplGllfGradDist * gd, CvMat * feature )
{
	CV_FUNCNAME( "vplGllfGradDistGetFeature" );
	__BEGIN__
	CvMat m;

	if( !gd ) EXIT;
	if( !feature || feature->cols != gd->cells*gd->bins || CV_MAT_TYPE(feature->type) != CV_32SC1 ) EXIT;

	m = cvMat( 1, gd->cells * gd->bins, gd->hist->type, gd->hist->data.ptr );
	cvConvertScale( &m, feature, 255., 0.0 );
	//cvMatPrint( feature, 0 );
	
	return 1;

	__END__
	return 0;
}

double vplGllfGradDistDist( struct VplGllfGradDist * gd, CvMat * f1, CvMat * f2 )
{
	if( !gd || !f1 || !f2 ) return -1.0;
	
	return cvNorm( f1, f2, CV_L1, 0 )/(255.*gd->cells);
	//return cvNorm( f1, f2, CV_L2, 0 );
}

int	vplGllfGradDistGetSize( struct VplGllfGradDist * gd )
{
	if( !gd ) return 0;
	return gd->cells*gd->bins;
}



//#define RESET_SUB_MAT( dst, orig ) { *(dst) = cvMat((orig)->rows, (orig)->cols, (dst)->type, (dst)->data.ptr ); (dst)->hdr_refcount = 1; *((dst)->refcount) = 1;}

#define SUB_MAT( subwnd ) \
			subframe = cvGetSubRect( subframe, &_subframe, subwnd );	\
			dx  = cvGetSubRect( dx,  &_dx,  subwnd );					\
			dy  = cvGetSubRect( dy,  &_dy,  subwnd );					\
			mag = cvGetSubRect( mag, &_mag, subwnd );					\
			ori = cvGetSubRect( ori, &_ori, subwnd );					


int	vplGllfGradDistExtract( struct VplGllfGradDist * gd, CvArr * image, CvMat * feature )
{

	CV_FUNCNAME( "vplGllfGradDistExtract" );
	__BEGIN__

	int i, j, row_i = 0;
	CvMat swap;
	double map;
    CvMat srcstub, * src = (CvMat*)image;
	CvMat * dx, * dy, * mag, * ori, * subframe, * frame;
	CvMat  _dx,  _dy,  _mag,  _ori,  _subframe,  _frame;
	CvRect subwnd;
	
	CvSize wnd;
	CvRect w;
	int gr;

	if( !gd || !image ) return -1;

	src = cvGetMat( image, &srcstub, NULL, 0 );
	if( CV_MAT_TYPE( src->type ) != CV_8UC1 ) return -2;

	if( gd->frame && !CV_ARE_SIZES_EQ( gd->frame, src ) )
	{
		if( gd->frame )		cvReleaseMat( &gd->frame );
		if( gd->subframe )	cvReleaseMat( &gd->subframe );
		if( gd->dx )		cvReleaseMat( &gd->dx );
		if( gd->dy )		cvReleaseMat( &gd->dy );
		if( gd->mag )		cvReleaseMat( &gd->mag );
		if( gd->ori )		cvReleaseMat( &gd->ori );
	}

	if( !gd->frame )
	{
		gd->frame    = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->subframe = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->dx		 = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->dy 		 = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->mag		 = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
		gd->ori		 = cvCreateMat( src->rows, src->cols, VPL_DATA_MAT_TYPE );
	}

	subwnd = cvRect( 0, 0, src->cols, src->rows );
	subframe = gd->subframe;
	dx  = gd->dx;
	dy  = gd->dy;
	mag = gd->mag;
	ori = gd->ori;
	frame = cvGetSubRect( gd->frame, &_frame, subwnd );
	cvConvert( src, frame );

	// down-scale input image
	for( i = 0; i < gd->scale_0; ++i )
	{
		subwnd = cvRect( 0, 0, frame->cols/2, frame->rows/2 );
		SUB_MAT( subwnd );
		cvPyrDown( frame, subframe, CV_GAUSSIAN_5x5 );
		CV_SWAP( _frame, _subframe, swap );
	}

	cvZero( gd->hist );
	map = 1.0 * gd->bins / 361;
	for( i = gd->pyrs-1; i >= 0; --i )
	{
		CvMat m;
		int x, y;
		float * pori, * pmag;

		cvSobel( frame, dx, 1, 0, gd->aperture_size );
		cvSobel( frame, dy, 0, 1, gd->aperture_size );
		cvCartToPolar( dx, dy, mag, ori, 1 );

		gr = ( 1 << i );
		wnd = cvSize( cvRound( frame->cols / ( gd->gridOverlap * ( gr - 1 ) + 1 ) ), 
					  cvRound( frame->rows / ( gd->gridOverlap * ( gr - 1 ) + 1 ) ) );

		for( j = 0; j < gr*gr; ++j )
		{
			w = cvRect( cvRound( (j%gr) * wnd.width  * gd->gridOverlap ), 
						cvRound( (j/gr) * wnd.height * gd->gridOverlap ), 
						wnd.width, wnd.height );

			if( w.x+w.width  >= frame->cols ) w.width  = frame->cols - w.x;
			if( w.y+w.height >= frame->rows ) w.height = frame->rows - w.y;

			cvGetRow( gd->hist, &m, row_i );

			pori = ((float *)( ori->data.ptr + w.y*ori->step )) + w.x;
			pmag = ((float *)( mag->data.ptr + w.y*mag->step )) + w.x;
			for( y = 0; y < w.height; ++y )
			{
				for( x = 0; x < w.width; ++x )	
					m.data.fl[(int)(pori[x] * map)] += pmag[x];

				pori = (float *)( (uchar *)pori + ori->step );
				pmag = (float *)( (uchar *)pmag + mag->step );
			}

			/* normalization */
			/* filter nema vliv */
			/* normalizace pro velikost obrazku - pocet pixelu */
			cvConvertScale( &m, &m, 1.0 / ( wnd.width * wnd.height ), 0 );
			cvNormalize( &m, &m, 1., 0., CV_L2, 0 );

			row_i++;
		}

		subwnd = cvRect( 0, 0, frame->cols>>1, frame->rows>>1 );
		SUB_MAT( subwnd );
		//vplTimeMeasuresAdd( vtm, 2 );

		//vplTimeMeasuresStart( vtm, 3 );
		cvPyrDown( frame, subframe, CV_GAUSSIAN_5x5 );
		CV_SWAP( _frame, _subframe, swap );
		//vplTimeMeasuresAdd( vtm, 3 );

	}
	//vplTimeMeasuresEnd( vtm, 0 );	vplTimeMeasuresEnd( vtm, 1 );	vplTimeMeasuresEnd( vtm, 2 );	vplTimeMeasuresEnd( vtm, 3 );
	//printf( "Sobel: %f; Polar: %f; Hist: %f; PyrDown: %f; Sum: %f; ", 		0.001*vplTimeMeasuresAvgGet( vtm, 0 ), 0.001*vplTimeMeasuresAvgGet( vtm, 1 ), 0.001*vplTimeMeasuresAvgGet( vtm, 2 ), 0.001*vplTimeMeasuresAvgGet( vtm, 3 ),		0.001*( vplTimeMeasuresAvgGet( vtm, 0 ) + vplTimeMeasuresAvgGet( vtm, 1 ) + vplTimeMeasuresAvgGet( vtm, 2 ) + vplTimeMeasuresAvgGet( vtm, 3 ) ) );

	/* normalization */
	/* dulezite - invariance na kontrast v obraze (v kontrastnim obraze jsou velikosti vektoru vetsi, nam jde o strukturu, ne o kontrast) */
	//cvNormalize( gd->hist, gd->hist, 1., 0., CV_L2, 0 );

	if( feature ) vplGllfGradDistGetFeature( gd, feature );

	__END__
	return 1;
}




//int	vplGllfGradDistRender( struct VplGllfGradDist * gd, CvArr * canvas )
//{
//	CV_FUNCNAME( "vplGllfGradDistRender" );
//	__BEGIN__
//
//    CvMat srcstub, *src = (CvMat*)canvas;
//	int i, j;
//	double xs, ys, mav;
//
//	if( !gd || !canvas ) return -1;
//
//	src = cvGetMat( canvas, &srcstub, &i, 0 );
//	cvZero( src );
//
//	xs = 1.0 * src->cols / gd->bins;
//	ys = 1.0 * src->rows / gd->pyrs;
//	for( j = 0; j < gd->pyrs; ++j )
//	{
//		CvMat m;
//		cvGetRow( gd->hist, &m, j );
//		cvMinMaxLoc( &m, 0, &mav, 0, 0, 0 );
//
//		for( i = 0; i < gd->bins; ++i )
//		{
//			int val = cvRound( m.data.db[i] * 255 / mav );
//			cvRectangle( src, cvPoint( (int)(i*xs), (int)(j*ys) ), cvPoint( (int)((i+1)*xs), (int)((j+1)*ys) ), CV_RGB(val,val,val), CV_FILLED, 8, 0 );
//		}
//	}
//	cvRectangle( src, cvPoint( 0, 0 ), cvPoint( src->cols-1, src->rows-1 ), CV_RGB(100,100,200), 3, 8, 0 );
//
//	__END__
//	return 1;
//}



int vplGllfGradDistClear( struct VplGllfGradDist * gd )
{
	CV_FUNCNAME( "vplGllfGradDistClear" );

	if( !gd ) return -1;

	if( gd->hist ) cvReleaseMat( &gd->hist );

	if( gd->frame )	   cvReleaseMat( &gd->frame );
	if( gd->subframe ) cvReleaseMat( &gd->subframe );
	if( gd->dx )	   cvReleaseMat( &gd->dx );
	if( gd->dy )	   cvReleaseMat( &gd->dy );
	if( gd->mag )	   cvReleaseMat( &gd->mag );
	if( gd->ori )	   cvReleaseMat( &gd->ori );

	gd->pyrs  = 0;
	gd->bins  = 0;
	gd->cells = 0;

	return 1;
}


int vplGllfGradDistRelease( struct VplGllfGradDist ** gd )
{
	CV_FUNCNAME( "vplGllfGradDistRelease" );

	struct VplGllfGradDist * g;

	if( !gd || !*gd ) return 1;

	g = *gd;
	vplGllfGradDistClear( g );

	free( g );
	*gd = NULL;

	return 1;
}
















struct cImagePyr * imagePyrCreate( int octaves, int levels, int mattype )
{
	struct cImagePyr * ip = NULL;
	ip = (struct cImagePyr *)malloc(sizeof(struct cImagePyr));
	ip->octaves = MAX( 1, octaves );
	ip->levels  = MAX( 1, levels );
	ip->mattype = mattype;
	ip->layers = (CvMat **)malloc( sizeof(CvMat*) * ip->octaves * ip->levels );
	memset( ip->layers, 0, sizeof(CvMat*) * ip->octaves * ip->levels );
	return ip;
}

int imagePyrInit( struct cImagePyr * imagePyr, CvArr * image )
{
	struct cImagePyr * ip = imagePyr;
	int i, k;
    CvMat srcstub, *src = (CvMat*)image, * l;

	if( !imagePyr ) return -1;
	if( image ) src = cvGetMat( image, &srcstub, &i, 0 );
	l = imagePyrGet(ip,0,0);

	if( !image || ( src && l && !CV_ARE_SIZES_EQ(src,l) ) )
		for( i = 0; i < ip->levels*ip->octaves; ++i ) 
			if( ip->layers[i] ) cvReleaseMat( &ip->layers[i] );

	if( !image ) return 1;

	if( !imagePyrGet(ip,0,0) )
		for( i = 0; i < ip->octaves; ++i ) 
			for( k = 0; k < ip->levels; ++k ) 
				imagePyrGet(ip,i,k) = cvCreateMat( src->rows>>i, src->cols>>i, ip->mattype );

	return 1;
}


void imagePyrRelease( struct cImagePyr ** imagePyr )
{

	if( !imagePyr || !*imagePyr ) return;
	imagePyrInit( *imagePyr, NULL );
	free( *imagePyr );
	*imagePyr = NULL;
	return;
}


struct cImagePyr * imagePyrSobel( struct cImagePyr * imagePyr, CvArr * image, int pyrCount, int aperture_size )
{
	struct cImagePyr * ip = imagePyr;
	int i;

	if( ip && ip->octaves != pyrCount )
		imagePyrRelease( &ip );
	if( !ip )
		ip = imagePyrCreate( pyrCount, 5, CV_32FC1 );
			
	if( !image ) return ip;

	if( imagePyrInit( ip, image ) < 0 ) return ip;
	
	aperture_size = MAX( 3, aperture_size ) | 1;

	for( i = 0; i < pyrCount; ++i )
	{
		if( i == 0 ) cvConvertScale( image, imagePyrGet( ip, i, 0 ), 1.0, 0.0 );
		else cvPyrDown( imagePyrGet(ip,i-1,0), imagePyrGet(ip,i,0), CV_GAUSSIAN_5x5 );

		cvSobel( imagePyrGet(ip,i,0), imagePyrGet(ip,i,1), 1, 0, aperture_size );
		cvSobel( imagePyrGet(ip,i,0), imagePyrGet(ip,i,2), 0, 1, aperture_size );
		cvCartToPolar( imagePyrGet(ip,i,1), imagePyrGet(ip,i,2), imagePyrGet(ip,i,3), imagePyrGet(ip,i,4), 1 );
	}

	return ip;
}


CvMat * gradWeighted( CvMat * magnitude, CvMat * orientation, CvRect wnd, CvMat * feature, CvMat * filter, CvMat * filterBuff )
{
	CvMat * mag, _mag, * ori, _ori;
	float * pmag, * pori, map, del;
	int x, y;

	if( !magnitude || !orientation || !feature ) return NULL;
	if( filter && !filterBuff ) return NULL;
	if( filter && ( filter->rows != wnd.height || filter->cols != wnd.width || (CV_MAT_TYPE(filter->type) != CV_32FC1 ) ) ) return NULL;
	if( !CV_ARE_SIZES_EQ(filter,filterBuff) ) return NULL;
	if( !CV_ARE_TYPES_EQ(filter,filterBuff) ) return NULL;

	if( wnd.x < 0 || wnd.y < 0 || wnd.x+wnd.width >= magnitude->cols || wnd.y+wnd.height >= magnitude->rows ) return feature;
	if( !CV_ARE_SIZES_EQ(magnitude,orientation) ) return NULL;

	mag = cvGetSubRect( magnitude,   &_mag, wnd );
	ori = cvGetSubRect( orientation, &_ori, wnd );
	map = (float) 1.0 * feature->cols / 361;
	del = (float) 180.0 / feature->cols;

	if( filter ) 
	{
		cvMul( filter, mag, filterBuff, 1.0 );
		mag = filterBuff;
	} 
	pmag = (float *)( mag->data.ptr );
	pori = (float *)( ori->data.ptr );

	for( y = 0; y < ori->rows; ++y )
	{
		for( x = 0; x < ori->cols; ++x )	
			//feature->data.fl[(int)((pori[x]+del)*map)%feature->cols] += pmag[x];
			feature->data.fl[(int)(pori[x]*map)] += pmag[x];

		pori = (float *)( (uchar *)pori + ori->step );
		pmag = (float *)( (uchar *)pmag + mag->step );
	}

	//cvConvertScale( &m, &m, 1.0 / ( frame->rows * frame->cols ), 0 );
	// smooth
	//{
	//	CvMat * tmp = cvCreateMat( 1, feature->cols*2, feature->type ), src, dst;
	//	cvCopy( feature, cvGetSubRect(     tmp, &dst, cvRect(0, 0, feature->cols, 1 ) ), 0 );
	//	cvCopy( feature, cvGetSubRect(     tmp, &dst, cvRect(feature->cols, 0, feature->cols, 1 ) ), 0 );
	//	cvSmooth( tmp, tmp, 2, 3, 0, 0, 0 );
	//	cvCopy( cvGetSubRect(     tmp, &dst, cvRect( feature->cols/2, 0, feature->cols/2, 1 ) ), 
	//			cvGetSubRect( feature, &src, cvRect(               0, 0, feature->cols/2, 1 ) ), 0 );
	//	cvCopy( cvGetSubRect(     tmp, &dst, cvRect(               0, 0, feature->cols/2, 1 ) ), 
	//			cvGetSubRect( feature, &src, cvRect( feature->cols/2, 0, feature->cols/2, 1 ) ), 0 );
	//	cvReleaseMat( &tmp );
	//}
	cvNormalize( feature, feature, 1.0, 0.0, CV_L2, 0 );

	return feature;
}
