#include <stdio.h>

#include "vplcore.h"
#include "vplKmeanVoc.h"




struct cKMeanVoc * kMeanVocCreate( void )
{
	struct cKMeanVoc * kmv = (struct cKMeanVoc *)malloc( sizeof( struct cKMeanVoc ) );
	kmv->dim = 0;
	kmv->count = 0;
	kmv->used = 0;
	kmv->center = NULL;
	kmv->weight  = NULL;

	return kmv;
}

struct cKMeanVoc * kMeanVocInit( struct cKMeanVoc * kmv, int dim, int count )
{
	if( !kmv ) kmv = kMeanVocCreate();

	if( kmv->center ) cvReleaseMat( &kmv->center );
	if( kmv->weight ) cvReleaseMat( &kmv->weight );
	kmv->center = NULL;
	kmv->weight = NULL;

	if( dim < 0 || count < 0 ) return kmv;

	kmv->dim = dim;
	kmv->count = count;
	kmv->used = 0;

	kmv->center = cvCreateMat( count, dim, CV_32FC1 );
	kmv->weight = cvCreateMat( count, 1, CV_32FC1 );
	cvZero( kmv->center );
	cvZero( kmv->weight );

	return kmv;
}

struct cKMeanVoc * kMeanVocLoad( const char * filename )
{
	CvFileStorage* fs = NULL;
	struct cKMeanVoc * kmv = NULL;

	if( !filename ) return NULL;

	fs = cvOpenFileStorage( filename, 0, CV_STORAGE_READ );
	if( !fs ) return NULL;

	kmv = kMeanVocCreate();
	kmv->dim   = cvReadIntByName( fs, NULL, "dim", 0 );
	kmv->count = cvReadIntByName( fs, NULL, "count", 0 );
	kmv->used  = cvReadIntByName( fs, NULL, "used", 0 );
	kmv->weight = (CvMat *)cvReadByName( fs, NULL, "weight", NULL );
	kmv->center = (CvMat *)cvReadByName( fs, NULL, "center", NULL );

	cvReleaseFileStorage( &fs );

	return kmv;
}

int kMeanVocSave( struct cKMeanVoc * kmv, const char * filename )
{
	CvFileStorage* fs = NULL;

	if( !kmv || !filename ) return -1;
	
	fs = cvOpenFileStorage( filename, 0, CV_STORAGE_WRITE );
	cvWriteInt( fs, "dim", kmv->dim );
	cvWriteInt( fs, "count", kmv->count );
	cvWriteInt( fs, "used", kmv->used );
	cvWrite( fs, "weight", kmv->weight, cvAttrList(0,0) );
	cvWrite( fs, "center", kmv->center, cvAttrList(0,0) );

	cvReleaseFileStorage( &fs );

	return 1;
}


void kMeanVocRelease( struct cKMeanVoc ** kmv )
{
	if( !kmv || !*kmv ) return;
	kMeanVocInit( *kmv, -1, -1 );
	free( *kmv );
	*kmv = NULL;
	return;
}




int kMeanVocFindClosest( struct cKMeanVoc * kmv, CvMat * ftr, double * dist, int norm_type )
{
	int i, imin = -1;
	double d, dmin = 100000000.;
	CvMat r;

	if( !kmv || !ftr || ftr->cols != kmv->dim ) return -1;

	for( i = 0; i < kmv->used; ++i )
	{
		d = cvNorm( cvGetRow( kmv->center, &r, i ), ftr, norm_type, 0 );
		if( d < dmin ) { dmin = d; imin = i; }
	}
	if( dist ) *dist = dmin;
	return imin;
}


CvMat * kMeanVocBagOfWords( struct cKMeanVoc * kmv, CvMat * ftrs, CvMat * wrds, int norm_type )
{
	int i;
	CvMat r;

	if( !kmv || !ftrs ) return wrds;

	if( !wrds ) cvCreateMat( 1, kmv->used, CV_32FC1 );
	cvZero( wrds );

	for( i = 0; i < ftrs->cols; ++i )
	{
		int idx = kMeanVocFindClosest( kmv, cvGetRow( ftrs, &r, i ), NULL, norm_type );
		if( idx < 0 ) continue;
		wrds->data.fl[ idx ] += kmv->weight->data.fl[ idx ];
	}

	return wrds;
}

CvMat * kMeanVocSetOfWords( struct cKMeanVoc * kmv, CvMat * ftrs, CvMat * wrds, int norm_type )
{
	int i;
	CvMat r;

	if( !kmv || !ftrs ) return wrds;

	if( !wrds ) cvCreateMat( 1, kmv->used, CV_32FC1 );
	cvZero( wrds );

	for( i = 0; i < ftrs->cols; ++i )
	{
		int idx = kMeanVocFindClosest( kmv, cvGetRow( ftrs, &r, i ), NULL, norm_type );
		if( idx < 0 ) continue;
		wrds->data.fl[ idx ] = 1;
	}

	return wrds;
}


CvMat * kMeanVocTranslateToWords( struct cKMeanVoc * kmv, CvMat * ftrs, CvMat * wrds, int norm_type )
{
	int i;
	CvMat r;

	if( !kmv || !ftrs ) return wrds;

	if( !wrds ) cvCreateMat( 1, ftrs->cols, CV_32FC1 );
	cvZero( wrds );

	for( i = 0; i < ftrs->cols; ++i )
		wrds->data.fl[ i ] = (float)kMeanVocFindClosest( kmv, cvGetRow( ftrs, &r, i ), NULL, norm_type );

	return wrds;
}



int kMeanVocEval( struct cKMeanVoc * kmv, CvMat * ftrs, int used, CvTermCriteria termcrit, CvMat ** map )
{
	int i;
	CvMat * labels = NULL;

	if( !kmv || !ftrs ) return -1;

	// clustering
	printf( "Cluster features\n" );
	labels = cvCreateMat( ftrs->rows, 1, CV_32SC1 );
	
	kmv->used = used;
	cvKMeans2( ftrs, kmv->used, labels, termcrit, 1, 0, 0, 0, 0 );

	// accumulate cluster centers
	printf( "Assign clusters\n" );
	for( i = 0; i < ftrs->rows; i++ )
	{
		CvMat cc, f;
		cvGetRow( kmv->center, &cc, labels->data.i[i] );
		cvAdd( cvGetRow( ftrs, &f, i ), &cc, &cc, 0 );
		kmv->weight->data.fl[ labels->data.i[i] ] += 1.0;
	}

	// weight cluster centers
	printf( "Re-weight clusters\n" );
	for( i = 0; i < kmv->used; i++ )
	{
		CvMat cc;
		cvGetRow( kmv->center, &cc, i );
		kmv->weight->data.fl[i] = (float)1.0/kmv->weight->data.fl[i];
		cvScale( &cc, &cc, kmv->weight->data.fl[i], 0 );
	}
	printf( "k-means done\n" );

	if( map ) *map = labels;
	else cvReleaseMat( &labels );

	return 1;
}

int kMeanVocUpdate( struct cKMeanVoc * kmv, CvMat * ftr, int i )
{
	CvMat cc;
	if( !kmv || !ftr ) return -1;
	if( i >= kmv->used ) return -2;
	if( ftr->rows > 1 || ftr->cols != kmv->dim ) return -3;

	cvGetRow( kmv->center, &cc, i );
	kmv->weight->data.fl[i] = (float)1.0/kmv->weight->data.fl[i];
	cvScale( &cc, &cc, kmv->weight->data.fl[i], 0 );

	cvAdd( ftr, &cc, &cc, 0 );
	kmv->weight->data.fl[i] += 1.0;

	kmv->weight->data.fl[i] = (float)1.0/kmv->weight->data.fl[i];
	cvScale( &cc, &cc, kmv->weight->data.fl[i], 0 );

	return 1;
}

int kMeanVocAdd( struct cKMeanVoc * kmv, CvMat * ftr )
{
	CvMat cc;
	if( !kmv || !ftr ) return -1;
	if( kmv->count == kmv->used ) return -2;
	if( ftr->rows > 1 || ftr->cols != kmv->dim ) return -3;
	
	cvGetRow( kmv->center, &cc, kmv->used );
	cvCopy( ftr, &cc, 0 );
	kmv->weight->data.fl[kmv->used] = 1.0;	
	
	return kmv->used++;
}









	//// ************************************************************************************* //
	//// load or create vocabulary
	//sprintf( buff, "%s.voc.xml", inputVideo );
	//VOC = kMeanVocLoad( buff );
	//if( !VOC )
	//{
	//	VOC = createVocabulary( GL, params.Points, 1000, 30, video );
	//	kMeanVocSave( VOC, buff );
	//}

	//// ************************************************************************************* //

	//detectSlides( video, GL, VOC, params.Points );









//
//struct cKMeanVoc * createVocabulary( struct cGradLocals * GL, CvPoint2D32f * Points, int sampleCount, int clusterCount, ffMedia * video )
//{
//	IplImage * imgColor = NULL, * imgPrezentation = NULL;
//	CvMat * gray = NULL;
//	IplImage * vFrame = NULL;
//	int frameCount = 0, frame = 0, ftr_length, ftr_count, i;
//	CvMat * ftrsBase = NULL, * ftrs = NULL, _ftrs;
//	struct cKMeanVoc * kmv = NULL;
//
//	CV_FUNCNAME( "createVocabulary" );
//	__BEGIN__
//
//	frameCount = ffLength( video );	
//	vFrame = ffCvFrameSeek( video, 0 );
//
//	imgPrezentation = cvCreateImage( cvSize(GL->cols,GL->rows), IPL_DEPTH_8U, 3); 
//	imgColor        = cvCreateImage( cvGetSize(vFrame), IPL_DEPTH_8U, 3 );
//	gray            = cvCreateMat( GL->rows, GL->cols, CV_8UC1 );
//
//	ftr_length = vplGllfGradDistGetSize( GL->gradWnd );
//	ftr_count  = glCount( GL );
//	ftrsBase = cvCreateMat( sampleCount*ftr_count, ftr_length, CV_32FC1 );
//	if( !ftrsBase ) return NULL;
//
//	printf( "\nExtract features from frames\n" );
//	i = 0; frame = 0;
//	while ( frame < frameCount && i < sampleCount*ftr_count+ftr_count )
//	{
//		if( i > 10 ) break;
//
//		if( !(i%1) ) printf( "\t Frame: % 8d/% 8d | % 8d\n", frame, frameCount, i );
//
//		vFrame = ffCvFramePreciseSeek( video, frame );
//		frame += frameCount/sampleCount;
//
//		if( !vFrame ) break;
//
//
//		cvCvtColor( vFrame, imgColor, CV_RGBA2RGB );
//		//GetPrezentation( imgColor, imgPrezentation, Points );
//		cvCvtColor( imgPrezentation, gray, CV_BGR2GRAY );
//		
// 		gradLocalsEval( GL, gray, cvGetRows( ftrsBase, &_ftrs, i*ftr_count, (i+1)*ftr_count, 1 ) );
//
//		//cvMatPrint( cvGetRows( ftrsBase, &_ftrs, i*ftr_count, (i+1)*ftr_count, 1 ), NULL );
//		//cvShowImage( "Video", imgPrezentation );	if( cvWaitKey( 0 ) == 27 ) break;
//
//		i++;
//	}
//
//	ftrs = cvGetRows( ftrsBase, &_ftrs, 0, i*ftr_count, 1 );
//
//	kmv = kMeanVocInit( NULL, ftr_length, clusterCount );
//
//	CvMat * map;
//	kMeanVocEval( kmv, ftrs, clusterCount, cvTermCriteria( CV_TERMCRIT_ITER, 5, 1.0 ), &map );
//
//	// render clusters
//	float minW = 100000.;
//	for( i = 0; i < kmv->weight->rows; ++i ) minW = MIN( minW, kmv->weight->data.fl[i] );
//	int maxSamples = cvRound(1.0/minW);
//	int k;
//	int pos[1000];
//	CvMat * canvas = cvCreateMat( GL->ws*maxSamples, GL->ws*clusterCount, CV_8UC1 );
//	cvZero( canvas );
//	memset( pos, 0, sizeof(int)*1000 );
//
//	frame = 0; i = 0;
//	while ( frame < frameCount && i < sampleCount*ftr_count+ftr_count )
//	{
//		if( i > 10 ) break;
//		if( !(i%1) ) printf( "\t Frame: % 8d/% 8d | % 8d\n", frame, frameCount, i );
//
//		vFrame = ffCvFramePreciseSeek( video, frame );
//		frame += frameCount/sampleCount;
//
//		if( !vFrame ) break;
//
//		cvCvtColor( vFrame, imgColor, CV_RGBA2RGB );
//		//GetPrezentation( imgColor, imgPrezentation, Points );
//		cvCvtColor( imgPrezentation, gray, CV_BGR2GRAY );
//		
// 		//gradLocalsEval( GL, gray, cvGetRows( ftrsBase, &_ftrs, i*ftr_count, (i+1)*ftr_count, 1 ) );
//		for( k = 0; k < ftr_count; ++k ) 
//		{
//			int idx = i*ftr_count+k;
//			int icl  = map->data.i[idx];
//			int x = k%GL->gw;
//			int y = k/GL->gw;
//			CvMat _gray, _c;
//
//			cvCopy( 
//				cvGetSubRect( gray, &_gray, cvRect( cvRound((x+0.5)*GL->dw), cvRound((y+0.5)*GL->dh), GL->ws, GL->ws ) ),
//				cvGetSubRect( canvas, &_c,  cvRect( icl*GL->ws, pos[icl]*GL->ws, GL->ws, GL->ws ) ),
//				NULL );
//			cvRectangle( &_c, cvPoint(0,0), cvPoint(_c.cols, _c.rows), CV_RGB(255,255,255), 1, 8,0 ); 
//			pos[icl]++;
//		}
//
//		cvShowImage( "Video", canvas );	if( cvWaitKey( 0 ) == 27 ) break;
//
//		i++;
//	}
//	cvReleaseMat( &canvas );
//
//	cvReleaseMat( &ftrsBase );
//
//	__END__
//
//	if( imgColor ) cvReleaseImage( &imgColor );
//	if( imgPrezentation ) cvReleaseImage( &imgPrezentation );
//	if( gray ) cvReleaseMat( &gray ); 
//
//	return kmv;
//}



//int detectSlides2( ffMedia * video, struct cGradLocals * GL, struct cKMeanVoc * VOC, CvPoint2D32f * Points )
//{
//	IplImage * imgColor = NULL, * imgPrezentation = NULL;
//	CvMat * gray = NULL;
//	IplImage * vFrame = NULL;
//	int frameCount = 0, frame = 0, ftr_length, ftr_count;
//	CvMat * imgWrds = NULL, * imgFtrs = NULL;
//	//CvMat * ftrBuff = NULL, ** imgBuff = NULL, ftrBuffR;
//	//int buffI;
//
//	struct cKMeanVoc * slides = NULL;
//
//	CV_FUNCNAME( "createVocabulary" );
//	__BEGIN__
//
//	frameCount = ffLength( video );	
//	vFrame = ffCvFrameSeek( video, 0 );
//
//	imgPrezentation = cvCreateImage( cvSize(GL->cols,GL->rows), IPL_DEPTH_8U, 3); 
//	imgColor        = cvCreateImage( cvGetSize(vFrame), IPL_DEPTH_8U, 3 );
//	gray            = cvCreateMat( GL->rows, GL->cols, CV_8UC1 );
//
//	ftr_length = vplGllfGradDistGetSize( GL->gradWnd );
//	ftr_count  = glCount( GL );
//
//	imgFtrs = cvCreateMat( ftr_count, ftr_length, CV_32FC1 );
//	//imgWrds = cvCreateMat( 1, ftr_count, CV_32FC1 );
//	imgWrds = cvCreateMat( 1, VOC->used, CV_32FC1 );
//
//	slides = kMeanVocInit( NULL, VOC->used/*ftr_count*/, 1000 );
//
//	printf( "\nStart process frames\n" );
//	frame = 0;
//	while ( frame < frameCount )
//	{
//		//if( !(i%100) ) printf( "\t Frame: % 8d/% 8d \n", frame, frameCount );
//
//		vFrame = ffCvFramePreciseSeek( video, frame );
//		frame += 25;
//		if( !vFrame ) break;
//
//		cvCvtColor( vFrame, imgColor, CV_RGBA2RGB );
//		//GetPrezentation( imgColor, imgPrezentation, Points );
//		cvCvtColor( imgPrezentation, gray, CV_BGR2GRAY );
//		
// 		gradLocalsEval( GL, gray, imgFtrs );
//
//		kMeanVocSetOfWords( VOC, imgFtrs, imgWrds, CV_L2 );
//		//kMeanVocTranslateToWords( VOC, imgFtrs, imgWrds );
//
//		// find closest imgWrds in slides
//		double dist;
//		int iBest = kMeanVocFindClosest( slides, imgWrds, &dist, CV_L1 );
//
//		printf( "\t Frame: % 8d/% 8d Dist:%f ", frame, frameCount, dist );
//
//		// update closest 
//		if( iBest >= 0 && dist < 50 )
//		{
//			kMeanVocUpdate( slides, imgWrds, iBest );
//			printf( "Update:%d\n", iBest );
//		}
//		else
//		// or add slide -> HIT
//		{
//			int newi = kMeanVocAdd( slides, imgWrds );
//			printf( "Add:%d\n", newi );
//		}
//
//		cvShowImage( "Video", imgPrezentation );
//		if( cvWaitKey( 0 ) == 27 ) break;
//
//		if( slides->used >= slides->count ) break;
//	}
//
//
//	__END__
//
//	if( imgWrds ) cvReleaseMat( &imgWrds ); 
//	if( imgFtrs ) cvReleaseMat( &imgFtrs ); 
//
//	if( imgColor ) cvReleaseImage( &imgColor );
//	if( imgPrezentation ) cvReleaseImage( &imgPrezentation );
//	if( gray ) cvReleaseMat( &gray ); 
//
//	if( slides ) kMeanVocRelease( &slides );
//
//	return 1;
//}

