/*
								+---------------------------------+
								|                                 |
								|  ***   AMI Transform lib   ***  |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|           Transform.h           |
								|                                 |
								+---------------------------------+
*/

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <gl/gl.h>
#include <vector>
#include "callstack.h"

#include "glext.h"
#include "wglext.h"

#include "opengl20.h"
#include "glstate.h"

#include "shader.h"

#include "renderpath.h"

#include "vertexbufferobject.h"

#include "transform.h"
#include "transformdata.h"


#if defined(_MSC_VER) && !defined(__MWERKS__) // msvc
#ifndef for
#define for if(0) {} else for
#endif
// msvc error with variable scoping in for loops
// (for(int i = 0; i < 10; i ++) { } if(i == 10) { } was allright in msvc)
#endif

COmniDirectionalTransform::COmniDirectionalTransform()
	:m_n_chosen_path(-1),
	m_n_chosen_lang(sh_lang_HighLevel),
	m_n_chosen_shader(-1),
	m_p_shader_param(0),
	m_n_shader_param_num(0),
	m_p_shader(0),
	m_vertex_buffer(GL_ARRAY_BUFFER_ARB),
	m_n_tesselation(-1)
{
	__FuncGuard("COmniDirectionalTransform::COmniDirectionalTransform");
}

COmniDirectionalTransform::~COmniDirectionalTransform()
{
	__FuncGuard("COmniDirectionalTransform::~COmniDirectionalTransform");

	FreeShader();
	for(size_t i = 0; i < m_path_list.size(); i ++) {
		m_path_list[i]->Free(); // TPath intentionaly doesn't have default destructor
		delete m_path_list[i];
	}
	for(size_t i = 0; i < m_shader_list.size(); i ++)
		delete m_shader_list[i];
}

// return true if card does support low level sl
int COmniDirectionalTransform::b_Have_LowLevel_ShadingLanguage() const
{
	__FuncGuard("COmniDirectionalTransform::b_Have_LowLevel_ShadingLanguage");

	return CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_program");
}

// return true if card does support high level sl
int COmniDirectionalTransform::b_Have_HighLevel_ShadingLanguage() const
{
	__FuncGuard("COmniDirectionalTransform::b_Have_HighLevel_ShadingLanguage");

	return CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_shader") &&
		CGLExtensionHandler::b_SupportedExtension("GL_ARB_shader_objects") &&
		CGLExtensionHandler::b_SupportedExtension("GL_ARB_shading_language_100");
}

int COmniDirectionalTransform::LoadPaths(int (*load_rp_funtion)(std::vector<TPath*>&))
{
	__FuncGuard("COmniDirectionalTransform::LoadPaths");

	if(m_path_list.size()) {
		for(size_t i = 0; i < m_path_list.size(); i ++) {
			m_path_list[i]->Free(); // TPath intentionaly doesn't have default destructor
			delete m_path_list[i];
		}
		m_path_list.clear();
	}
	// delete paths if there were any

	m_n_chosen_path = -1;
	m_n_chosen_shader = -1;

	return load_rp_funtion(m_path_list); // included from transformdata.h
}

int COmniDirectionalTransform::n_CompatibleRenderPath_Num() const
{
	__FuncGuard("COmniDirectionalTransform::n_CompatibleRenderPath_Num");

	int n_count = 0;
	for(size_t i = 0; i < m_path_list.size(); i ++) {
		if(m_path_list[i]->b_ValidPath(0) || m_path_list[i]->b_ValidPath(1))
			n_count ++;
	}
	// count paths that would validate

	return n_count;
}

const TPath *COmniDirectionalTransform::p_CompatibleRenderPath(int n_index) const
{
	__FuncGuard("COmniDirectionalTransform::p_CompatibleRenderPath");

	int n_count = 0;
	for(int n_pass = 0; n_pass < 2; n_pass ++) {
		for(size_t i = 0; i < m_path_list.size(); i ++) {
			if(n_pass && m_path_list[i]->b_ValidPath(0))
				continue;
			// this path was already valid in first pass

			if(m_path_list[i]->b_ValidPath(n_pass) && n_count ++ == n_index)
				return m_path_list[i];
		}
	}
	// return pointer to n_index-th valid path, sorted by passes

	return 0;
}

int COmniDirectionalTransform::ChooseRenderPath(int n_index)
{
	__FuncGuard("COmniDirectionalTransform::ChooseRenderPath");

	if(n_index < 0 || n_index >= n_CompatibleRenderPath_Num())
		return false;

	m_n_chosen_path = n_index;

	return true;
}

int COmniDirectionalTransform::LoadShaders(int (*load_sh_function)(std::vector<CShaderInfo*>&, const char*)) // can be called after render path was chosen
{
	__FuncGuard("COmniDirectionalTransform::LoadShaders");

	if(m_n_chosen_path == -1)
		return false;

	if(m_shader_list.size()) {
		for(size_t i = 0; i < m_shader_list.size(); i ++)
			delete m_shader_list[i];
		m_shader_list.clear();
	}
	// free shaders if there were any

	return load_sh_function(m_shader_list, m_path_list[m_n_chosen_path]->p_s_path_name);
	// included from shaderdata.h
}

int COmniDirectionalTransform::UseShadingLanguage(int b_high_level)
{
	__FuncGuard("COmniDirectionalTransform::UseShadingLanguage");

	m_n_chosen_lang = (b_high_level)? sh_lang_HighLevel : sh_lang_LowLevel;

	return true;
}

int COmniDirectionalTransform::n_Shader_Num()
{
	__FuncGuard("COmniDirectionalTransform::n_Shader_Num");

	int n_count = 0;
	for(size_t i = 0; i < m_shader_list.size(); i ++) {
		m_shader_list[i]->PreferLanguage(m_n_chosen_lang); // say "we want this lang"
		if(m_shader_list[i]->n_Language() == m_n_chosen_lang) // do we get it?
			n_count ++;
	}
	// count shaders with support for selected language

	return n_count;
}

const char *COmniDirectionalTransform::p_s_ShaderName(int n_index)
{
	__FuncGuard("COmniDirectionalTransform::p_s_ShaderName");

	int n_count = 0;
	for(size_t i = 0; i < m_shader_list.size(); i ++) {
		m_shader_list[i]->PreferLanguage(m_n_chosen_lang); // say "we want this lang"
		if(m_shader_list[i]->n_Language() == m_n_chosen_lang && n_count ++ == n_index) // do we get it?
			return m_shader_list[i]->p_s_Name();
	}
	// count shaders with support for selected language

	return 0;
}

int COmniDirectionalTransform::ChooseShader(int n_index)
{
	__FuncGuard("COmniDirectionalTransform::ChooseShader");

	if(n_index < 0 || n_index >= n_Shader_Num())
		return false;

	m_n_chosen_shader = n_index;

	return true;
}

int COmniDirectionalTransform::n_ChosenShader()
{
	__FuncGuard("COmniDirectionalTransform::n_ChosenShader");

	return m_n_chosen_shader;
}

int COmniDirectionalTransform::CompileShader(char *&r_p_error_message)
{
	__FuncGuard("COmniDirectionalTransform::CompileShader");

	r_p_error_message = 0;

	if(m_n_chosen_shader < 0 || m_n_chosen_shader >= n_Shader_Num())
		return false;

	FreeShader();

	if(!(m_p_shader = new CGLProgramObject))
		return false;
	while(1) {
		CShaderInfo *p_fragment = 0;
		CShaderInfo *p_vertex = 0;

		if(m_shader_list[m_n_chosen_shader]->n_ShaderType() == sh_type_Vertex)
			p_vertex = m_shader_list[m_n_chosen_shader];
		else
			p_fragment = m_shader_list[m_n_chosen_shader];
		// set one of the shaders ...

		for(size_t i = 0; i < m_shader_list.size(); i ++) {
			if(!p_fragment && !_stricmp(m_shader_list[i]->p_s_Name(), p_vertex->p_s_Name()) &&
			   m_shader_list[i]->n_ShaderType() == sh_type_Fragment) {
				p_fragment = m_shader_list[i];
				break;
			}
			if(!p_vertex && !_stricmp(m_shader_list[i]->p_s_Name(), p_fragment->p_s_Name()) &&
			   m_shader_list[i]->n_ShaderType() == sh_type_Vertex) {
				p_vertex = m_shader_list[i];
				break;
			}
		}
		// see if there's shader with the same name into the couple (vertex + fragment)

		if(!m_p_shader->BindShaders(p_vertex, p_fragment))
			break;
		// attach shader source

		m_p_shader->PreferLang(m_n_chosen_lang);
		// prefer language we chose

		if(!m_p_shader->CompileGLShaders(r_p_error_message))
			break;
		if(r_p_error_message) {
			delete[] r_p_error_message;
			r_p_error_message = 0;
		}
		// try to compile shaders

		if(!m_p_shader->LinkGLShaders(r_p_error_message))
			break;
		if(r_p_error_message) {
			delete[] r_p_error_message;
			r_p_error_message = 0;
		}
		// try to link shaders

		if(!m_p_shader->ResolveUniforms())
			break;
		// high level shading language feature - texture samplers can be mapped to
		// user-chosen texture units, so we're going to find adresses of registers
		// on graphic card where we'll have to load numbers of texture units later
		// when binding shader

		int n_transform_type = -1, n_ipp_type = -1, n_off;
		const char *p_transform_name[] = {"imgproc_", "simple_", "geom_", "polynet_", "persp_", "polypersp_"};
		const int n_transform_register_num[] = {0, 2, 4, 4, 4, 4};
		const char *p_ipp_name[] = {"noipp_", "skin_", "gray_", "gauss_",
			"radialsobel_", "localmax_", "frontline_"};
		const int n_ipp_register_num[] = {0, 2, 0, 1, 2, 2, 2};
		for(int i = 0; i < sizeof(p_transform_name) / sizeof(p_transform_name[0]); i ++) {
			if(!strncmp(p_s_ShaderName(m_n_chosen_shader), p_transform_name[i],
			   strlen(p_transform_name[i]))) {
				n_transform_type = i;
				n_off = strlen(p_transform_name[i]);
				break;
			}
		}
		for(int i = 0; i < sizeof(p_ipp_name) / sizeof(p_ipp_name[0]); i ++) {
			if(!strncmp(p_s_ShaderName(m_n_chosen_shader) + n_off, p_ipp_name[i],
			   strlen(p_ipp_name[i]))) {
				n_ipp_type = i;
				break;
			}
		}
		// see which transformation and post-processing was chosen

		if(n_transform_type == -1 || n_ipp_type == -1)
			break;

		if(m_p_shader_param)
			delete[] m_p_shader_param;
		m_n_shader_param_num = n_transform_register_num[n_transform_type] +
			n_ipp_register_num[n_ipp_type];
		if(!(m_p_shader_param = new TGLShaderParam[m_n_shader_param_num]))
			break;
		// alloc array of shader parameters (two only for simple shaders)

		struct {
			char *p_s_glsl_name;
			char p_s_arb_name[256];
			int n_size;
			int n_type;
		} p_transform_param_table[][5] = {
			{
				{0}
			},

			{
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4}
			}, // simple

			{
				/*{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4},
				{"mirror_geom", "local%cmirror_geom%c%d", 4, sh_param_float4},
				{"mirror_geom2", "local%cmirror_geom2%c%d", 4, sh_param_float4},
				{"geom_offset", "local%cmirror_geom%c%d", 2, sh_param_float2}*/
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4},
				{"mirror_geom", "local%cmirror_geom%c%d", 4, sh_param_float4},
				{"geom_offset", "local%cmirror_geom%c%d", 4, sh_param_float4}
			}, // geom

			{
				/*{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4},
				{"mirror_geom", "local%cmirror_geom%c%d", 4, sh_param_float4},
				{"mirror_geom2", "local%cmirror_geom2%c%d", 4, sh_param_float4},
				{"geom_offset", "local%cmirror_geom%c%d", 2, sh_param_float2}*/
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4},
				{"mirror_geom", "local%cmirror_geom%c%d", 4, sh_param_float4},
				{"geom_offset", "local%cmirror_geom%c%d", 4, sh_param_float4}
			}, // polynet

			{
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4},
				{"mirror_geom", "local%cmirror_geom%c%d", 4, sh_param_float4},
				{"geom_offset", "local%cmirror_geom%c%d", 4, sh_param_float4}
			}, // perspective correct geom
			{
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4},
				{"mirror_geom", "local%cmirror_geom%c%d", 4, sh_param_float4},
				{"geom_offset", "local%cmirror_geom%c%d", 4, sh_param_float4}
			} // perspective correct polynet geom
		}, p_ipp_param_table[][2] = {
			{
				{0},
			}, // noipp

			{
				{"covariant", "local%ccovariant%c%d", 4, sh_param_float4},
				{"avg_rg", "local%cavg_rg%c%d", 2, sh_param_float2}
			}, // skin

			{
				{0}
			}, // gray
				
			{
				{"gauss_dir", "local%cgauss_dir%c%d", 2, sh_param_float2}
			}, // gauss

			{
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4}
			}, // radialsobel

			{
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4}
			}, // localmax

			{
				{"mirror_area", "local%cmirror_area%c%d", 4, sh_param_float4},
				{"offset_ratio", "local%coffset_ratio%c%d", 4, sh_param_float4}
			} // frontline
		}, *p_table_ptr;
		// table with shader param description

		for(int j = 0, n_used_regs = 0; j < 2; j ++) {
			int n_reg_num;
			if(!j) {
				p_table_ptr = p_transform_param_table[n_transform_type];
				n_reg_num = n_transform_register_num[n_transform_type];
			} else {
				p_table_ptr = p_ipp_param_table[n_ipp_type];
				n_reg_num = n_ipp_register_num[n_ipp_type];
			}

			for(int i = 0; i < n_reg_num; i ++, n_used_regs ++) {
				if(m_p_shader->n_PreferredLang() == sh_lang_LowLevel) {
					char p_s_buffer[256];
					sprintf(p_s_buffer, p_table_ptr[i].p_s_arb_name, 0, 0, n_used_regs);
					strcpy(p_table_ptr[i].p_s_arb_name, p_s_buffer);

					m_p_shader_param[n_used_regs].p_s_name = p_table_ptr[i].p_s_arb_name;
				} else
					m_p_shader_param[n_used_regs].p_s_name = p_table_ptr[i].p_s_glsl_name;
				// name for ARB_*_program must contain index of register, parameter resides in

				m_p_shader_param[n_used_regs].b_skip = false;
				m_p_shader_param[n_used_regs].p_previous = 0; // used for concatenation of multiple params
				//
				m_p_shader_param[n_used_regs].n_size = p_table_ptr[i].n_size;
				m_p_shader_param[n_used_regs].n_type = p_table_ptr[i].n_type;
				if(!(m_p_shader_param[n_used_regs].f_value = new float[p_table_ptr[i].n_size]))
					break;
			}
		}
		// fill params array

		/* // older version of the same parameter mess
			m_p_shader_param[0].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"mirror_area" : "local\0mirror_area\00";
			// again - lowlevel vs. highlevel shader difference - high level shader params
			// don't distinguish scope ("local") and they're placed in registers by compiller.
			// on the other side, low level shading language needs to know this is going to
			// be local or enviromental parameter (there are two different register sets on
			// gfx card) and they need to know which register parameter is in (it's specified
			// by source code and there's no mechanism to retrieve the address so we "must know
			// where we put it")
			//
			m_p_shader_param[0].b_skip = false;
			m_p_shader_param[0].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[0].n_size = 4;
			m_p_shader_param[0].n_type = sh_param_float4;
			if(!(m_p_shader_param[0].f_value = new float[4]))
				break;
			// param named "mirror_area", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 0

			m_p_shader_param[1].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"offset_ratio" : "local\0offset_ratio\01";
			// again - lowlevel vs. highlevel shader difference - high level shader params
			// don't distinguish scope ("local") and they're placed in register by compiller.
			// on the other side, low level shading language needs to know this is going to
			// be local or enviromental parameter (there are two different register sets on
			// gfx card) and they need to know which register parameter is in (it's specified
			// by source code and there's no mechanism to retrieve the address so we "must know
			// where we put it")
			//
			m_p_shader_param[1].b_skip = false;
			m_p_shader_param[1].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[1].n_size = 4;
			m_p_shader_param[1].n_type = sh_param_float4;
			if(!(m_p_shader_param[1].f_value = new float[4]))
				break;
			// param named "offset_ratio", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 1

			m_p_shader_param[2].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"covariant" : "local\0covariant\02";
			//
			m_p_shader_param[2].b_skip = false;
			m_p_shader_param[2].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[2].n_size = 4;
			m_p_shader_param[2].n_type = sh_param_float4;
			if(!(m_p_shader_param[2].f_value = new float[4]))
				break;
			// param named "covariant", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 2

			m_p_shader_param[3].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"avg_rg" : "local\0avg_rg\03";
			//
			m_p_shader_param[3].b_skip = false;
			m_p_shader_param[3].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[3].n_size = 2;
			m_p_shader_param[3].n_type = sh_param_float2;
			if(!(m_p_shader_param[3].f_value = new float[2]))
				break;
			// param named "avg_rg", holding vector of 2 floats
			// for low level sl it's local parameter, stored in register 3

			// manually create array of shader parameter structures
			// (more typical would be to store it in some file along with shader source)
		} else if(!strncmp(p_s_ShaderName(m_n_chosen_shader), "geom", strlen("geom"))) {
			if(m_p_shader_param)
				delete[] m_p_shader_param;
			m_n_shader_param_num = 7;
			if(!(m_p_shader_param = new TGLShaderParam[m_n_shader_param_num]))
				break;
			// alloc array of shader parameters (two only for simple shaders)

			m_p_shader_param[0].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"mirror_area" : "local\0mirror_area\00";
			//
			m_p_shader_param[0].b_skip = false;
			m_p_shader_param[0].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[0].n_size = 4;
			m_p_shader_param[0].n_type = sh_param_float4;
			if(!(m_p_shader_param[0].f_value = new float[4]))
				break;
			// param named "mirror_area", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 0

			m_p_shader_param[1].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"offset_ratio" : "local\0offset_ratio\01";
			//
			m_p_shader_param[1].b_skip = false;
			m_p_shader_param[1].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[1].n_size = 4;
			m_p_shader_param[1].n_type = sh_param_float4;
			if(!(m_p_shader_param[1].f_value = new float[4]))
				break;
			// param named "offset_ratio", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 1

			m_p_shader_param[2].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"mirror_geom" : "local\0mirror_geom\02";
			//
			m_p_shader_param[2].b_skip = false;
			m_p_shader_param[2].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[2].n_size = 4;
			m_p_shader_param[2].n_type = sh_param_float4;
			if(!(m_p_shader_param[2].f_value = new float[4]))
				break;
			// param named "mirror_geom", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 2

			m_p_shader_param[3].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"mirror_geom2" : "local\0mirror_geom2\03";
			//
			m_p_shader_param[3].b_skip = false;
			m_p_shader_param[3].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[3].n_size = 4;
			m_p_shader_param[3].n_type = sh_param_float4;
			if(!(m_p_shader_param[3].f_value = new float[4]))
				break;
			// param named "mirror_geom2", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 3

			m_p_shader_param[4].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"geom_offset" : "local\0geom_offset\04";
			//
			m_p_shader_param[4].b_skip = false;
			m_p_shader_param[4].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[4].n_size = 2;
			m_p_shader_param[4].n_type = sh_param_float2;
			if(!(m_p_shader_param[4].f_value = new float[2]))
				break;
			// param named "mirror_geom", holding vector of 2 floats
			// for low level sl it's local parameter, stored in register 4

			m_p_shader_param[5].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"covariant" : "local\0covariant\05";
			//
			m_p_shader_param[5].b_skip = false;
			m_p_shader_param[5].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[5].n_size = 4;
			m_p_shader_param[5].n_type = sh_param_float4;
			if(!(m_p_shader_param[5].f_value = new float[4]))
				break;
			// param named "covariant", holding vector of 4 floats
			// for low level sl it's local parameter, stored in register 5

			m_p_shader_param[6].p_s_name =
				(m_p_shader->n_PreferredLang() == sh_lang_HighLevel)?
				"avg_rg" : "local\0avg_rg\06";
			//
			m_p_shader_param[6].b_skip = false;
			m_p_shader_param[6].p_previous = 0; // used for concatenation of multiple params
			//
			m_p_shader_param[6].n_size = 2;
			m_p_shader_param[6].n_type = sh_param_float2;
			if(!(m_p_shader_param[6].f_value = new float[2]))
				break;
			// param named "avg_rg", holding vector of 2 floats
			// for low level sl it's local parameter, stored in register 5

			// manually create array of shader parameter structures
			// (more typical would be to store it in some file along with shader source)
		} else
			break; // bad shader name
		*/

		if(!m_p_shader->FindParams(m_p_shader_param, m_n_shader_param_num))
			break;
		// find adresses of registers on graphic card where we're going to store
		// shader parameters

		if(!m_p_shader->b_Ready())
			break;

		return true;
	}

	delete m_p_shader;
	m_p_shader = 0;
	if(m_p_shader_param) {
		for(int i = 0; i < m_n_shader_param_num; i ++)
			delete[] m_p_shader_param[i].f_value;
		delete[] m_p_shader_param;
	}
	m_p_shader_param = 0;
	m_n_shader_param_num = 0;

	return false;
}

int COmniDirectionalTransform::FreeShader()
{
	__FuncGuard("COmniDirectionalTransform::FreeShader");

	if(m_n_chosen_shader == -1)
		return false;

	if(m_p_shader)
		delete m_p_shader;
	m_p_shader = 0;
	if(m_p_shader_param) {
		for(int i = 0; i < m_n_shader_param_num; i ++)
			delete[] m_p_shader_param[i].f_value;
		delete[] m_p_shader_param;
	}
	m_p_shader_param = 0;
	m_n_shader_param_num = 0;

	return true;
}

void COmniDirectionalTransform::FreeShadersPaths()
{
	__FuncGuard("COmniDirectionalTransform::FreeShadersPaths");

	for(size_t i = 0; i < m_path_list.size(); i ++) {
		m_path_list[i]->Free(); // TPath intentionaly doesn't have default destructor
		delete m_path_list[i];
	}
	m_path_list.clear();
	for(size_t i = 0; i < m_shader_list.size(); i ++)
		delete m_shader_list[i];
	m_shader_list.clear();
}

float *COmniDirectionalTransform::p_ParamVector(const char *p_s_param_name)
{
	__FuncGuard("COmniDirectionalTransform::p_ParamVector");

	_ASSERTE(m_p_shader);
	_ASSERTE(m_p_shader_param);

	for(int i = 0; i < m_n_shader_param_num; i ++) {
		if((m_p_shader->n_PreferredLang() == sh_lang_HighLevel &&
		   !_stricmp(m_p_shader_param[i].p_s_name, p_s_param_name)) ||
		   (m_p_shader->n_PreferredLang() == sh_lang_LowLevel &&
		   !_stricmp(m_p_shader_param[i].p_s_name +
		   strlen(m_p_shader_param[i].p_s_name) + 1, p_s_param_name)))
			return m_p_shader_param[i].f_value;
	}
	return 0;
}

static float f;

float &COmniDirectionalTransform::f_MirrorCenter_x()
{
	__FuncGuard("COmniDirectionalTransform::f_MirrorCenter_x");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("mirror_area")[2];
}

float &COmniDirectionalTransform::f_MirrorCenter_y()
{
	__FuncGuard("COmniDirectionalTransform::f_MirrorCenter_y");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("mirror_area")[3];
}

float &COmniDirectionalTransform::f_SideRatio()
{
	__FuncGuard("COmniDirectionalTransform::f_SideRatio");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("offset_ratio")[1];
}

float &COmniDirectionalTransform::f_OuterRadius()
{
	__FuncGuard("COmniDirectionalTransform::f_OuterRadius");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("mirror_area")[1];
}

float &COmniDirectionalTransform::f_InnerRadius()
{
	__FuncGuard("COmniDirectionalTransform::f_InnerRadius");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("mirror_area")[0];
}

float &COmniDirectionalTransform::f_AngularOffset()
{
	__FuncGuard("COmniDirectionalTransform::f_AngularOffset");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("offset_ratio")[0];
}

float &COmniDirectionalTransform::f_Pixel_Width()
{
	__FuncGuard("COmniDirectionalTransform::f_Pixel_Width");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("offset_ratio")[2];
}

float &COmniDirectionalTransform::f_Pixel_Height()
{
	__FuncGuard("COmniDirectionalTransform::f_Pixel_Height");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("offset_ratio")[3];
}

float &COmniDirectionalTransform::f_MirrorGeom(int n_component)
{
	__FuncGuard("COmniDirectionalTransform::f_Pixel_Height");

	if(m_n_chosen_shader == -1 || !m_p_shader || n_component < 0 || n_component > 3)
		return f;

	return p_ParamVector("mirror_geom")[n_component];
}

float &COmniDirectionalTransform::f_MirrorGeom2(int n_component)
{
	__FuncGuard("COmniDirectionalTransform::f_Pixel_Height");

	if(m_n_chosen_shader == -1 || !m_p_shader || n_component < 0 || n_component > 3)
		return f;

	return p_ParamVector("mirror_geom2")[n_component];
}

float &COmniDirectionalTransform::f_Bottom_Y()
{
	__FuncGuard("COmniDirectionalTransform::f_Pixel_Height");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("geom_offset")[0];
}

float &COmniDirectionalTransform::f_Cyl_Height() // contains top_y - bottom_y
{
	__FuncGuard("COmniDirectionalTransform::f_Pixel_Height");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("geom_offset")[1];
}

float &COmniDirectionalTransform::f_Fov()
{
	__FuncGuard("COmniDirectionalTransform::f_Fov");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("geom_offset")[2];
}

float &COmniDirectionalTransform::f_MinusEccentricity()
{
	__FuncGuard("COmniDirectionalTransform::f_MinusEccentricity");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("geom_offset")[3];
}

float &COmniDirectionalTransform::f_Covariant(int n_i, int n_j)
{ // get k(ij), i, j = <0, 1>
	__FuncGuard("COmniDirectionalTransform::f_Covariant");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("covariant")[n_i + 2 * n_j];
}

float &COmniDirectionalTransform::f_AvgComponents(int n_index)
{
	__FuncGuard("COmniDirectionalTransform::f_AvgComponents");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("avg_rg")[n_index];
}

float &COmniDirectionalTransform::f_GaussDir(int n_index)
{
	__FuncGuard("COmniDirectionalTransform::f_GaussDir");

	if(m_n_chosen_shader == -1 || !m_p_shader)
		return f;

	return p_ParamVector("gauss_dir")[n_index];
}

void COmniDirectionalTransform::DrawUniformFullscreenQuad(unsigned int n_source_texture, CGLState *p_state)
{
	__FuncGuard("COmniDirectionalTransform::DrawFullscreenQuad");

	_ASSERTE(m_p_shader);

	if(glGetError() != GL_NO_ERROR)
		return;

	p_state->ActiveTextureUnit(0);
	p_state->EnableTexture2D();
	p_state->BindTexture2D(n_source_texture);
	// bind texture with video

	if(glGetError() != GL_NO_ERROR)
		return;

	m_p_shader->BindGL();

	if(glGetError() != GL_NO_ERROR)
		return;

	m_p_shader->SetGLShaderParams(m_p_shader_param, m_n_shader_param_num);
	// bind shader to set it active

	if(glGetError() != GL_NO_ERROR)
		return;

	glBegin(GL_QUADS);
	glTexCoord2f(0, 0);
	glVertex2f(-1, 1);
	glTexCoord2f(1, 0);
	glVertex2f(1, 1);
	glTexCoord2f(1, 1);
	glVertex2f(1, -1);
	glTexCoord2f(0, 1);
	glVertex2f(-1, -1);
	glEnd();
	// draw fullscreen quad, shader is called for every
	// pixel drawn and transformation is applied

	if(glGetError() != GL_NO_ERROR)
		return;

	CGLShaderUnBinder::_UnBind();
	// unbind shader

	if(glGetError() != GL_NO_ERROR)
		return;
}
// draw fullscreen quad with shader bound, causing transform to compute

void COmniDirectionalTransform::DrawFullscreenQuad(unsigned int n_source_texture, CGLState *p_state)
{
	__FuncGuard("COmniDirectionalTransform::DrawFullscreenQuad");

	_ASSERTE(m_p_shader);

	p_state->ActiveTextureUnit(0);
	p_state->EnableTexture2D();
	p_state->BindTexture2D(n_source_texture);
	// bind texture with video

	m_p_shader->BindGL();

	m_p_shader->SetGLShaderParams(m_p_shader_param, m_n_shader_param_num);
	// bind shader to set it active

	glMatrixMode(GL_TEXTURE);
	glPushMatrix();
	glScalef(1, f_OuterRadius() - f_InnerRadius(), 1);

	glBegin(GL_QUADS);
	glTexCoord2f(0, 0);
	glVertex2f(-1, 1);
	glTexCoord2f(1, 0);
	glVertex2f(1, 1);
	glTexCoord2f(1, 1);
	glVertex2f(1, -1);
	glTexCoord2f(0, 1);
	glVertex2f(-1, -1);
	glEnd();
	// draw fullscreen quad, shader is called for every
	// pixel drawn and transformation is applied

	glPopMatrix();

	CGLShaderUnBinder::_UnBind();
	// unbind shader
}
// draw fullscreen quad with shader bound, causing transform to compute

int COmniDirectionalTransform::DrawFullscreenQuad(unsigned int n_source_texture,
												  int n_tesselation, CGLState *p_state)
{
	__FuncGuard("COmniDirectionalTransform::DrawFullscreenQuad");

	_ASSERTE(n_tesselation > 0);
	_ASSERTE(m_p_shader);

	if(n_tesselation != m_n_tesselation) {
		/*

		  1---3---5
		  |   |   |
		 0,11-2,9-4,7
		  |   |   |
		  10--8---6

		on 1 line there's 2 * (tess + 1) vertices
		so number of vertices is 2 * (tess + 1) * tess

		*/
		m_vertex_buffer.Bind();
		int n_vertex_num = 2 * (n_tesselation + 1) * n_tesselation;
		if(!m_vertex_buffer.Alloc(n_vertex_num * (2 + 2) * sizeof(float)))
			return false;
		// alloc buffer for vertex array, 2 floats per (2D) vertex and the other 2 per texcoord

		void *p_data = m_vertex_buffer.p_Map(GL_WRITE_ONLY_ARB);
		if(p_data == NULL)
			return false;

		for(int i = 0; i < n_vertex_num; i ++) {
			int n_y = i / (2 * (n_tesselation + 1)) + !(i & 1);
			int n_x = (i / 2) % (n_tesselation + 1);

			float f_s = (float)n_x / (float)n_tesselation;
			float f_t = (float)n_y / (float)n_tesselation;
			// calc s, t ... <0, 1>

			float f_x = f_s * 2.0f - 1.0f;
			float f_y = (1.0f - f_t) * 2.0f - 1.0f;
			// calc x, y ... <-1, 1>

			if((i / (2 * (n_tesselation + 1))) & 1) {
				f_x *= -1;
				f_s = 1.0f - f_s;
			}
			// flip direction on odd lines

			*(((float*)p_data) + i * 2) = f_x;
			*(((float*)p_data) + i * 2 + 1) = f_y;
			// store vertex coords

			*(((float*)p_data) + i * 2 + n_vertex_num * 2) = f_s;
			*(((float*)p_data) + i * 2 + 1 + n_vertex_num * 2) = f_t;
			// texture coords are in the other half of array
		}

		if(!m_vertex_buffer.UnMap())
			return false;

		m_n_texcoords_offset = n_vertex_num * 2 * sizeof(float);
		m_n_tesselation = n_tesselation;
	}
	// create or re-tesselate vertex buffer

	m_vertex_buffer.Bind();

	glTexCoordPointer(2, GL_FLOAT, 0, m_vertex_buffer.p_OffsetPointer(m_n_texcoords_offset));
	glVertexPointer(2, GL_FLOAT, 0, m_vertex_buffer.p_OffsetPointer(0));

	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glEnableClientState(GL_VERTEX_ARRAY);

	p_state->ActiveTextureUnit(0);
	p_state->EnableTexture2D();
	p_state->BindTexture2D(n_source_texture);
	// bind texture with video

	m_p_shader->BindGL();

	m_p_shader->SetGLShaderParams(m_p_shader_param, m_n_shader_param_num);
	// bind shader to set it active

	glMatrixMode(GL_TEXTURE);
	glPushMatrix();
	glScalef(1, f_OuterRadius() - f_InnerRadius(), 1);

	glDrawArrays(GL_QUAD_STRIP, 0, 2 * (m_n_tesselation + 1) * m_n_tesselation);
	// draw a single big (degenerate) quadriliteral strip, vertex data
	// are on the GPU side so we can afford lots of vertices

	glPopMatrix();

	CGLShaderUnBinder::_UnBind();
	// unbind shader

	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);

	CVBOUnBinder::UnBind();
	// unbind vertex buffer

	return true;
}

/*
 *		-end-of-file-
 */
