/*
								+---------------------------------+
								|                                 |
								|      ***   OpenGL2.0   ***      |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|            Shader.cpp           |
								|                                 |
								+---------------------------------+
*/

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <gl/gl.h>
#include <vector>

#include "callstack.h"
#include "glext.h"
#include "wglext.h"
#include "opengl20.h"
#include "glstate.h"
#include "shader.h"


/*
 *								=== CShaderInfo ===
 */

int CShaderInfo::m_n_id_space = 0;

/*
 *	CShaderInfo::CShaderInfo()
 *		- default constructor
 */
CShaderInfo::CShaderInfo(const char *p_s_name, int n_default_lang, int n_type)
	:m_n_default_lang(n_default_lang),
	m_n_use_lang(n_default_lang),
	m_n_type(n_type),
	m_b_high_level_presented(false),
	m_b_low_level_presented(false),
	m_p_s_name(0)
{
	__FuncGuard("CShaderInfo::CShaderInfo");

	m_n_id = m_n_id_space ++;

	if(p_s_name && (m_p_s_name = new char[strlen(p_s_name) + 1]))
		strcpy(m_p_s_name, p_s_name);
	// copy name
}

/*
 *	int CShaderInfo::SetHighLevel_Code(const TShaderInfo &r_code)
 *		- set high level language shader source code
 */
int CShaderInfo::SetHighLevel_Code(const TShaderInfo &r_code)
{
	m_b_high_level_presented = true;

	return m_t_high_level = r_code;
}

/*
 *	int CShaderInfo::SetLowLevel_Code(const TShaderInfo &r_code)
 *		- set low level language shader source code
 */
int CShaderInfo::SetLowLevel_Code(const TShaderInfo &r_code)
{
	m_b_low_level_presented = true;

	return m_t_low_level = r_code;
}

/*
 *	CShaderInfo::~CShaderInfo()
 *		- default destructor
 */
CShaderInfo::~CShaderInfo()
{
	__FuncGuard("CShaderInfo::~CShaderInfo");

	if(m_p_s_name) {
		delete[] m_p_s_name;
		m_p_s_name = 0;
	}
	if(m_b_high_level_presented)
		m_t_high_level.Free();
	if(m_b_low_level_presented)
		m_t_low_level.Free();
}

/*
 *	int CShaderInfo::n_Language()
 *		- return language, shader will compile with
 */
int CShaderInfo::n_Language() const
{
	__FuncGuard("CShaderInfo::n_Language");

	return m_n_use_lang;
}

/*
 *	void CShaderInfo::PreferLanguage(int n_language)
 *		- tell shader which language is preferred
 */
void CShaderInfo::PreferLanguage(int n_language)
{
	__FuncGuard("CShaderInfo::PreferLanguage");

	if(n_language == sh_lang_None)
		n_language = m_n_default_lang;
	if(n_language == sh_lang_HighLevel && m_b_high_level_presented)
		m_n_use_lang = sh_lang_HighLevel;
	else if(n_language == sh_lang_LowLevel && m_b_low_level_presented)
		m_n_use_lang = sh_lang_LowLevel;
}

/*
 *	int CShaderInfo::b_CanBeLinkedWith(CShaderInfo &r_shader)
 *		- return true if r_shader is type and language compatible with <this>
 */
int CShaderInfo::b_CanBeLinkedWith(CShaderInfo &r_shader)
{
	__FuncGuard("CShaderInfo::b_CanBeLinkedWith");

	if(r_shader.m_n_type == m_n_type)
		return false;
	// dva fragment shadery nebo dva vertex shadery linkovat nejde

	if(r_shader.m_n_use_lang == m_n_use_lang)
		return true;
	// pokud pouvaj stejn jazyk, je to bez problmu

	if(m_b_high_level_presented && r_shader.m_b_high_level_presented) {
		r_shader.m_n_use_lang = m_n_use_lang = sh_lang_HighLevel;
		return true;
	}
	if(m_b_low_level_presented && r_shader.m_b_low_level_presented) {
		r_shader.m_n_use_lang = m_n_use_lang = sh_lang_LowLevel;
		return true;
	}
	// zkus nastavit stejn jazyk, ale nebude to ten preferovan

	return false;
}

/*
 *	void CShaderInfo::Free()
 *		- free shader data
 */
void CShaderInfo::Free()
{
	__FuncGuard("CShaderInfo::Free");

	this->~CShaderInfo();
}

/*
 *	int CShaderInfo::n_Sampler_Num()
 *		- return no. of texture samplers in high-level (glsl) shader
 *		- in case low-level shader is used, return 0
 */
int CShaderInfo::n_Sampler_Num() const
{
	__FuncGuard("CShaderInfo::n_Sampler_Num");

	if(m_n_use_lang != sh_lang_HighLevel)
		return 0;

	_ASSERTE(m_b_high_level_presented);

	return m_t_high_level.n_param_num;
}

/*
 *	int CShaderInfo::n_Param_Num()
 *		- return no. of program parameters in low-level shader
 *		- in case high-level (glsl) shader is used, return 0
 */
int CShaderInfo::n_Param_Num() const
{
	__FuncGuard("CShaderInfo::n_Param_Num");

	if(m_n_use_lang != sh_lang_LowLevel)
		return 0;

	_ASSERTE(m_b_low_level_presented);

	return m_t_low_level.n_param_num;
}

/*
 *	const char *CShaderInfo::p_s_Sampler(int n_index)
 *		- return name of sampler, corresponding with texture unit [n_index]
 *		- can return 0 if no sampler is supposed to operate in current unit
 *		- return 0 if index is out of range <0, n_Sampler_Num())
 *		- return 0 if language is not high-level (glsl)
 */
const char *CShaderInfo::p_s_Sampler(int n_index) const
{
	__FuncGuard("CShaderInfo::p_s_Sampler");

	if(m_n_use_lang != sh_lang_HighLevel)
		return 0;

	_ASSERTE(m_b_high_level_presented);

	if(n_index < 0 || n_index >= m_t_high_level.n_param_num)
		return 0;

	return m_t_high_level.p_param_name_array[n_index];
}


/*
 *	const char *CShaderInfo::p_s_Param(int n_index)
 *		- return name of sampler, corresponding with parameter [n_index]
 *		- return 0 if index is out of range <0, n_Param_Num())
 *		- return 0 if shader language is not low-level
 *		- param name is formated like "env\0param_name\0register_index"
 *		  for enviroment params or "local\0param_name\0register_index" for
 *		  local params
 */
const char *CShaderInfo::p_s_Param(int n_index) const
{
	__FuncGuard("CShaderInfo::p_s_Param");

	if(m_n_use_lang != sh_lang_LowLevel)
		return 0;

	_ASSERTE(m_b_low_level_presented);

	if(n_index < 0 || n_index >= m_t_low_level.n_param_num)
		return 0;

	return m_t_low_level.p_param_name_array[n_index];
}

/*
 *	const char *CShaderInfo::p_s_Shader_Code()
 *		- return zero-terminated string, containig source code of shader
 *		- return 0 when no shader is loaded / presented
 */
const char *CShaderInfo::p_s_Shader_Code() const
{
	__FuncGuard("CShaderInfo::p_s_Shader_Code");

	if(m_n_use_lang == sh_lang_HighLevel && m_b_high_level_presented)
		return m_t_high_level.p_s_code;
	if(m_n_use_lang == sh_lang_LowLevel && m_b_low_level_presented)
		return m_t_low_level.p_s_code;
	return 0;
}

/*
 *	int CShaderInfo::n_ShaderType() const
 *		- return type (either vertex or fragment) of shader
 */
int CShaderInfo::n_ShaderType() const
{
	__FuncGuard("CShaderInfo::n_ShaderType");

	return m_n_type;
}

/*
 *								=== ~CShaderInfo ===
 */

/*
 *								=== TGLShaderParam ===
 */

/*
 * format desc:

params { // params by mon mly bt sp v materilu ..? jo. pro kompatibilitu s low-level shaderem sm bt parametry jen float a jen do velikosti vec4f (cokoliv vtho mus nastavovat skript a jen v shaderu, kde low-level verze nen. - a mlo by tam bt napsan e low-level verze bt nesm)
	param[0] { p_s_name = "time"; p_s_value = "#t"; };
	param[1] { p_s_name = "local_time"; p_s_value = "#local_t"; };
	param[2] { p_s_name = "delta_time"; p_s_value = "#dt"; };
	param[3] { p_s_name = "constant"; f_value = 3.14; /* vec3f_value = {0.0, 3.1, 1.2}; * / };
	// f_value, vec2f_value, vec3f_value, vec4f_value

	param[4] { p_s_name = "oscillator"; p_s_value = "#osc sin 2.5 4.7 0.0 0"; }; // osciltory jsou f_frekvence, f_perioda, f_dc_offset, b_runonce
	// vdy globln as :(

	param[5] { p_s_name = "by_script"; p_s_script = "__extern void glUniform1f(int, float); int main() { return 0; } void SetUniform(int n_location) { glUniform1f(n_location, 5.5); } "; };
	// void SetUniform(int n_location); !!! pouvat v glsl-only shaderech !!!

	param[6] { p_s_name = "sys_var"; p_s_value = "#sys f_time"; };
	// shaderu se ped pointer na promnnou, promnn mus zstat platn tak dlouho, dokud se povrch kresl!
};

*/

TGLShaderParam::TGLShaderParam()
	:p_s_name(0),
	f_value(0),
	b_skip(false),
	p_previous(0)
{
	__FuncGuard("TGLShaderParam::TGLShaderParam");
}

void TGLShaderParam::Free()
{
	__FuncGuard("TGLShaderParam::Free");

	if(!b_skip && f_value) {
		delete[] f_value;
		f_value = 0;
	}
	if(p_s_name) {
		delete[] p_s_name;
		p_s_name = 0;
	}
}

/*
 *								=== ~TGLShaderParam ===
 */

/*
 *								=== CGLProgram ===
 */

CGLProgram *CGLProgram::m_p_bound_program = 0;

/*
 *								=== ~CGLProgram ===
 */

/*
 *								=== CShaderStringTable ===
 */

class CShaderStringTable {
public:
	static const char *p_s_ARB_VertexProgram_String() { return "=== vertex program (ARB) ===\n"; }
	static const char *p_s_ARB_FragmentProgram_String() { return "=== fragment program (ARB) ===\n"; }
	static const char *p_s_ARB_VertexShader_String() { return "=== vertex shader (ARB) ===\n"; }
	static const char *p_s_ARB_FragmentShader_String() { return "=== fragment shader (ARB) ===\n"; }
	static const char *p_s_Core_VertexShader_String() { return "=== vertex shader (OpenGL 2.0 core) ===\n"; }
	static const char *p_s_Core_FragmentShader_String() { return "=== fragment shader (OpenGL 2.0 core) ===\n"; }
};

/*
 *								=== ~CShaderStringTable ===
 */

/*
 *								=== CGL_ARB_shader ===
 */

CGL_ARB_shader::CGL_ARB_shader(const char *p_s_vertex_shader, const char *p_s_fragment_shader)
	:CGLShader(p_s_vertex_shader, p_s_fragment_shader),
	m_b_compiled(false),
	m_n_program_object(0),
	m_n_vs_object(0),
	m_n_fs_object(0)
{
	__FuncGuard("CGL_ARB_shader::CGL_ARB_shader");
}

CGL_ARB_shader::~CGL_ARB_shader()
{
	__FuncGuard("CGL_ARB_shader::~CGL_ARB_shader");

	GL_Free();
}

/*
 *	int CGL_ARB_shader::Compile(char *&r_p_s_info_log)
 *		- compile program and return info-log, which has to be freed
 *		- if you re-compile program, it call GL_Free() first
 */
int CGL_ARB_shader::Compile(char *&r_p_s_info_log)
{
	__FuncGuard("CGL_ARB_shader::Compile");

	GL_Free();
	r_p_s_info_log = 0;

	m_n_program_object = glCreateProgramObjectARB();

	int n_length, b_compiled;

	m_n_vs_object = (m_p_s_vertex_shader)? glCreateShaderObjectARB(GL_VERTEX_SHADER_ARB) : 0;
	if(m_n_vs_object) {
		n_length = strlen(m_p_s_vertex_shader);
		glShaderSourceARB(m_n_vs_object, 1, &m_p_s_vertex_shader, &n_length);
		glCompileShaderARB(m_n_vs_object);
		// compile vs ...

		glGetObjectParameterivARB(m_n_vs_object, GL_OBJECT_COMPILE_STATUS_ARB, &b_compiled);
		glGetObjectParameterivARB(m_n_vs_object, GL_OBJECT_INFO_LOG_LENGTH_ARB, &n_length);
		// query status ...

		if(n_length > 1) {
			if(!(r_p_s_info_log = new char[n_length + strlen(CShaderStringTable::p_s_ARB_VertexShader_String())]))
				return false;
			glGetInfoLogARB(m_n_vs_object, n_length, &n_length, r_p_s_info_log + strlen(CShaderStringTable::p_s_ARB_VertexShader_String()));
			strncpy(r_p_s_info_log, CShaderStringTable::p_s_ARB_VertexShader_String(), strlen(CShaderStringTable::p_s_ARB_VertexShader_String()));
		}
		// get info-log

		if(!b_compiled)
			return false;
	}
	// compile vertex shader

	m_n_fs_object = (m_p_s_fragment_shader)? glCreateShaderObjectARB(GL_FRAGMENT_SHADER_ARB) : 0;
	if(m_n_fs_object) {
		n_length = strlen(m_p_s_fragment_shader);
		glShaderSourceARB(m_n_fs_object, 1, &m_p_s_fragment_shader, &n_length);
		glCompileShaderARB(m_n_fs_object);
		// compile vs ...

		glGetObjectParameterivARB(m_n_fs_object, GL_OBJECT_COMPILE_STATUS_ARB, &b_compiled);
		glGetObjectParameterivARB(m_n_fs_object, GL_OBJECT_INFO_LOG_LENGTH_ARB, &n_length);
		// query status ...

		if(n_length > 1) {
			if(m_n_vs_object && r_p_s_info_log) {
				char *p_s_temp_info_log;
				if(!(p_s_temp_info_log = new char[n_length + strlen(r_p_s_info_log) + 2 + strlen(CShaderStringTable::p_s_ARB_FragmentShader_String())]))
					return false;
				strcpy(p_s_temp_info_log, r_p_s_info_log);
				strcat(p_s_temp_info_log, "\n\n");
				strcat(p_s_temp_info_log, CShaderStringTable::p_s_ARB_FragmentShader_String());
				glGetInfoLogARB(m_n_fs_object, n_length, &n_length,
					p_s_temp_info_log + strlen(r_p_s_info_log) + 2 + strlen(CShaderStringTable::p_s_ARB_FragmentShader_String()));
				delete[] r_p_s_info_log;
				r_p_s_info_log = p_s_temp_info_log;
			} else {
				if(!(r_p_s_info_log = new char[n_length + strlen(CShaderStringTable::p_s_ARB_FragmentShader_String())]))
					return false;
				glGetInfoLogARB(m_n_fs_object, n_length, &n_length, r_p_s_info_log + strlen(CShaderStringTable::p_s_ARB_FragmentShader_String()));
				strncpy(r_p_s_info_log, CShaderStringTable::p_s_ARB_FragmentShader_String(), strlen(CShaderStringTable::p_s_ARB_FragmentShader_String()));
			}
		}
		// get info-log

		if(!b_compiled)
			return false;
	}
	// compile fragment shader

	if(m_n_vs_object)
		glAttachObjectARB(m_n_program_object, m_n_vs_object);
	if(m_n_fs_object)
		glAttachObjectARB(m_n_program_object, m_n_fs_object);
	// attach shaders to a program

	if(glGetError() == GL_NO_ERROR)
		m_b_compiled = true;

	//m_p_bound_program = this; // i_fxme - program isn't bound

	return m_b_compiled;
}

/*
 *	int CGL_ARB_shader::BindAttribLocation(int n_location, const char *p_s_attrib_name)
 *		- bind vertex attrib location
 *		- can be called only if program was succesfuly compiled and not linked
 */
int CGL_ARB_shader::BindAttribLocation(int n_location, const char *p_s_attrib_name)
{
	__FuncGuard("CGL_ARB_shader::BindAttribLocation");

	if(!m_b_compiled || m_b_linked)
		return false;
	// must be compiled, but not linked!

	glBindAttribLocationARB(m_n_program_object, n_location, p_s_attrib_name);

	return glGetError() == GL_NO_ERROR;
}

/*
 *	int CGL_ARB_shader::Link(char *&r_p_s_info_log)
 *		- link program, return info-log which has to be freed
 *		- can be called only if program was succesfuly compiled and not linked
 */
int CGL_ARB_shader::Link(char *&r_p_s_info_log)
{
	__FuncGuard("CGL_ARB_shader::Link");

	r_p_s_info_log = 0;
	if(!m_b_compiled || m_b_linked)
		return false;

	glLinkProgramARB(m_n_program_object);
	glGetObjectParameterivARB(m_n_program_object, GL_OBJECT_LINK_STATUS_ARB, &m_b_linked);
	// link

	int n_length;
	glGetObjectParameterivARB(m_n_program_object, GL_OBJECT_INFO_LOG_LENGTH_ARB, &n_length);
	if(n_length > 1) {
		if(!(r_p_s_info_log = new char[n_length]))
			return false;
		glGetInfoLogARB(m_n_program_object, n_length, &n_length, r_p_s_info_log);
	}
	// get info-log

	return m_b_linked && glGetError() == GL_NO_ERROR; // fixme - does it ever return true if only one shader is presented ??
}

int CGL_ARB_shader::GetUniformLocation(const char *p_s_uniform_name) const
{
	__FuncGuard("CGL_ARB_shader::GetUniformLocation");

	if(!m_b_compiled || !m_b_linked)
		return -1; // f_ixme - 0 or -1 - answer is -1

	return glGetUniformLocationARB(m_n_program_object, p_s_uniform_name);
}

int CGL_ARB_shader::SetParam1fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_ARB_shader::SetParam1fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform1fvARB(n_location, 1, p_value);

	return true;
}

int CGL_ARB_shader::SetParam2fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_ARB_shader::SetParam2fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform2fvARB(n_location, 1, p_value);

	return true;
}

int CGL_ARB_shader::SetParam3fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_ARB_shader::SetParam3fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform3fvARB(n_location, 1, p_value);

	return true;
}

int CGL_ARB_shader::SetParam4fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_ARB_shader::SetParam4fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform4fvARB(n_location, 1, p_value);

	return true;
}

int CGL_ARB_shader::SetParam1i(int n_location, int n_value)
{
	__FuncGuard("CGL_ARB_shader::SetParam1i");

	if(!m_b_compiled || !m_b_linked)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);

	glUniform1iARB(n_location, n_value);

	return true;
}

/*
 *	void CGL_ARB_shader::Bind()
 *		- binds shader
 */
void CGL_ARB_shader::Bind()
{
	__FuncGuard("CGL_ARB_shader::Bind");

	CGLProgram::Bind();

	glUseProgramObjectARB(m_n_program_object);
}

/*
 *	void CGL_ARB_shader::UnBind()
 *		- un-binds shader
 *		- used for safe switching between shaders with different api-s
 */
void CGL_ARB_shader::UnBind()
{
	__FuncGuard("CGL_ARB_shader::UnBind");

	CGLProgram::UnBind();

	glUseProgramObjectARB(0);
}

/*
 *	void CGL_ARB_shader::GL_Free()
 *		- free open-gl resources
 *		- if program is about to be used again, it must be re-compiled,
 *		  re-bound vertex attribs, re-linked and re-queried for unifroms
 */
void CGL_ARB_shader::GL_Free()
{
	__FuncGuard("CGL_ARB_shader::GL_Free");

	m_b_compiled = false;
	m_b_linked = false;

	if(m_n_program_object) {
		glDeleteObjectARB(m_n_program_object);
		m_n_program_object = 0;
	}
	if(m_n_vs_object) {
		glDeleteObjectARB(m_n_vs_object);
		m_n_vs_object = 0;
	}
	if(m_n_fs_object) {
		glDeleteObjectARB(m_n_fs_object);
		m_n_fs_object = 0;
	}
}

/*
 *	int CGL_ARB_shader::n_Api() const
 *		- return which api is used
 */
int CGL_ARB_shader::n_Api() const
{
	__FuncGuard("CGL_ARB_shader::n_Api");

	return api_ARB_shader;
}

/*
 *								 === ~CGL_ARB_shader ===
 */

/*
 *								 === CGL_Core_shader ===
 */

CGL_Core_shader::CGL_Core_shader(const char *p_s_vertex_shader, const char *p_s_fragment_shader)
	:CGLShader(p_s_vertex_shader, p_s_fragment_shader),
	m_b_compiled(false),
	m_b_linked(false),
	m_n_program_object(0),
	m_n_vs_object(0),
	m_n_fs_object(0)
{
	__FuncGuard("CGL_Core_shader::CGL_Core_shader");
}

CGL_Core_shader::~CGL_Core_shader()
{
	__FuncGuard("CGL_Core_shader::~CGL_Core_shader");

	GL_Free();
}

/*
 *	int CGL_Core_shader::Compile(char *&r_p_s_info_log)
 *		- compile program and return info-log, which has to be freed
 *		- if you re-compile program, it call GL_Free() first
 */
int CGL_Core_shader::Compile(char *&r_p_s_info_log)
{
	__FuncGuard("CGL_Core_shader::Compile");

	GL_Free();
	r_p_s_info_log = 0;

	m_n_program_object = glCreateProgram();

	int n_length, b_compiled;

	m_n_vs_object = (m_p_s_vertex_shader)? glCreateShader(GL_VERTEX_SHADER) : 0;
	if(m_n_vs_object) {
		n_length = strlen(m_p_s_vertex_shader);
		glShaderSource(m_n_vs_object, 1, &m_p_s_vertex_shader, &n_length);
		glCompileShader(m_n_vs_object);
		// compile vs ...

		glGetShaderiv(m_n_vs_object, GL_COMPILE_STATUS, &b_compiled);
		glGetShaderiv(m_n_vs_object, GL_INFO_LOG_LENGTH, &n_length);
		// query status ...

		if(n_length > 1) {
			if(!(r_p_s_info_log = new char[n_length + strlen(CShaderStringTable::p_s_Core_VertexShader_String())]))
				return false;
			glGetShaderInfoLog(m_n_vs_object, n_length, &n_length, r_p_s_info_log + strlen(CShaderStringTable::p_s_Core_VertexShader_String()));
			strncpy(r_p_s_info_log, CShaderStringTable::p_s_Core_VertexShader_String(), strlen(CShaderStringTable::p_s_Core_VertexShader_String()));
		}
		// get info-log

		if(!b_compiled)
			return false;
	}
	// compile vertex shader

	m_n_fs_object = (m_p_s_fragment_shader)? glCreateShader(GL_FRAGMENT_SHADER) : 0;
	if(m_n_fs_object) {
		n_length = strlen(m_p_s_fragment_shader);
		glShaderSource(m_n_fs_object, 1, &m_p_s_fragment_shader, &n_length);
		glCompileShader(m_n_fs_object);
		// compile vs ...

		glGetShaderiv(m_n_fs_object, GL_COMPILE_STATUS, &b_compiled);
		glGetShaderiv(m_n_fs_object, GL_INFO_LOG_LENGTH, &n_length);
		// query status ...

		if(n_length > 1) {
			if(m_n_vs_object && r_p_s_info_log) {
				char *p_s_temp_info_log;
				if(!(p_s_temp_info_log = new char[n_length + strlen(r_p_s_info_log) + 2 + strlen(CShaderStringTable::p_s_Core_FragmentShader_String())]))
					return false;
				strcpy(p_s_temp_info_log, r_p_s_info_log);
				strcat(p_s_temp_info_log, "\n\n");
				strcat(p_s_temp_info_log, CShaderStringTable::p_s_Core_FragmentShader_String());
				glGetShaderInfoLog(m_n_fs_object, n_length, &n_length,
					p_s_temp_info_log + strlen(r_p_s_info_log) + 2 + strlen(CShaderStringTable::p_s_Core_FragmentShader_String()));
				delete[] r_p_s_info_log;
				r_p_s_info_log = p_s_temp_info_log;
			} else {
				if(!(r_p_s_info_log = new char[n_length + strlen(CShaderStringTable::p_s_Core_FragmentShader_String())]))
					return false;
				glGetShaderInfoLog(m_n_fs_object, n_length, &n_length, r_p_s_info_log + strlen(CShaderStringTable::p_s_Core_FragmentShader_String()));
				strncpy(r_p_s_info_log, CShaderStringTable::p_s_Core_FragmentShader_String(), strlen(CShaderStringTable::p_s_Core_FragmentShader_String()));
			}
		}
		// get info-log

		if(!b_compiled)
			return false;
	}
	// compile fragment shader

	if(m_n_vs_object)
		glAttachShader(m_n_program_object, m_n_vs_object);
	if(m_n_fs_object)
		glAttachShader(m_n_program_object, m_n_fs_object);
	// attach shaders to a program

	if(glGetError() == GL_NO_ERROR)
		m_b_compiled = true;

	//m_p_bound_program = this; // f_ixme - program isn't bound

	/*glGetIntegerv(GL_CURRENT_PROGRAM, &m_b_compiled);
	m_b_compiled = glGetError();*/

	return m_b_compiled;
}

/*
 *	int CGL_Core_shader::BindAttribLocation(int n_location, const char *p_s_attrib_name)
 *		- bind vertex attrib location
 *		- can be called only if program was succesfuly compiled and not linked
 */
int CGL_Core_shader::BindAttribLocation(int n_location, const char *p_s_attrib_name)
{
	__FuncGuard("CGL_Core_shader::BindAttribLocation");

	if(!m_b_compiled || m_b_linked)
		return false;
	// must be compiled, but not linked!

	glBindAttribLocation(m_n_program_object, n_location, p_s_attrib_name);

	return glGetError() == GL_NO_ERROR;
}

/*
 *	int CGL_Core_shader::Link(char *&r_p_s_info_log)
 *		- link program, return info-log which has to be freed
 *		- can be called only if program was succesfuly compiled and not linked
 */
int CGL_Core_shader::Link(char *&r_p_s_info_log)
{
	__FuncGuard("CGL_Core_shader::Link");

	r_p_s_info_log = 0;
	if(!m_b_compiled || m_b_linked)
		return false;

	glLinkProgram(m_n_program_object);
	glGetProgramiv(m_n_program_object, GL_LINK_STATUS, &m_b_linked);
	// link

	int n_length;
	glGetProgramiv(m_n_program_object, GL_INFO_LOG_LENGTH, &n_length);
	if(n_length > 1) {
		if(!(r_p_s_info_log = new char[n_length]))
			return false;
		glGetProgramInfoLog(m_n_program_object, n_length, &n_length, r_p_s_info_log);
	}
	// get info-log

	return m_b_linked;
}

int CGL_Core_shader::GetUniformLocation(const char *p_s_uniform_name) const
{
	__FuncGuard("CGL_Core_shader::GetUniformLocation");

	if(!m_b_compiled || !m_b_linked)
		return -1; // f_ixme - 0 or -1 - answer is -1

	return glGetUniformLocation(m_n_program_object, p_s_uniform_name);
}

int CGL_Core_shader::SetParam1fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_Core_shader::SetParam1fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != (CGLProgram*)this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform1fv(n_location, 1, p_value);

	return true;
}

int CGL_Core_shader::SetParam2fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_Core_shader::SetParam2fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform2fv(n_location, 1, p_value);

	return true;
}

int CGL_Core_shader::SetParam3fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_Core_shader::SetParam3fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform3fv(n_location, 1, p_value);

	return true;
}

int CGL_Core_shader::SetParam4fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_Core_shader::SetParam4fv");

	if(!m_b_compiled || !m_b_linked || n_param_type != shader_param_Uniform)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	glUniform4fv(n_location, 1, p_value);

	return true;
}

int CGL_Core_shader::SetParam1i(int n_location, int n_value)
{
	__FuncGuard("CGL_Core_shader::SetParam1i");

	if(!m_b_compiled || !m_b_linked)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);

	glUniform1i(n_location, n_value);

	return true;
}

/*
 *	void CGL_Core_shader::Bind()
 *		- binds shader
 */
void CGL_Core_shader::Bind()
{
	__FuncGuard("CGL_Core_shader::Bind");

	CGLProgram::Bind();

	glUseProgram(m_n_program_object);
}

/*
 *	void CGL_Core_shader::UnBind()
 *		- un-binds shader
 *		- used for safe switching between shaders with different api-s
 */
void CGL_Core_shader::UnBind()
{
	__FuncGuard("CGL_Core_shader::UnBind");

	CGLProgram::UnBind();

	glUseProgram(0);
}

/*
 *	void CGL_Core_shader::GL_Free()
 *		- free open-gl resources
 *		- if program is about to be used again, it must be re-compiled,
 *		  re-bound vertex attribs, re-linked and re-queried for unifroms
 */
void CGL_Core_shader::GL_Free()
{
	__FuncGuard("CGL_Core_shader::GL_Free");

	CGLProgram::UnBind();

	m_b_compiled = false;
	m_b_linked = false;

	if(m_n_program_object) {
		glDeleteProgram(m_n_program_object);
		m_n_program_object = 0;
	}
	if(m_n_vs_object) {
		glDeleteShader(m_n_vs_object);
		m_n_vs_object = 0;
	}
	if(m_n_fs_object) {
		glDeleteShader(m_n_fs_object);
		m_n_fs_object = 0;
	}
}

/*
 *	int CGL_Core_shader::n_Api() const
 *		- return which api is used
 */
int CGL_Core_shader::n_Api() const
{
	__FuncGuard("CGL_Core_shader::n_Api");

	return api_Core_shader;
}

/*
 *								=== ~CGL_Core_shader ===
 */

/*
 *								=== CGL_ARB_program ===
 */

CGL_ARB_program::CGL_ARB_program(const char *p_s_vertex_shader, const char *p_s_fragment_shader)
	:CGLProgram(p_s_vertex_shader, p_s_fragment_shader),
	m_b_compiled(false),
	m_b_linked(false),
	m_n_vs_object(0),
	m_n_fs_object(0)
{
	__FuncGuard("CGL_ARB_program::CGL_ARB_program");
}

CGL_ARB_program::~CGL_ARB_program()
{
	__FuncGuard("CGL_ARB_program::~CGL_ARB_program");

	GL_Free();
}

/*
 *	int CGL_ARB_program::Compile(char *&r_p_s_info_log)
 *		- compile program and return info-log, which has to be freed
 *		- if you re-compile program, it call GL_Free() first
 */
int CGL_ARB_program::Compile(char *&r_p_s_info_log)
{
	__FuncGuard("CGL_ARB_program::Compile");

	GL_Free();
	r_p_s_info_log = 0;

	int n_length, b_compiled;

	if(m_p_s_vertex_shader) {
		glGenProgramsARB(1, &m_n_vs_object);
		glBindProgramARB(GL_VERTEX_PROGRAM_ARB, m_n_vs_object);
		// create shader

		n_length = strlen(m_p_s_vertex_shader);
		glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
			n_length, m_p_s_vertex_shader);
		// compile vs ...

		int n_error;
		glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &n_error);
		b_compiled = n_error == -1;

		const char *p_s_error = (char*)glGetString(GL_PROGRAM_ERROR_STRING_ARB);
		// query status ...
		if(p_s_error && (n_length = strlen(p_s_error) + 1) > 1) {
			if(!(r_p_s_info_log = new char[n_length + strlen(CShaderStringTable::p_s_ARB_VertexProgram_String())]))
				return false;
			strcpy(r_p_s_info_log, CShaderStringTable::p_s_ARB_VertexProgram_String());
			strcat(r_p_s_info_log, p_s_error);
		}
		// get info-log

		if(!b_compiled)
			return false;
	}
	// compile vertex shader

	if(m_p_s_fragment_shader) {
		glGenProgramsARB(1, &m_n_fs_object);
		glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, m_n_fs_object);
		// create shader

		n_length = strlen(m_p_s_fragment_shader);
		glProgramStringARB(GL_FRAGMENT_PROGRAM_ARB, GL_PROGRAM_FORMAT_ASCII_ARB,
			n_length, m_p_s_fragment_shader);
		// compile fs ...

		int n_error;
		glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, &n_error);
		b_compiled = n_error == -1;

		const char *p_s_error = (char*)glGetString(GL_PROGRAM_ERROR_STRING_ARB);
		// query status ...
		if(p_s_error && (n_length = strlen(p_s_error) + 1) > 1) {
			if(m_n_vs_object && r_p_s_info_log) {
				char *p_s_temp_info_log;
				if(!(p_s_temp_info_log = new char[n_length + strlen(r_p_s_info_log) + 2 + strlen(CShaderStringTable::p_s_ARB_FragmentProgram_String())]))
					return false;
				strcpy(p_s_temp_info_log, r_p_s_info_log);
				strcat(p_s_temp_info_log, "\n\n");
				strcat(p_s_temp_info_log, CShaderStringTable::p_s_ARB_FragmentProgram_String());
				strcat(p_s_temp_info_log, p_s_error);
				delete[] r_p_s_info_log;
				r_p_s_info_log = p_s_temp_info_log;
			} else {
				if(!(r_p_s_info_log = new char[n_length + strlen(CShaderStringTable::p_s_ARB_FragmentProgram_String())]))
					return false;
				strcpy(r_p_s_info_log, CShaderStringTable::p_s_ARB_FragmentProgram_String());
				strcat(r_p_s_info_log, p_s_error);
			}
		}
		// get info-log

		if(!b_compiled)
			return false;
	}
	// compile fragment shader

	if((m_b_compiled = glGetError()) == GL_NO_ERROR)
		m_b_compiled = true;

	m_p_bound_program = this;
	// f_ixme - we have calls of glBindProgramARB, so i suppose program is bound

	return m_b_compiled;
}

/*
 *	int CGL_ARB_program::SetParam4fv(int n_param_type, int n_location, const float *p_value) const
 *		- set shader parameter; shader has to be bound
 *		- n_param_type must be one of shader_param_Parameter_Env,
 *		  shader_param_Parameter_Local_VP or shader_param_Parameter_Local_FP
 *		- n_location is index of register, parameter is located in
 *		- p_value is pointer to 4 floats, data will be read from
 */
int CGL_ARB_program::SetParam4fv(int n_param_type, int n_location, const float *p_value)
{
	__FuncGuard("CGL_ARB_program::SetParam4fv");

	if(!m_b_compiled)
		return false;
	if(m_p_bound_program != this)
		Bind();

	_ASSERTE(n_location >= 0);
	_ASSERTE(p_value);

	if(n_param_type == shader_param_Parameter_Env) {
		if(m_p_s_vertex_shader)
			glProgramEnvParameter4fvARB(GL_VERTEX_PROGRAM_ARB, n_location, p_value);
		else if(m_p_s_fragment_shader)
			glProgramEnvParameter4fvARB(GL_FRAGMENT_PROGRAM_ARB, n_location, p_value);
		else
			return false; // no shader available
	} else if(n_param_type == shader_param_Parameter_Local_VP) {
		if(m_p_s_vertex_shader)
			glProgramLocalParameter4fvARB(GL_VERTEX_PROGRAM_ARB, n_location, p_value);
		else
			return false; // bad shader type
	} else if(n_param_type == shader_param_Parameter_Local_FP) {
		if(m_p_s_vertex_shader)
			glProgramLocalParameter4fvARB(GL_FRAGMENT_PROGRAM_ARB, n_location, p_value);
		else
			return false; // bad shader type
	} else
		return false; // bad param type

	return true;
}

/*
 *	void CGL_ARB_program::Bind()
 *		- binds shader
 */
void CGL_ARB_program::Bind()
{
	__FuncGuard("CGL_ARB_program::Bind");

	CGLProgram::Bind();

	glBindProgramARB(GL_VERTEX_PROGRAM_ARB, m_n_vs_object);
	glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, m_n_fs_object);
}

/*
 *	void CGL_ARB_program::UnBind()
 *		- un-binds shader
 *		- used for safe switching between shaders with different api-s
 */
void CGL_ARB_program::UnBind()
{
	__FuncGuard("CGL_ARB_program::UnBind");

	CGLProgram::UnBind();

	glBindProgramARB(GL_VERTEX_PROGRAM_ARB, 0);
	glBindProgramARB(GL_FRAGMENT_PROGRAM_ARB, 0);
}

/*
 *	void CGL_ARB_program::GL_Free()
 *		- free open-gl resources
 *		- if program is about to be used again, it must be re-compiled,
 *		  re-bound vertex attribs, re-linked and re-queried for unifroms
 */
void CGL_ARB_program::GL_Free()
{
	__FuncGuard("CGL_ARB_program::GL_Free");

	CGLProgram::UnBind();

	m_b_compiled = false;
	m_b_linked = false;

	if(m_n_vs_object) {
		glDeleteProgramsARB(1, &m_n_vs_object);
		m_n_vs_object = 0;
	}
	if(m_n_fs_object) {
		glDeleteProgramsARB(1, &m_n_fs_object);
		m_n_fs_object = 0;
	}
}

/*
 *	int CGL_ARB_program::n_Api() const
 *		- return which api is used
 */
int CGL_ARB_program::n_Api() const
{
	__FuncGuard("CGL_ARB_program::n_Api");

	return api_ARB_program;
}

/*
 *								=== ~CGL_ARB_program ===
 */

/*
 *								=== CGLProgramObject ===
 */

int CGLProgramObject::m_n_id_space = 0;

/*
 *	CGLProgramObject::CGLProgramObject()
 *		- default constructor
 *		- all gl-dependent members (shader gl-id's, samplers) are invalid if m_b_ready is false
 */
CGLProgramObject::CGLProgramObject()
	:m_p_vertex_shader_info(0),
	m_p_fragment_shader_info(0),
	m_p_shader(0),
	m_b_ready(false),
	m_n_reference_num(0),
	m_n_preferred_shading_lang(sh_lang_None)
{
	__FuncGuard("CGLProgramObject::CGLProgramObject");
	m_n_id = m_n_id_space ++;
}

/*
 *	CGLProgramObject::~CGLProgramObject()
 *		- default destructor
 */
CGLProgramObject::~CGLProgramObject()
{
	__FuncGuard("CGLProgramObject::~CGLProgramObject");

	Free();
}

/*
 *	int CGLProgramObject::BindShaders(CShaderInfo *p_vertex_shader, CShaderInfo *p_s_fragment_shader)
 *		- bind shaders; program object is usable after both compiling and linking, not before
 */
int CGLProgramObject::BindShaders(CShaderInfo *p_vertex_shader, CShaderInfo *p_s_fragment_shader)
{
	__FuncGuard("CGLProgramObject::BindShaders");

	Free();
	m_p_vertex_shader_info = p_vertex_shader;
	m_p_fragment_shader_info = p_s_fragment_shader;

	return true;
}

/*
 *	const char *CGLProgramObject::p_s_VertexShader_Name()
 *		- return name of vertex shader or 0 if there's no vertex shader
 */
const char *CGLProgramObject::p_s_VertexShader_Name()
{
	__FuncGuard("CGLProgramObject::p_s_VertexShader_Name");

	if(m_p_vertex_shader_info)
		return m_p_vertex_shader_info->p_s_Name();
	return 0;
}

/*
 *	const char *CGLProgramObject::p_s_FragmentShader_Name()
 *		- return name of fragment shader or 0 if there's no fragment shader
 */
const char *CGLProgramObject::p_s_FragmentShader_Name()
{
	__FuncGuard("CGLProgramObject::p_s_FragmentShader_Name");

	if(m_p_fragment_shader_info)
		return m_p_fragment_shader_info->p_s_Name();
	return 0;
}

/*
 *	int CGLProgramObject::AddReference(int n_referencer_id)
 *		- increase reference counter
 *		- for release version return always true
 *		- for debug version check if program object wasn't already referenced by referencer
 *		  and return false if it actually was
 */
int CGLProgramObject::AddReference(int n_referencer_id)
{
	__FuncGuard("CGLProgramObject::AddReference");

	m_n_reference_num ++;

#ifdef _DEBUG
	for(size_t i = 0; i < m_referencer_id_list.size(); i ++) {
		if(m_referencer_id_list[i] == n_referencer_id) {
			m_referencer_id_list.erase(m_referencer_id_list.begin() + i);
			m_n_reference_num --;
			_ASSERTE(0);
			return false;
		}
	}
	m_referencer_id_list.push_back(n_referencer_id);
	if(!m_referencer_id_list.size())
		return false;
	return true;
#else
	return true;
#endif
}

/*
 *	int CGLProgramObject::RemoveReference(int n_referencer_id)
 *		- decrease reference counter
 *		- for release version return always true
 *		- for debug version check if program object was really referenced by referencer
 *		  and return false if it actually wasn't
 */
int CGLProgramObject::RemoveReference(int n_referencer_id)
{
	__FuncGuard("CGLProgramObject::RemoveReference");

	if(m_n_reference_num > 0)
		m_n_reference_num --;

#ifdef _DEBUG
	for(size_t i = 0; i < m_referencer_id_list.size(); i ++) {
		if(m_referencer_id_list[i] == n_referencer_id) {
			m_referencer_id_list.erase(m_referencer_id_list.begin() + i);
			return true;
		}
	}
	_ASSERTE(0);
	return false;
#else
	return true;
#endif
}

/*
 *	int CGLProgramObject::n_ReferenceCounter()
 *		- return value of reference counter
 */
int CGLProgramObject::n_ReferenceCounter()
{
	__FuncGuard("CGLProgramObject::n_ReferenceCounter");

	return m_n_reference_num;
}

/*
 *	void CGLProgramObject::PreferLang(int n_preferred_shading_lang)
 *		- tell shader object to prefer selected language (call before compiling)
 *		- language can be one of sh_lang_None, sh_lang_LowLevel or sh_lang_HighLevel
 */
void CGLProgramObject::PreferLang(int n_preferred_shading_lang)
{
	__FuncGuard("CGLProgramObject::PreferLang");

	_ASSERTE(n_preferred_shading_lang == sh_lang_None ||
			 n_preferred_shading_lang == sh_lang_LowLevel ||
			 n_preferred_shading_lang == sh_lang_HighLevel);
	m_n_preferred_shading_lang = n_preferred_shading_lang;
}

/*
 *	int CGLProgramObject::b_HaveCompatibleShaders()
 *		- return true if shaders are compatible (may set another
 *		  language is one of shaders is'n presented in desired language)
 */
int CGLProgramObject::b_HaveCompatibleShaders()
{
	__FuncGuard("CGLProgramObject::b_HaveCompatibleShaders");

	if(!m_p_vertex_shader_info || !m_p_fragment_shader_info)
		return true;

	m_p_vertex_shader_info->PreferLanguage(m_n_preferred_shading_lang);
	m_p_fragment_shader_info->PreferLanguage(m_n_preferred_shading_lang);

	return m_p_vertex_shader_info->b_CanBeLinkedWith(*m_p_fragment_shader_info);
}

/*
 *	int CGLProgramObject::n_PreferredLang()
 *		- return language, shaders are about to be compiled with
 *		- can be either sh_lang_LowLevel or sh_lang_HighLevel
 */
int CGLProgramObject::n_PreferredLang()
{
	__FuncGuard("CGLProgramObject::n_PreferredLang");

	if(b_HaveCompatibleShaders()) {
		if(m_p_vertex_shader_info)
			m_n_preferred_shading_lang = m_p_vertex_shader_info->n_Language();
		else if(m_p_fragment_shader_info)
			m_n_preferred_shading_lang = m_p_fragment_shader_info->n_Language();
	}
	return m_n_preferred_shading_lang;
}

/*
 *	int CGLProgramObject::CompileGLShaders(char *&r_p_s_compile_log)
 *		- compile shaders; if function fails because of compile error,
 *		  r_p_s_compile_log will contain pointer to string containing
 *		  compiler error message (it has to be freed)
 */
int CGLProgramObject::CompileGLShaders(char *&r_p_s_compile_log)
{
	__FuncGuard("CGLProgramObject::CompileGLShaders");

	if(m_p_shader) {
		delete m_p_shader;
		m_p_shader = 0;
	}
	m_sampler_list.clear();
	m_b_ready = false;

	r_p_s_compile_log = 0;

	if(!b_HaveCompatibleShaders())
		return false;

	if(n_PreferredLang() == sh_lang_LowLevel) {
		if(CGLExtensionHandler::b_SupportedExtension("GL_ARB_vertex_program") &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_program") &&
		   !CGLExtensionHandler::n_GetVertexProgramARBFuncPointers() &&
		   !CGLExtensionHandler::n_GetFragmentProgramARBFuncPointers()) {
			if(!(m_p_shader = new CGL_ARB_program(
			   (m_p_vertex_shader_info)? m_p_vertex_shader_info->p_s_Shader_Code() : 0,
			   (m_p_fragment_shader_info)? m_p_fragment_shader_info->p_s_Shader_Code() : 0)))
				return false;
		} else
			return false; // desired shading language not supported
	} else {
		if(CGLExtensionHandler::b_Support_OpenGL(2, 0) &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_shading_language_100") &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_vertex_shader") &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_shader") &&
		   !CGLExtensionHandler::n_GetGL20FuncPointers()) {
			if(!(m_p_shader = new CGL_Core_shader(
			   (m_p_vertex_shader_info)? m_p_vertex_shader_info->p_s_Shader_Code() : 0,
			   (m_p_fragment_shader_info)? m_p_fragment_shader_info->p_s_Shader_Code() : 0)))
				return false;
		} else if(CGLExtensionHandler::b_SupportedExtension("GL_ARB_shading_language_100") &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_vertex_shader") &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_shader") &&
		   CGLExtensionHandler::b_SupportedExtension("GL_ARB_shader_objects") &&
		   !CGLExtensionHandler::n_GetVertexShaderARBFuncPointers() &&
		   !CGLExtensionHandler::n_GetFragmentShaderARBFuncPointers() &&
		   !CGLExtensionHandler::n_GetShaderObjectARBFuncPointers()) {
			if(!(m_p_shader = new CGL_ARB_shader(
			   (m_p_vertex_shader_info)? m_p_vertex_shader_info->p_s_Shader_Code() : 0,
			   (m_p_fragment_shader_info)? m_p_fragment_shader_info->p_s_Shader_Code() : 0)))
				return false;
		} else
			return false; // desired shading language not supported
	}

	return m_p_shader->Compile(r_p_s_compile_log);
	// compile shaders ...
}

/*
 *	int CGLProgramObject::LinkGLShaders(char *&r_p_s_link_log)
 *		- link shaders; if it fails because of link error,
 *		  r_p_s_link_log will contain pointer to string containing
 *		  compiler error message (it has to be freed)
 */
int CGLProgramObject::LinkGLShaders(char *&r_p_s_link_log)
{
	__FuncGuard("CGLProgramObject::LinkGLShaders");

	r_p_s_link_log = 0;

	if(!m_p_shader || m_b_ready)
		return false;

	if(m_p_shader->n_Api() == api_ARB_shader || m_p_shader->n_Api() == api_Core_shader)
		return m_b_ready = ((CGLShader*)m_p_shader)->Link(r_p_s_link_log);
	return m_b_ready = true;
}

int CGLProgramObject::ResolveUniforms()
{
	__FuncGuard("CGLProgramObject::ResolveUniforms");

	if(!m_p_shader || !m_b_ready)
		return false;

	m_sampler_list.clear();

	if(m_p_shader->n_Api() == api_ARB_shader || m_p_shader->n_Api() == api_Core_shader) {
		if(m_p_vertex_shader_info) {
			for(int i = 0; i < m_p_vertex_shader_info->n_Sampler_Num(); i ++) {
				const char *p_s_sampler_name = m_p_vertex_shader_info->p_s_Sampler(i);

				TSamplerLoc t_sampler_loc;

				t_sampler_loc.p_s_sampler_name = p_s_sampler_name;
				if(!(t_sampler_loc.n_sampler_location =
				   ((CGLShader*)m_p_shader)->GetUniformLocation(p_s_sampler_name)))
					return false;
				// fixme 0 = bad or -1 = bad?

				m_sampler_list.push_back(t_sampler_loc);
				if(!m_sampler_list.size())
					return false;
			}
		}
		if(m_p_fragment_shader_info) {
			for(int i = 0; i < m_p_fragment_shader_info->n_Sampler_Num(); i ++) {
				const char *p_s_sampler_name = m_p_fragment_shader_info->p_s_Sampler(i);

				int b_found = false;
				for(size_t j = 0; j < m_sampler_list.size(); j ++) {
					if(!_stricmp(m_sampler_list[j].p_s_sampler_name, p_s_sampler_name)) {
						b_found = true;
						break;
					}
				}
				if(b_found)
					continue;

				TSamplerLoc t_sampler_loc;

				t_sampler_loc.p_s_sampler_name = p_s_sampler_name;
				if((t_sampler_loc.n_sampler_location =
				   ((CGLShader*)m_p_shader)->GetUniformLocation(p_s_sampler_name)) == -1)
					return false;
				// f_ixme 0 = bad or -1 = bad? -1=bad!

				m_sampler_list.push_back(t_sampler_loc);
				if(!m_sampler_list.size())
					return false;
			}
		}
	} else
		return true; // no uniforms for ARB_program

	return true;
}

/*
 *	int CGLProgramObject::FindParams(TGLShaderParam *p_param, int n_param_num)
 *		- find param registers and fill in param types (uniform / env /
 *		  local in vertex program / local in fragment program)
 *		- return false if no such a param found or if program wasn't compiled and linked
 */
int CGLProgramObject::FindParams(TGLShaderParam *p_param, int n_param_num)
{
	__FuncGuard("CGLProgramObject::FindParams");

	if(!m_p_shader || !m_b_ready)
		return false;

	for(int i = 0; i < n_param_num; i ++) {
		if(!strcmp(p_param[i].p_s_name, "#append"))
			continue;
		if(m_p_shader->n_Api() == api_ARB_shader || m_p_shader->n_Api() == api_Core_shader) {
			if((p_param[i].n_gl_location = ((CGLShader*)m_p_shader)->GetUniformLocation(p_param[i].p_s_name)) == -1)
				return false; // f_ixme - 0 = bad or -1 = bad ? -1=bad!
			p_param[i].n_gl_type = shader_param_Uniform;
		} else {
			int b_found = false;
			for(int j = 0; j < 2 && !b_found; j ++) {
				CShaderInfo *p_shader_info = (!j)? m_p_vertex_shader_info :
					m_p_fragment_shader_info;
				if(!p_shader_info)
					continue;

				for(int i = 0; i < p_shader_info->n_Param_Num(); i ++) {
					const char *p_s_param[3];

					p_s_param[0] = p_shader_info->p_s_Param(i);
					p_s_param[1] = p_s_param[0] + strlen(p_s_param[0]) + 1;
					p_s_param[2] = p_s_param[1] + strlen(p_s_param[1]) + 1;
					// extract params ...

					if(!_stricmp(p_s_param[1], p_param->p_s_name)) {
						p_param->n_gl_type = (!_stricmp(p_s_param[0], "env"))?
							shader_param_Parameter_Env : ((p_shader_info == m_p_vertex_shader_info)?
							shader_param_Parameter_Local_VP : shader_param_Parameter_Local_FP);
						p_param->n_gl_location = atol(p_s_param[2]);
						b_found = true;
						break;
					}
					// if name equals to p_param, set type and register index
				}
			}
			// try to find param ...

			if(!b_found)
				return false;
			// param not found ...
		}
	}

	return true;
}

/*
 *	void CGLProgramObject::FreeGLShaders()
 *		- free open-gl resources
 *		- this may invalidate param locations (in case prorgam
 *		  is recompiled with another language)
 *		- 
 */
void CGLProgramObject::FreeGLShaders()
{
	__FuncGuard("CGLProgramObject::FreeGLShaders");

	m_b_ready = false;
	if(m_p_shader) {
		m_p_shader->GL_Free();
		delete m_p_shader;
		m_p_shader = 0;
	}
}

/*
 *	void CGLProgramObject::Free()
 *		- free program object data
 *		- also call CGLProgramObject::FreeGLShaders()
 */
void CGLProgramObject::Free()
{
	__FuncGuard("CGLProgramObject::Free");

	m_sampler_list.clear();
	m_b_ready = false;
	m_n_reference_num = 0;

	FreeGLShaders();
}

/*
 *	int CGLProgramObject::BindGL()
 *		- bind shader (return true if succesful)
 */
int CGLProgramObject::BindGL()
{
	__FuncGuard("CGLProgramObject::BindGL");

	if(!m_p_shader || !m_b_ready)
		return false;

	for(size_t i = 0; i < m_sampler_list.size(); i ++)
		((CGLShader*)m_p_shader)->SetParam1i(m_sampler_list[i].n_sampler_location, i);

	m_p_shader->Bind();

	return true;
}

/*
 *	int CGLProgramObject::n_UsedApi()
 *		- return used api
 *		- in case shaer isn't ready, return api_Unsupported
 */
int CGLProgramObject::n_UsedApi()
{
	__FuncGuard("CGLProgramObject::n_UsedApi");

	if(!m_p_shader || !m_b_ready)
		return api_Unsupported;

	return m_p_shader->n_Api();
}

/*
 *	int CGLProgramObject::ReleaseGL()
 *		- release shader from gl
 *		- for best performance call only when previously set shader
 *		  was using different api
 */
int CGLProgramObject::ReleaseGL()
{
	__FuncGuard("CGLProgramObject::ReleaseGL");

	if(!m_p_shader || !m_b_ready)
		return false;

	m_p_shader->UnBind();

	return true;
}

int CGLProgramObject::SetGLShaderParams(const TGLShaderParam *p_param, int n_param_num)
{
	__FuncGuard("CGLProgramObject::SetGLShaderParams");

	if(!m_p_shader || !m_b_ready)
		return false;

	for(int i = 0; i < n_param_num; i ++) {
		if(p_param[i].b_skip)
			continue;
#if 1
		if(m_p_shader->n_Api() == api_ARB_shader ||
		   m_p_shader->n_Api() == api_Core_shader) {
			switch(p_param[i].n_size) {
			case 1:
				if(!((CGLShader*)m_p_shader)->SetParam1fv(p_param[i].n_gl_type,
				   p_param[i].n_gl_location, p_param[i].f_value))
					return false;
				break;
			case 2:
				if(!((CGLShader*)m_p_shader)->SetParam2fv(p_param[i].n_gl_type,
				   p_param[i].n_gl_location, p_param[i].f_value))
					return false;
				break;
			case 3:
				if(!((CGLShader*)m_p_shader)->SetParam3fv(p_param[i].n_gl_type,
				   p_param[i].n_gl_location, p_param[i].f_value))
					return false;
				break;
			case 4:
				if(!((CGLShader*)m_p_shader)->SetParam4fv(p_param[i].n_gl_type,
				   p_param[i].n_gl_location, p_param[i].f_value))
					return false;
				break;
			}
			// minimize data amount to run between cpu and gpu ...
			// fixme - maybe loss of time in switch() statement, try disabling this block
		} else
#endif
		{
			if(!m_p_shader->SetParam4fv(p_param[i].n_gl_type,
			   p_param[i].n_gl_location, p_param[i].f_value))
				return false;
		}
	}

	return true;
}

/*
 *								=== ~CGLProgramObject ===
 */

/*
 *		-end-of-file-
 */
