/*
								+---------------------------------+
								|                                 |
								|   ***   Rendering paths   ***   |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|         RenderPath.cpp          |
								|                                 |
								+---------------------------------+
*/

#include <crtdbg.h>
#include <windows.h>
#include <gl/gl.h>
#include <gl/glu.h>
#include <gl/glaux.h>
#include <vector>

#include "glext.h"
#include "wglext.h"
#include "opengl20.h"
#include "glstate.h"

#include "renderpath.h"
#include "callstack.h"

/*
 *								=== TPath ===
 */

/*
 *	TPath::TPath()
 *		- default constructor
 */
TPath::TPath()
	:p_extension_list(0),
	p_vendor_list(0),
	p_card_list(0),
	p_s_path_name(0)
{
	__FuncGuard("TPath::TPath");
}

/*
 *	int TPath::Create(const char *p_s_name, const char *p_s_extensions,
 *		const char *p_s_vendors, const char *p_s_models)
 *		- create path, based on required extensions, supported vendors and model list
 */
int TPath::Create(const char *p_s_name, const char *p_s_extensions,
				  const char *p_s_vendors, const char *p_s_models)
{
	__FuncGuard("TPath::Create");

	Free();

	_ASSERTE(p_s_name);
	if(!(p_s_path_name = new char[strlen(p_s_name) + 1]))
		return false;
	strcpy(p_s_path_name, p_s_name);
	if(p_s_extensions && !(p_extension_list = p_ParseStrList(p_s_extensions)))
		return false;
	if(p_s_vendors && !(p_vendor_list = p_ParseStrList(p_s_vendors)))
		return false;
	if(p_s_models && !(p_card_list = p_ParseStrList(p_s_models)))
		return false;
	// parse path data

	return true;
}

/*
 *	const char *TPath::p_s_ScanString(const char *p_s_str, char *p_s_dest_buffer)
 *		- scan string separated by whitespace (can be surrounded by quotes) from p_s_str
 *		  and store it to p_s_dest_buffer (it should be allocated to length of p_s_str
 *		  to avoid crossing array bounds)
 *		- return p_s_str, offset so next string can be scanned or 0 if there is no other string
 *		  (either p_s_str contain no characters or only whitespaces)
 */
const char *TPath::p_s_ScanString(const char *p_s_str, char *p_s_dest_buffer)
{
	__FuncGuard("TPath::p_s_ScanString");

	while(strlen(p_s_str) && isspace(*p_s_str))
		p_s_str ++;
	if(!strlen(p_s_str))
		return 0;
	// seer whitespace

	if(*p_s_str == '"') {
		while(strlen(++ p_s_str) && *p_s_str != '"')
			*p_s_dest_buffer ++ = *p_s_str;
		if(strlen(p_s_str))
			p_s_str ++; // poer 2. vozovky
		*p_s_dest_buffer = 0;

		return p_s_str;
	}
	// text v uvozovkach

	while(strlen(p_s_str) && !isspace(*p_s_str))
		*p_s_dest_buffer ++ = *p_s_str ++;
	*p_s_dest_buffer = 0;
	// text

	return p_s_str;
}

/*
 *	std::vector<char*> *TPath::p_ParseStrList(const char *p_s_list)
 *		- parse string, containing values, separated by whitespaces
 *		  (can group more of them together using quotation marks)
 *		- return vector of pointers to new allocated strinngs (i.e. must be freed)
 *		  containing values in order of their apperance in p_s_list
 *		  or 0 of there's not enough memory
 */
std::vector<char*> *TPath::p_ParseStrList(const char *p_s_list)
{
	__FuncGuard("TPath::p_ParseStrList");

	std::vector<char*> *p_string_list;
	char *p_s_buffer, *p_s_new_str;

	if(!(p_s_buffer = new char[strlen(p_s_list) + 1]))
		return 0;

	if(!(p_string_list = new std::vector<char*>)) {
		delete[] p_s_buffer;
		return 0;
	}

	while(p_s_list = p_s_ScanString(p_s_list, p_s_buffer)/*sscanf(p_s_list, "%s", p_s_buffer) != -1*/) { // t_odo - handle "strings with spaces in quotes"
		if(!(p_s_new_str = new char[strlen(p_s_buffer) + 1])) {
			delete[] p_s_buffer;
			for(size_t i = 0; i < p_string_list->size(); i ++)
				delete[] (*p_string_list)[i];
			delete p_string_list;

			return 0;
		}
		strcpy(p_s_new_str, p_s_buffer);

		p_string_list->push_back(p_s_new_str);

		while(strlen(p_s_list) && isspace(*p_s_list))
			p_s_list ++;
	}
	delete[] p_s_buffer;
	
	return p_string_list;
}

/*
 *	int TPath::b_IsSupportedByHW() const
 *		- return true if all required extensions are suported by target hw
 */
int TPath::b_IsSupportedByHW() const
{
	__FuncGuard("TPath::b_IsSupportedByHW");

	if(p_extension_list) {
		for(size_t i = 0; i < p_extension_list->size(); i ++) {
			if(!CGLExtensionHandler::b_SupportedExtension((*p_extension_list)[i]))
				return false;
		}
		// pokud nepodporuje njakou extension ...
	}

	return true;
}

/*
 *	int TPath::b_HasCorrectVendor(int b_exact) const
 *		- return true if vendor of target videocard matches one of vendors, required in path
 *		- vendor names are wildcat-able (i.e. "nv*" means "nv", followed by anything)
 *		- if b_exact is true, name "*" is forbidden
 *		- if vendor list is empty and b_exact is true, return value is false
 *		- if vendor list is empty and b_exact is false, return value is true
 */
int TPath::b_HasCorrectVendor(int b_exact) const
{
	__FuncGuard("TPath::b_HasCorrectVendor");

	const unsigned char *p_s_vendor = glGetString(GL_VENDOR);

	if(p_vendor_list) {
		for(size_t i = 0; i < p_vendor_list->size(); i ++) {
			if(!b_exact && !strcmp((*p_vendor_list)[i], "*"))
				return true;
			if(CWildcatStringComparer::b_wildcat_stricmp((*p_vendor_list)[i], (const char*)p_s_vendor))
				return true;
		}

		return false;
	}

	return !b_exact; // pokud je seznam przdn (nebo chyb), je tam hvzdika
}

/*
 *	int TPath::b_IsCorrectModel() const
 *		- return true if target videocard name with prefix "+" is found in model list
 *		- return false if prefix is "-" or if list doesn't contain target videocard
 */
int TPath::b_IsCorrectModel() const // karty jsou ve formtu "[+/-]card name" ("+GeForce 6800 GT")
{
	__FuncGuard("TPath::b_IsCorrectModel");

	const unsigned char *p_s_renderer = glGetString(GL_RENDERER);

	if(p_card_list) {
		for(size_t i = 0; i < p_card_list->size(); i ++) {
			if(CWildcatStringComparer::b_wildcat_stricmp((*p_card_list)[i] + 1, (const char*)p_s_renderer))
				return *(*p_card_list)[i] == '+';
		}
	}

	return false;
}

/*
 *	int TPath::b_IsIncorrectModel() const
 *		- return true if target videocard name with prefix "-" is found in model list
 *		- return false if prefix is "+" or if list doesn't contain target videocard
 */
int TPath::b_IsIncorrectModel() const // karty jsou ve formtu "[+/-]card name" ("+GeForce 6800 GT")
{
	__FuncGuard("TPath::b_IsIncorrectModel");

	const unsigned char *p_s_renderer = glGetString(GL_RENDERER);

	if(p_card_list) {
		for(size_t i = 0; i < p_card_list->size(); i ++) {
			if(CWildcatStringComparer::b_wildcat_stricmp((*p_card_list)[i] + 1, (const char*)p_s_renderer))
				return *(*p_card_list)[i] == '-';
		}
	}

	return false;
}

/*
 *	void TPath::Free()
 *		- function that delete all path data
 *		- must be called, since TPath doesn't have destructor
 */
void TPath::Free()
{
	__FuncGuard("TPath::Free");

	if(p_extension_list) {
		for(size_t i = 0; i < p_extension_list->size(); i ++)
			delete[] (*p_extension_list)[i];
		delete p_extension_list;
		p_extension_list = 0;
	}
	if(p_vendor_list) {
		for(size_t i = 0; i < p_vendor_list->size(); i ++)
			delete[] (*p_vendor_list)[i];
		delete p_vendor_list;
		p_vendor_list = 0;
	}
	if(p_card_list) {
		for(size_t i = 0; i < p_card_list->size(); i ++)
			delete[] (*p_card_list)[i];
		delete p_card_list;
		p_card_list = 0;
	}
	if(p_s_path_name)
		delete[] p_s_path_name;
	p_s_path_name = 0;
}

/*
 *	int TPath::b_ValidPath(int n_pass)
 *		- return true if path can be used on current hardware (opengl must be initialized)
 *		- pass is 0 or 1, in pass 0 only paths with exact vendor validates,
 *		  in pass 1 paths with vendor "*" = any validates
 */
int TPath::b_ValidPath(int n_pass)
{
	__FuncGuard("TPath::b_ValidPath");

	if(b_IsSupportedByHW() && ((b_HasCorrectVendor(n_pass == 0) &&
	   !b_IsIncorrectModel()) || b_IsCorrectModel()))
		return true;
	return false;
	// all required extensions must be supported
	// and either we have correct vendor and it's not incorrect model
	// (i.e. path can support all NVidia family except say Geforce 4 MX)
	// OR it's one of correct models (say Radeon 9500 Radeon 9700)
}

/*
 *								=== ~TPath ===
 */

/*
 *								=== CWildcatStringComparer ===
 */

int CWildcatStringComparer::strincmp(const char *p_s_str_a, const char *p_s_str_b, int n_len)
{
	for(size_t i = 0; i <= strlen(p_s_str_a) && i <= strlen(p_s_str_b) && i < size_t(n_len); i ++) {
		if(tolower(p_s_str_a[i]) > tolower(p_s_str_b[i]))
			return 1;
		else if(tolower(p_s_str_a[i]) < tolower(p_s_str_b[i]))
			return -1;
	}
	return 0;
}

const char *CWildcatStringComparer::strinstr(const char *p_str, const char *p_substr, int n_substr_len)
{
	_ASSERTE(p_str);
	_ASSERTE(p_substr);

	for(size_t i = 0; i <= strlen(p_str) - n_substr_len; i ++) {
		int b_difference = false;
		for(size_t j = i; j < i + n_substr_len; j ++) {
			if(tolower(p_str[j]) != tolower(p_substr[j - i])) {
				b_difference = true;
				break;
			}
		}
		if(!b_difference)
			return p_str + i;
	}
	return 0;
}

const char *CWildcatStringComparer::stristr(const char *p_str, const char *p_substr)
{
	return strinstr(p_str, p_substr, strlen(p_substr));
}

// wildcat-able stricmp
//
// case 0 "" -> return equal
// case 1 "*x" -> end of string must contain "x"
// case 2 "*x*" -> string must contain "x" and continue loop
// case 3 "*" -> return equal
// case 4 "x*" -> compare "x" and continue loop
// case 5 "x" -> return strcmp
int CWildcatStringComparer::b_wildcat_stricmp(const char *p_s_wildcat, const char *p_s_str)
{
	while(1) {
		if(!strlen(p_s_wildcat))
			return true;
		// case 0 "" -> return equal

		char *p_s_ptr;
		if(*p_s_wildcat == '*') {
			if(*(p_s_wildcat + 1)) {
				if(p_s_ptr = (char*)strchr(p_s_wildcat + 1, '*')) {
					// case 2 "*x*" -> string must contain "x" and continue loop
					if(p_s_str = (char*)strinstr(p_s_str, p_s_wildcat + 1, p_s_ptr - (p_s_wildcat + 1))) {
						p_s_str += p_s_ptr - (p_s_wildcat + 1);
						p_s_wildcat = p_s_ptr;

						continue;
					}
					// substring "x" found ...

					return false;
				} else {
					// case 1 "*x" -> end of string must contain "x"
					return stristr(p_s_str, p_s_wildcat + 1) != 0;
				}
			} else {
				// case 3 "*" -> return equal
				return true;
			}
		}

		if(p_s_ptr = (char*)strchr(p_s_wildcat, '*')) {
			// case 4 "x*" -> compare "x" and continue loop
			if(strincmp(p_s_wildcat, p_s_str, p_s_ptr - p_s_wildcat))
				return false;
			p_s_str += p_s_ptr - p_s_wildcat;
			p_s_wildcat = p_s_ptr;
		} else {
			return !_stricmp(p_s_wildcat, p_s_str);
			// case 5 "x" -> return strcmp
		}
	}
}

/*
 *								=== ~CWildcatStringComparer ===
 */

/*
 *		-end-of-file-
 */
