/*
								+---------------------------------+
								|                                 |
								|      ***   OpenGL2.0   ***      |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|         RenderBuffer.cpp        |
								|                                 |
								+---------------------------------+
*/

#include <crtdbg.h>
#include <windows.h>
#include <gl/gl.h>
#include <stdio.h>
#include <vector>

#include "glext.h"
#include "wglext.h"
#include "opengl20.h"
#include "glstate.h"

#include "renderbuffer.h"
#include "callstack.h"

/*
 *								=== CGLRenderBuffer ===
 */

/*
 *	CGLRenderBuffer::CGLRenderBuffer(int n_width, int n_height, int n_format, int n_color_bits,
 *		int b_float, int n_texture_target, int b_mipmap, int b_depth_texture_target,
 *		int b_depth_buffer, int n_depth_bits, int b_stencil_buffer, int n_stencil_bits)
 *		- default constructor, assign all parameters and call Create()
 *		- doesn't create m_p_state, it mas to be done in Create()
 *		- m_b_status is set to return value of Create(), it isn't supposed to change it
 */
CGLRenderBuffer::CGLRenderBuffer(int n_width, int n_height, int n_format, int n_color_bits,
	int b_float, int n_texture_target, int b_mipmap, int b_depth_texture_target,
	int b_depth_buffer, int n_depth_bits, int b_stencil_buffer, int n_stencil_bits)
	:m_p_state(0),
	m_n_width(n_width),
	m_n_height(n_height),
	m_n_format(n_format),
	m_n_color_bits(n_color_bits),
	m_b_float_buffer(b_float),
	m_b_depth_buffer(b_depth_buffer),
	m_n_depth_bits(n_depth_bits),
	m_b_stencil_buffer(b_stencil_buffer),
	m_n_stencil_bits(n_stencil_bits),
	m_n_texture_target(n_texture_target),
	m_b_mipmaps(b_mipmap),
	m_b_depth_texture(b_depth_texture_target),
	m_b_active(false),
	m_n_bound_texture(0)
{
	__FuncGuard("CGLRenderBuffer::CGLRenderBuffer");

	m_b_status = false;
}

/*
 *	int CGLRenderBuffer::b_Status() const
 *		- return buffer status; if it's true, it should be ready to use
 */
int CGLRenderBuffer::b_Status() const
{
	__FuncGuard("CGLRenderBuffer::b_Status");

	return m_b_status;
}

/*
 *	int CGLRenderBuffer::n_Width() const
 *		- return width of the buffer
 */
int CGLRenderBuffer::n_Width() const
{
	__FuncGuard("CGLRenderBuffer::n_Width");

	return m_n_width;
}

/*
 *	int CGLRenderBuffer::n_Height() const
 *		- return height of the buffer
 */
int CGLRenderBuffer::n_Height() const
{
	__FuncGuard("CGLRenderBuffer::n_Height");

	return m_n_height;
}

/*
 *	int CGLRenderBuffer::CopyTextureData(GLenum n_texture = GL_TEXTURE_2D)
 *		- copy texture data (work only in case we can't render directly to texture)
 *		- must be active!
 */
int CGLRenderBuffer::CopyTextureData(GLenum n_texture)
{
	__FuncGuard("CGLRenderBuffer::CopyTextureData");

    if(!m_b_active)
        return false;
	
	if(b_DirectRenderToTexture())
		return true;
	// no need to copy then ...

	switch(n_texture) {
	case GL_TEXTURE_1D:
		m_p_state->BindTexture1D(m_n_bound_texture);
		glCopyTexSubImage1D(n_texture, 0, 0, 0, 0, m_n_width);
		break;
	case GL_TEXTURE_2D:
	case GL_TEXTURE_RECTANGLE_ARB:
	case GL_TEXTURE_CUBE_MAP_POSITIVE_X:
	case GL_TEXTURE_CUBE_MAP_NEGATIVE_X:
	case GL_TEXTURE_CUBE_MAP_POSITIVE_Y:
	case GL_TEXTURE_CUBE_MAP_NEGATIVE_Y:
	case GL_TEXTURE_CUBE_MAP_POSITIVE_Z:
	case GL_TEXTURE_CUBE_MAP_NEGATIVE_Z:
		if(n_texture == GL_TEXTURE_2D)
			m_p_state->BindTexture2D(m_n_bound_texture);
		else if (n_texture==  GL_TEXTURE_RECTANGLE_ARB)
			m_p_state->BindTexture(GL_TEXTURE_RECTANGLE_ARB, m_n_bound_texture);
		else
			m_p_state->BindTexture(GL_TEXTURE_BINDING_CUBE_MAP_ARB, m_n_bound_texture);
		glCopyTexSubImage2D(n_texture, 0, 0, 0, 0, 0, m_n_width, m_n_height);
		break;
	default:
		return false;
	}
	// copy data (mip-map levels are updated automatically
	// via SGIS_GENERATE_MIPMAP or gl20 GenerateMipmap)

	return true;
}

/*
 *								=== ~CGLRenderBuffer ===
 */

/*
 *								=== CGLRenderBuffer_PBuffer ===
 */

PFNWGLGETPIXELFORMATATTRIBIVARBPROC CGLRenderBuffer_PBuffer::wglGetPixelFormatAttribivARB = 0;
PFNWGLCHOOSEPIXELFORMATARBPROC CGLRenderBuffer_PBuffer::wglChoosePixelFormatARB = 0;
PFNWGLCREATEPBUFFERARBPROC CGLRenderBuffer_PBuffer::wglCreatePbufferARB = 0;
PFNWGLGETPBUFFERDCARBPROC CGLRenderBuffer_PBuffer::wglGetPbufferDCARB = 0;
PFNWGLRELEASEPBUFFERDCARBPROC CGLRenderBuffer_PBuffer::wglReleasePbufferDCARB = 0;
PFNWGLDESTROYPBUFFERARBPROC CGLRenderBuffer_PBuffer::wglDestroyPbufferARB = 0;
PFNWGLQUERYPBUFFERARBPROC CGLRenderBuffer_PBuffer::wglQueryPbufferARB = 0;

PFNWGLBINDTEXIMAGEARBPROC CGLRenderBuffer_PBuffer::wglBindTexImageARB = 0;
PFNWGLRELEASETEXIMAGEARBPROC CGLRenderBuffer_PBuffer::wglReleaseTexImageARB = 0;

CGLRenderBuffer_PBuffer::CGLRenderBuffer_PBuffer(int n_width, int n_height, int n_format,
	int n_color_bits, int b_float, int n_texture_target, int b_mipmap, int b_depth_texture_target,
	int b_double_buffer, int b_depth_buffer, int n_depth_bits, int b_stencil_buffer,
	int n_stencil_bits)
	:CGLRenderBuffer(n_width, n_height, n_format, n_color_bits,
		b_float, n_texture_target, b_mipmap, b_depth_texture_target,
		b_depth_buffer, n_depth_bits, b_stencil_buffer, n_stencil_bits),
	m_b_double_buffer(b_double_buffer)
{
	__FuncGuard("CGLRenderBuffer_PBuffer::CGLRenderBuffer_PBuffer");

	m_b_status = Create();
}

int CGLRenderBuffer_PBuffer::b_Supported() const
{
	__FuncGuard("CGLRenderBuffer_PBuffer::b_Supported");

	int n_failed_functions = 0;

	if(!((PFNWGLGETPIXELFORMATATTRIBIVARBPROC)wglGetProcAddress("wglGetPixelFormatAttribivARB"))) n_failed_functions ++;
	if(!((PFNWGLCHOOSEPIXELFORMATARBPROC)wglGetProcAddress("wglChoosePixelFormatARB"))) n_failed_functions ++;
	if(!((PFNWGLCREATEPBUFFERARBPROC)wglGetProcAddress("wglCreatePbufferARB"))) n_failed_functions ++;
	if(!((PFNWGLGETPBUFFERDCARBPROC)wglGetProcAddress("wglGetPbufferDCARB"))) n_failed_functions ++;
	if(!((PFNWGLRELEASEPBUFFERDCARBPROC)wglGetProcAddress("wglReleasePbufferDCARB"))) n_failed_functions ++;
	if(!((PFNWGLDESTROYPBUFFERARBPROC)wglGetProcAddress("wglDestroyPbufferARB"))) n_failed_functions ++;
	if(!((PFNWGLQUERYPBUFFERARBPROC)wglGetProcAddress("wglQueryPbufferARB"))) n_failed_functions ++;

	if(!((PFNWGLBINDTEXIMAGEARBPROC)wglGetProcAddress("wglBindTexImageARB"))) n_failed_functions ++;
	if(!((PFNWGLRELEASETEXIMAGEARBPROC)wglGetProcAddress("wglReleaseTexImageARB"))) n_failed_functions ++;

	return CGLExtensionHandler::b_SupportedWGLExtension("WGL_ARB_pbuffer") &&
		CGLExtensionHandler::b_SupportedWGLExtension("WGL_ARB_pixel_format") && !n_failed_functions;
}

/*
 *	int CGLRenderBuffer_PBuffer::n_GetWGLFunctionPointers()
 *		- get pointers to wgl-arb functions, used for handling p-buffers
 *		- return number of functions we didn't found (i.e. 0 = success)
 */
int CGLRenderBuffer_PBuffer::n_GetWGLFunctionPointers()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::n_GetWGLFunctionPointers");

	int n_failed_functions = 0;

	if(!(wglGetPixelFormatAttribivARB = (PFNWGLGETPIXELFORMATATTRIBIVARBPROC)wglGetProcAddress("wglGetPixelFormatAttribivARB"))) n_failed_functions ++;
	if(!(wglChoosePixelFormatARB = (PFNWGLCHOOSEPIXELFORMATARBPROC)wglGetProcAddress("wglChoosePixelFormatARB"))) n_failed_functions ++;
	if(!(wglCreatePbufferARB = (PFNWGLCREATEPBUFFERARBPROC)wglGetProcAddress("wglCreatePbufferARB"))) n_failed_functions ++;
	if(!(wglGetPbufferDCARB = (PFNWGLGETPBUFFERDCARBPROC)wglGetProcAddress("wglGetPbufferDCARB"))) n_failed_functions ++;
	if(!(wglReleasePbufferDCARB = (PFNWGLRELEASEPBUFFERDCARBPROC)wglGetProcAddress("wglReleasePbufferDCARB"))) n_failed_functions ++;
	if(!(wglDestroyPbufferARB = (PFNWGLDESTROYPBUFFERARBPROC)wglGetProcAddress("wglDestroyPbufferARB"))) n_failed_functions ++;
	if(!(wglQueryPbufferARB = (PFNWGLQUERYPBUFFERARBPROC)wglGetProcAddress("wglQueryPbufferARB"))) n_failed_functions ++;

	if(!(wglBindTexImageARB = (PFNWGLBINDTEXIMAGEARBPROC)wglGetProcAddress("wglBindTexImageARB"))) n_failed_functions ++;
	if(!(wglReleaseTexImageARB = (PFNWGLRELEASETEXIMAGEARBPROC)wglGetProcAddress("wglReleaseTexImageARB"))) n_failed_functions ++;

	return n_failed_functions;
}

enum {
	_Max_Attribs = 256,
	_Max_PFormats = 256
};

/*
 *	int CGLRenderBuffer_PBuffer::Create()
 *		- create p-buffer, according to parameters, passed to constructor
 *		- return true if p-buffer was created succesfuly, otherwise return false
 */
int CGLRenderBuffer_PBuffer::Create()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::Create");

	if(!b_Supported() || n_GetWGLFunctionPointers())
		return false;

	int n_attributes[2 * _Max_Attribs];
	int n_i_attribs = 0;

	for(int i = 0; i < 2 * _Max_Attribs; i ++)
		n_attributes[i] = 0;

	n_attributes[n_i_attribs] = WGL_DRAW_TO_PBUFFER_ARB;
	n_attributes[n_i_attribs + 1] = true;
	n_i_attribs += 2;
	// we want to draw to p-buffer

	n_attributes[n_i_attribs] = WGL_SUPPORT_OPENGL_ARB;
	n_attributes[n_i_attribs + 1] = true;
	n_i_attribs += 2;
	// support open-gl

	int n_pixel_type = WGL_TYPE_RGBA_ARB;
	if(m_b_float_buffer) {
		if(CGLExtensionHandler::b_SupportedWGLExtension("WGL_ATI_pixel_format_float"))
			n_pixel_type = WGL_TYPE_RGBA_FLOAT_ATI;
		else if(CGLExtensionHandler::b_SupportedWGLExtension("WGL_NV_float_buffer")) {
			n_attributes[n_i_attribs] = WGL_FLOAT_COMPONENTS_NV;
			n_attributes[n_i_attribs + 1] = true;
			n_i_attribs += 2;
		} else
			return false;
	}
	//
	n_attributes[n_i_attribs] = WGL_PIXEL_TYPE_ARB;
	n_attributes[n_i_attribs + 1] = n_pixel_type;
	n_i_attribs += 2;
	// set pixel type to int / float

	int n_components;
	switch(m_n_format) {
	case GL_RGB:
		n_components = 3;
		break;
	case GL_RGBA:
	default:
		n_components = 4;
		break;
	case GL_INTENSITY:
	case GL_LUMINANCE:
	case GL_ALPHA:
		n_components = 1;
		break;
	case GL_LUMINANCE_ALPHA:
		n_components = 2;
		break;
	}

	if(m_b_depth_buffer) {
		n_attributes[n_i_attribs] = WGL_DEPTH_BITS_ARB;
		n_attributes[n_i_attribs + 1] = m_n_depth_bits;
		n_i_attribs += 2;
	}
	if(m_b_stencil_buffer) {
		n_attributes[n_i_attribs] = WGL_STENCIL_BITS_ARB;
		n_attributes[n_i_attribs + 1] = m_n_stencil_bits;
		n_i_attribs += 2;
	}
	// depth + stencil buffer setup

	if(m_b_double_buffer) {
		n_attributes[n_i_attribs] = WGL_DOUBLE_BUFFER_ARB;
		n_attributes[n_i_attribs + 1] = true;
		n_i_attribs += 2;
	}
	// double buffering setup

	if(!CGLExtensionHandler::b_SupportedWGLExtension("WGL_ARB_render_texture"))
		m_n_texture_target = 0;
	// only if supported

	if(m_n_texture_target) {
		n_attributes[n_i_attribs] = WGL_TEXTURE_TARGET_ARB;
		switch(m_n_texture_target) {
		case GL_TEXTURE_1D:
			n_attributes[n_i_attribs + 1] = WGL_TEXTURE_1D_ARB;
			break;
		case GL_TEXTURE_2D:
			n_attributes[n_i_attribs + 1] = WGL_TEXTURE_2D_ARB;
			break;
		case GL_TEXTURE_CUBE_MAP:
			n_attributes[n_i_attribs + 1] = WGL_TEXTURE_CUBE_MAP_ARB;
			break;
		case GL_TEXTURE_RECTANGLE_NV: // == GL_TEXTURE_RECTANGLE_ARB
			n_attributes[n_i_attribs + 1] = WGL_TEXTURE_RECTANGLE_NV;
			break;
		default:
			return false; // invalid texture target
		}
		n_i_attribs += 2;
		// select texture type ...

		if(m_b_float_buffer &&
		   CGLExtensionHandler::b_SupportedWGLExtension("WGL_NV_float_buffer")) { // ati-float doesn't need this ?
			switch(n_components) {
			case 1:
				n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_RECTANGLE_FLOAT_R_NV;
				n_attributes[n_i_attribs + 1] = true;
				n_i_attribs += 2;
				//
				n_attributes[n_i_attribs] = WGL_TEXTURE_FORMAT_ARB;
				n_attributes[n_i_attribs + 1] = WGL_TEXTURE_FLOAT_R_NV;
				n_i_attribs += 2;
				break;
			case 2:
				n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_RECTANGLE_FLOAT_RG_NV;
				n_attributes[n_i_attribs + 1] = true;
				n_i_attribs += 2;
				//
				n_attributes[n_i_attribs] = WGL_TEXTURE_FORMAT_ARB;
				n_attributes[n_i_attribs + 1] = WGL_TEXTURE_FLOAT_RG_NV;
				n_i_attribs += 2;
				break;
			case 3:
				n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_RECTANGLE_FLOAT_RGB_NV;
				n_attributes[n_i_attribs + 1] = true;
				n_i_attribs += 2;
				//
				n_attributes[n_i_attribs] = WGL_TEXTURE_FORMAT_ARB;
				n_attributes[n_i_attribs + 1] = WGL_TEXTURE_FLOAT_RGB_NV;
				n_i_attribs += 2;
				break;
			case 4:
				n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_RECTANGLE_FLOAT_RGBA_NV;
				n_attributes[n_i_attribs + 1] = true;
				n_i_attribs += 2;
				//
				n_attributes[n_i_attribs] = WGL_TEXTURE_FORMAT_ARB;
				n_attributes[n_i_attribs + 1] = WGL_TEXTURE_FLOAT_RGBA_NV;
				n_i_attribs += 2;
				break;
			default:
				return false; // bad components selected
			}
		} else {
			switch(n_components) {
			case 3:
				n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_RGB_ARB;
				n_attributes[n_i_attribs + 1] = true;
				n_i_attribs += 2;
				//
				n_attributes[n_i_attribs] = WGL_TEXTURE_FORMAT_ARB;
				n_attributes[n_i_attribs + 1] = WGL_TEXTURE_RGB_ARB;
				n_i_attribs += 2;
				break;
			case 4:
				n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_RGBA_ARB;
				n_attributes[n_i_attribs + 1] = true;
				n_i_attribs += 2;
				//
				n_attributes[n_i_attribs] = WGL_TEXTURE_FORMAT_ARB;
				n_attributes[n_i_attribs + 1] = WGL_TEXTURE_RGBA_ARB;
				n_i_attribs += 2;
				break;
			default:
				return false; // bad components selected
			}
		}
		// select texture format
	}
	// render_texture extension setup (if requested and available)

	if(m_b_depth_texture) {
		n_attributes[n_i_attribs] = WGL_BIND_TO_TEXTURE_DEPTH_NV;
		n_attributes[n_i_attribs + 1] = true;
		n_i_attribs += 2;
		//
		n_attributes[n_i_attribs] = WGL_DEPTH_TEXTURE_FORMAT_NV;
		n_attributes[n_i_attribs + 1] = WGL_TEXTURE_DEPTH_COMPONENT_NV;
		n_i_attribs += 2;
	}
	// depth texture ...

	if(m_b_mipmaps && m_n_texture_target) {
		n_attributes[n_i_attribs] = WGL_MIPMAP_TEXTURE_ARB;
		n_attributes[n_i_attribs + 1] = true;
		n_i_attribs += 2;
	}
	// texture target setup

	n_attributes[n_i_attribs] = WGL_RED_BITS_ARB;
	n_attributes[n_i_attribs + 1] = (n_components > 0)? m_n_color_bits : 0;
	n_i_attribs += 2;
	n_attributes[n_i_attribs] = WGL_GREEN_BITS_ARB;
	n_attributes[n_i_attribs + 1] = (n_components > 1)? m_n_color_bits : 0;
	n_i_attribs += 2;
	n_attributes[n_i_attribs] = WGL_BLUE_BITS_ARB;
	n_attributes[n_i_attribs + 1] = (n_components > 2)? m_n_color_bits : 0;
	n_i_attribs += 2;
	n_attributes[n_i_attribs] = WGL_ALPHA_BITS_ARB;
	n_attributes[n_i_attribs + 1] = (n_components > 3)? m_n_color_bits : 0;
	n_i_attribs += 2;
	// color buffer attribs

	HGLRC h_glrc;
	HDC h_dc;
	h_dc = wglGetCurrentDC();
	h_glrc = wglGetCurrentContext();
	// get device contexts

	unsigned int n_format_num;
	int p_format[_Max_PFormats];
	//
	float f_attrib[1] = {0};
	if(!wglChoosePixelFormatARB(h_dc, n_attributes, f_attrib, _Max_PFormats, p_format, &n_format_num))
		return false;
	if(n_format_num <= 0)
		return false;
	int n_format = p_format[1];
	// choose pixel format ...

	n_attributes[0] = 0;
	if(!(m_h_buffer = wglCreatePbufferARB(h_dc, n_format, m_n_width, m_n_height, n_attributes)))
		return false;
	// create the p-buffer.

	if(!(m_h_dc = wglGetPbufferDCARB(m_h_buffer)))
		return false;
	// get the device context.

	if(!(m_h_glrc = wglCreateContext(m_h_dc)))
		return false;
	// create a gl context for the p-buffer.

	//if(b_share) {
		if(!wglShareLists(h_glrc, m_h_glrc))
			return false;
	//}
	// share textures, etc ...

	wglQueryPbufferARB(m_h_buffer, WGL_PBUFFER_WIDTH_ARB, &m_n_width);
	wglQueryPbufferARB(m_h_buffer, WGL_PBUFFER_HEIGHT_ARB, &m_n_height);

#ifdef _DEBUG
    // query pixel format
    int iattributes[] = {
      WGL_RED_BITS_ARB,
      WGL_GREEN_BITS_ARB,
      WGL_BLUE_BITS_ARB,
      WGL_ALPHA_BITS_ARB,
      WGL_FLOAT_COMPONENTS_NV,
      WGL_DEPTH_BITS_ARB,
      WGL_SAMPLES_EXT,
      WGL_AUX_BUFFERS_ARB
    };
    int ivalues[sizeof(iattributes) / sizeof(int)];

    if(wglGetPixelFormatAttribivARB(m_h_dc, n_format, 0, sizeof(iattributes) / sizeof(int), iattributes, ivalues)) {
		char p_s_buffer[256];
		sprintf(p_s_buffer, "r:%d g:%d b:%d a:%d float:%d depth:%d samples:%d aux:%d\n",
			ivalues[0], ivalues[1], ivalues[2], ivalues[3], ivalues[4], ivalues[5], ivalues[6], ivalues[7]);
		//MessageBox(0, p_s_buffer, "PBuffer format", MB_OK);
    }
#endif

	return true;
}

/*
 *	int CGLRenderBuffer_PBuffer::HandleModeSwitch()
 *		- p-buffer can be lost in mode switch, this tries to restore it
 *		- return true if p-buffer is operational
 */
int CGLRenderBuffer_PBuffer::HandleModeSwitch()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::HandleModeSwitch");

	int b_lost = false;
    wglQueryPbufferARB(m_h_buffer, WGL_PBUFFER_LOST_ARB, &b_lost);
    if(b_lost) {
        Destroy();
        m_b_status = Create();
    }

	return m_b_status;
}

/*
 *	int CGLRenderBuffer_PBuffer::Destroy()
 *		- destroy p-buffer
 *		- always return true
 */
int CGLRenderBuffer_PBuffer::Destroy()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::Destroy");

	if(m_h_buffer) {
        wglDeleteContext(m_h_glrc);
        wglReleasePbufferDCARB(m_h_buffer, m_h_dc);
        wglDestroyPbufferARB(m_h_buffer);
		m_h_buffer = 0;
    }

	if(m_p_state) {
		delete m_p_state;
		m_p_state = 0;
	}

	m_b_status = false;

	return true;
}

/*
 *	int CGLRenderBuffer_PBuffer::BindRenderTexture(unsigned int n_texture)
 *		- set texture, we're going to render to
 *		- return false if p-buffer is not ready and active or we already bound texture
 *		  (then we have to release it first)
 */
int CGLRenderBuffer_PBuffer::BindRenderTexture(unsigned int n_texture)
{
	__FuncGuard("CGLRenderBuffer_PBuffer::BindRenderTexture");

	if(!m_b_active)
		return false;
	if(m_n_bound_texture == n_texture)
		return true;
	if(m_n_bound_texture && !ReleaseRenderTexture())
		return false;

	if(m_n_texture_target || m_b_depth_texture) {
		if(!wglBindTexImageARB(m_h_buffer, n_texture))
			return false;
	}
	// bind only if supported

	m_n_bound_texture = n_texture;
	// save bound texture id

	return true;
}

/*
 *	int CGLRenderBuffer_PBuffer::ReleaseRenderTexture()
 *		- release texture, we've been rendering to
 */
int CGLRenderBuffer_PBuffer::ReleaseRenderTexture()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::ReleaseRenderTexture");

	if((!m_n_bound_texture && m_n_texture_target) || !m_b_active)
		return false;

	if(m_n_texture_target || m_b_depth_texture) {
		if(!wglReleaseTexImageARB(m_h_buffer, m_n_bound_texture))
			return false;
	}
	// release only if supported

	m_n_bound_texture = 0;
	// clear bound texture id

	return true;
}

/*
 *	int CGLRenderBuffer_PBuffer::Bind()
 *		- make p-buffer active
 */
int CGLRenderBuffer_PBuffer::Bind()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::Bind");

	if(!m_b_status)
		return false;

	if(!m_b_active) {
		m_h_prev_glrc = wglGetCurrentContext();
		m_h_prev_dc = wglGetCurrentDC();

		if(!wglMakeCurrent(m_h_dc, m_h_glrc))
			return false;
	}

	if(!m_p_state) {
		if(!(m_p_state = new CGLState(m_h_dc, m_h_glrc)))
			return false;
		m_p_state->Update_States();
	}
	// create a new state

    m_b_active = true;

	return true;
}

/*
 *	int CGLRenderBuffer_PBuffer::Release()
 *		- deactivate p-buffer
 */
int CGLRenderBuffer_PBuffer::Release()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::Release");

    if(!m_b_active)
        return false;
    if(!wglMakeCurrent(m_h_prev_dc, m_h_prev_glrc))
        return false;

    m_h_prev_glrc = 0;
    m_h_prev_dc = 0;

    m_b_active = false;

	return true;
}

/*
 *	int CGLRenderBuffer_PBuffer::b_DirectRenderToTexture() const
 *		- return true if texture is bound to rendering
 *		- false means that texture data has to be copied after rendering
 *		- return value is only valid if it's ready
 */
int CGLRenderBuffer_PBuffer::b_DirectRenderToTexture() const
{
	__FuncGuard("CGLRenderBuffer_PBuffer::b_DirectRenderToTexture");

	return m_n_texture_target;
}

/*
 *	CGLState *CGLRenderBuffer_PBuffer::p_State()
 *		- return open-gl state, render-buffer uses
 */
CGLState *CGLRenderBuffer_PBuffer::p_State()
{
	__FuncGuard("CGLRenderBuffer_PBuffer::p_State");

	if(!m_b_status)
		return 0;
	return m_p_state;
}

/*
 *								=== ~CGLRenderBuffer_PBuffer ===
 */

/*
 *								=== CGLRenderBuffer_FBO ===
 */

CGLRenderBuffer_FBO::CGLRenderBuffer_FBO(int n_width, int n_height, int n_format,
	int n_color_bits, int b_float, int n_texture_target, int b_mipmap, int b_depth_texture_target,
	int b_double_buffer, int b_depth_buffer, int n_depth_bits, int b_stencil_buffer,
	int n_stencil_bits)
	:CGLRenderBuffer(n_width, n_height, n_format, n_color_bits,
		b_float, n_texture_target, b_mipmap, b_depth_texture_target,
		b_depth_buffer, n_depth_bits, b_stencil_buffer, n_stencil_bits)
{
	__FuncGuard("CGLRenderBuffer_FBO::CGLRenderBuffer_FBO");

	m_b_status = Create();
}

/*
 *	int CGLRenderBuffer_FBO::b_Supported() const
 *		- return true if extension is supported
 */
int CGLRenderBuffer_FBO::b_Supported() const
{
	__FuncGuard("CGLRenderBuffer_FBO::b_Supported");

	return CGLExtensionHandler::b_SupportedExtension("GL_EXT_framebuffer_object") &&
		!CGLExtensionHandler::n_GetFramebufferObjectEXTFuncPointers();
}

/*
 *	int CGLRenderBuffer_FBO::HandleModeSwitch()
 *		- always return true, since fbo-s doesn't suffer from being lost in mode switch (fixme)
 */
int CGLRenderBuffer_FBO::HandleModeSwitch()
{
	__FuncGuard("CGLRenderBuffer_FBO::HandleModeSwitch");

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::Create()
 *		- create new FBO, according to params, passed to constructor
 *		- return true in case of success, otherwise false
 */
int CGLRenderBuffer_FBO::Create()
{
	__FuncGuard("CGLRenderBuffer_FBO::Create");

	glGenFramebuffersEXT(1, &m_n_framebuffer);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, m_n_framebuffer);
	if(glGetError() != GL_NO_ERROR)
		return false;

	if(!m_n_texture_target) {
		glGenRenderbuffersEXT(1, &m_n_color_rb);
		glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, m_n_color_rb);

		GLenum n_format_array[][4] = {
			{GL_ALPHA4, GL_ALPHA8, GL_ALPHA16, GL_ALPHA16},
			{GL_LUMINANCE4, GL_LUMINANCE8, GL_LUMINANCE16, GL_LUMINANCE16},
			{GL_LUMINANCE4_ALPHA4, GL_LUMINANCE8_ALPHA8, GL_LUMINANCE16_ALPHA16, GL_LUMINANCE16_ALPHA16},
			{GL_INTENSITY4, GL_INTENSITY8, GL_INTENSITY16, GL_INTENSITY16},
			{GL_RGB4, GL_RGB8, GL_RGB16, GL_RGB16},
			{GL_RGBA4, GL_RGBA8, GL_RGBA16, GL_RGBA16},
		};
		GLenum n_float_format_array[][4] = {
			{GL_ALPHA16F_ARB, GL_ALPHA16F_ARB, GL_ALPHA16F_ARB, GL_ALPHA32F_ARB},
			{GL_LUMINANCE16F_ARB, GL_LUMINANCE16F_ARB, GL_LUMINANCE16F_ARB, GL_LUMINANCE32F_ARB},
			{GL_LUMINANCE_ALPHA16F_ARB, GL_LUMINANCE_ALPHA16F_ARB, GL_LUMINANCE_ALPHA16F_ARB, GL_LUMINANCE_ALPHA32F_ARB},
			{GL_INTENSITY16F_ARB, GL_INTENSITY16F_ARB, GL_INTENSITY16F_ARB, GL_INTENSITY32F_ARB},
			{GL_RGB16F_ARB, GL_RGB16F_ARB, GL_RGB16F_ARB, GL_RGB32F_ARB},
			{GL_RGBA16F_ARB, GL_RGBA16F_ARB, GL_RGBA16F_ARB, GL_RGBA32F_ARB},
		};
		//
		int n_col_index;
		switch(m_n_color_bits) {
		case 4:
			if(m_b_float_buffer) {
				m_n_color_bits = 16;
				n_col_index = 2;
			} else
				n_col_index = 0;
			break;
		case 8:
			if(m_b_float_buffer) {
				m_n_color_bits = 16;
				n_col_index = 2;
			} else
				n_col_index = 1;
			break;
		case 16:
			m_n_color_bits = 16;
			n_col_index = 2;
			break;
		case 32:
			n_col_index = 3;
			break;
		default:
			if(m_b_float_buffer) {
				m_n_color_bits = 16; // fixme - 16 or 32 ?
				n_col_index = 2;
			} else {
				m_n_color_bits = 8;
				n_col_index = 1;
			}
			break;
		}
		//
		int n_color_format;
		switch(m_n_format) {
		case GL_ALPHA:
			n_color_format = ((m_b_float_buffer)? n_float_format_array : n_format_array)[0][n_col_index];
			break;
		case GL_LUMINANCE:
			n_color_format = ((m_b_float_buffer)? n_float_format_array : n_format_array)[1][n_col_index];
			break;
		case GL_LUMINANCE_ALPHA:
			n_color_format = ((m_b_float_buffer)? n_float_format_array : n_format_array)[2][n_col_index];
			break;
		case GL_INTENSITY:
			n_color_format = ((m_b_float_buffer)? n_float_format_array : n_format_array)[3][n_col_index];
			break;
		case GL_RGB:
			n_color_format = ((m_b_float_buffer)? n_float_format_array : n_format_array)[4][n_col_index];
			break;
		default:
		case GL_RGBA:
			n_color_format = ((m_b_float_buffer)? n_float_format_array : n_format_array)[5][n_col_index];
			break;
		}
		// get color format

		glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, n_color_format, m_n_width, m_n_height);
		glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
			GL_COLOR_ATTACHMENT0_EXT, GL_RENDERBUFFER_EXT, m_n_color_rb);
	}
	if(glGetError() != GL_NO_ERROR)
		return false;
	// color (only in case texture is not going to be bound)

	if(m_b_depth_buffer && !m_b_depth_texture) {
		glGenRenderbuffersEXT(1, &m_n_depth_rb);
		glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, m_n_depth_rb);
		int n_depth_format;
		switch(m_n_depth_bits) {
		case 16:
			n_depth_format = GL_DEPTH_COMPONENT16;
			break;
		case 24:
			n_depth_format = GL_DEPTH_COMPONENT24;
			break;
		case 32:
		default:
			m_n_depth_bits = 32;
			n_depth_format = GL_DEPTH_COMPONENT32;
			break;
		}
		glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, n_depth_format, m_n_width, m_n_height);
		glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
			GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, m_n_depth_rb);
	}
	if(glGetError() != GL_NO_ERROR)
		return false;
	// depth (only if depth texture is not going to be bound)

	if(m_b_stencil_buffer) {
		glGenRenderbuffersEXT(1, &m_n_stencil_rb);
		int n_stencil_format;
		switch(m_n_stencil_bits) {
		case 1:
			n_stencil_format = GL_STENCIL_INDEX1_EXT;
			break;
		case 4:
			n_stencil_format = GL_STENCIL_INDEX4_EXT;
			break;
		case 8:
			n_stencil_format = GL_STENCIL_INDEX8_EXT;
			break;
		case 16:
			n_stencil_format = GL_STENCIL_INDEX16_EXT;
			break;
		default:
			n_stencil_format = GL_STENCIL_INDEX_EXT;
			break;
		}
		glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, n_stencil_format, m_n_width, m_n_height);
		glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT,
			GL_STENCIL_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, m_n_stencil_rb);
	}
	if(glGetError() != GL_NO_ERROR)
		return false;
	// stencil

	// create buffers ...

	if(!(m_p_state = new CGLState(wglGetCurrentDC(), wglGetCurrentContext()))) {
		Destroy();
		return false;
	}
	m_p_state->Update_States();
	// create state-tracking object

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	m_b_active = false;

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::Destroy()
 *		- free open-gl resources
 */
int CGLRenderBuffer_FBO::Destroy()
{
	__FuncGuard("CGLRenderBuffer_FBO::Destroy");

	if(!m_b_status)
		return false;

	if(!m_n_texture_target)
		glDeleteRenderbuffersEXT(1, &m_n_color_rb);
	if(m_b_depth_buffer && !m_b_depth_texture)
		glDeleteRenderbuffersEXT(1, &m_n_depth_rb);
	if(m_b_stencil_buffer)
		glDeleteRenderbuffersEXT(1, &m_n_stencil_rb);
	glDeleteFramebuffersEXT(1, &m_n_framebuffer);

	m_b_status = false;

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::Bind()
 *		- set framebuffer active
 *		- can be called only if it's ready and inactive, otherwise it returns false
 */
int CGLRenderBuffer_FBO::Bind()
{
	__FuncGuard("CGLRenderBuffer_FBO::Bind");

	if(!m_b_status)
		return false;

	if(m_b_active)
		return true;

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, m_n_framebuffer);

	m_b_active = true;

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::Release()
 *		- set framebuffer inactive
 *		- can be called only if it's ready and active, otherwise it returns false
 */
int CGLRenderBuffer_FBO::Release()
{
	__FuncGuard("CGLRenderBuffer_FBO::Release");

	if(!m_b_status || !m_b_active)
		return false;

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	m_b_active = false;

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::BindRenderTexture(unsigned int n_texture)
 *		- bind color texture
 *		- can be called only if it's ready and active and no texture is bound
 *		  (then return value is true; otherwise false)
 */
int CGLRenderBuffer_FBO::BindRenderTexture(unsigned int n_texture)
{
	__FuncGuard("CGLRenderBuffer_FBO::BindRenderTexture");

	if(!m_b_status || !m_b_active)
		return false;
	if(m_n_bound_texture == n_texture)
		return true;
	if(m_n_bound_texture && !ReleaseRenderTexture())
		return false;

	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,
                              GL_COLOR_ATTACHMENT0_EXT,
                              GL_TEXTURE_2D, n_texture, 0);
	if(glGetError() != GL_NO_ERROR)
		return false;

	m_n_bound_texture = n_texture;

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::ReleaseRenderTexture()
 *		- release texture, being rendered
 *		- must be ready, active and there must be some valid texture bound to it
 */
int CGLRenderBuffer_FBO::ReleaseRenderTexture()
{
	__FuncGuard("CGLRenderBuffer_FBO::ReleaseRenderTexture");

	if(!m_b_status || !m_b_active || !m_n_bound_texture)
		return false;

	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT,
                              GL_COLOR_ATTACHMENT0_EXT,
                              GL_TEXTURE_2D, 0, 0);

	m_n_bound_texture = 0;

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::b_DirectRenderToTexture() const
 *		- return true if we are able to render directly into texture (always for FBO-s)
 */
int CGLRenderBuffer_FBO::b_DirectRenderToTexture() const
{
	__FuncGuard("CGLRenderBuffer_FBO::b_DirectRenderToTexture");

	return true;
}

/*
 *	int CGLRenderBuffer_FBO::b_Status() const
 *		- return framebuffer status
 *		- if called when it's active, textures must be bound in order to work correctly
 *		  (checks framebuffer object completeness)
 */
int CGLRenderBuffer_FBO::b_Status() const
{
	__FuncGuard("CGLRenderBuffer_FBO::b_Status");

	return m_b_status && (!m_b_active ||
		glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT) == GL_FRAMEBUFFER_COMPLETE_EXT);
}

/*
 *	CGLState *CGLRenderBuffer_FBO::p_State()
 *		- return render-state
 */
CGLState *CGLRenderBuffer_FBO::p_State()
{
	__FuncGuard("CGLRenderBuffer_FBO::p_State");

	return m_p_state;
}

/*
 *								=== ~CGLRenderBuffer_FBO ===
 */

/*
 *		-end-of-file-
 */
