/*****************************************************************************/
/*                       Image processing functions                          */
/*  Source file : color conversion functions                                 */
/*  Designed by Igor Potucek (potucek@fit.vutbr.cz)                          */
/*  created 2005                                                             */
/*****************************************************************************/

#include <math.h>
#include <stdio.h>
#include "gausscolor.h"

#define max(a,b) ((a)>(b))?(a):(b)

// define minimal value of saturation part -> hardly defined color
const float MIN_SATURATION=0.1;                     // ideal 10% of range
// define minimal value of intensity part -> dark hardly defined colors
const float MIN_VALUE=0.1;                          // ideal 10% of range
// count of bins in histogram for each part - altogether BINS_COUNT x BINS_COUNT
const int BINS_COUNT = 100;                     // ideal value 20-30


void c_ColorDetect::InitBackground(ImageStruct *Input)
{
   if (Background!=NULL)
     delete Background;
   Background = new TBackground(Input->XSize, Input->YSize, 1);
   Background->update(Input);
   Background->update(Input);   
   Count = 0;
}
void c_ColorDetect::AddColor(unsigned char r,unsigned char g,unsigned char b)
{
  s_ColorRGB ColorRGB;
  ColorRGB.r=r; ColorRGB.g=g; ColorRGB.b=b;
  ColorRGBList.push_back(ColorRGB);
}

void c_ColorDetect::ConverColor(unsigned char r,unsigned char g,unsigned char b, int &c1, int &c2)
{
   int sum = r+g+b;
   if (sum>0)
   {
     c1=(BINS_COUNT*r)/sum + 0.5;
     c2=(BINS_COUNT*g)/sum + 0.5;
   }
   else
   {
     c1=0;
     c2=0;
   }
}

void c_ColorDetect::ConvertColor(unsigned char r,unsigned char g,unsigned char b, float &c1, float &c2)
{
   int sum = r+g+b;
   if (sum>0)
   {
     c1=(float)r/sum;
     c2=(float)g/sum;
   }
   else
   {
     c1=0;
     c2=0;
   }
}


void c_ColorDetectRG::ConverColor(unsigned char r,unsigned char g,unsigned char b, int &c1, int &c2)
{
   int sum = r+g+b;
   if (sum>0)
   {
     c1=(BINS_COUNT*r)/sum + 0.5;
     c2=(BINS_COUNT*g)/sum + 0.5;
   }
   else
   {
     c1=0;
     c2=0;
   }
}

void nc_ColorDetectHSV::ConverColor(unsigned char r,unsigned char g,unsigned char b, int &c1, int &c2)
{
  int min, max, delta;
  int i, w = 0;
  min = max = r;
  if (g<min) min=g;
  if (b<min) min=b;
  if (g>max) { max=g; w=1; }
  if (b>max) { max=b; w=2; }

  /* calculate V */
  int value = BINS_COUNT*max/255;
  if (value<MIN_VALUE)
  {
     c1=0; c2=0; value=0;
     return;
  }
  /* calculate S */
  c2 = (BINS_COUNT*(max-min))/max;

  /* calculate H */
  delta = max-min;
  if (delta==0)
  {
    c1=0; c2=0;
    return ;
  }
  switch(w) {
  case 0: c1 = BINS_COUNT*(0.0 + (g-b)/(float)delta)/6+0.5; break;
  case 1: c1 = BINS_COUNT*(2.0 + (b-r)/(float)delta)/6+0.5; break;
  case 2: c1 = BINS_COUNT*(4.0 + (r-g)/(float)delta)/6+0.5; break;
  }
  while (c1 < 0) c1 += BINS_COUNT;

}

int c_ColorDetect::SaveColors(char *FileName)
{
  FILE *f;
  if ((f=fopen(FileName,"w"))!=NULL)
   {
     t_ColorRGBList::iterator bi = ColorRGBList.begin(), ei = ColorRGBList.end();
     for (; bi != ei; ++bi)
     {
        fprintf(f,"%d %d %d\n",bi->r,bi->g,bi->b);
     }
     fclose(f);
     return 1;
   }
   else
   return 0;
}

int c_ColorDetect::LoadColors(char *FileName)
{
  FILE *f;
  if ((f=fopen(FileName,"r"))!=NULL)
   {
     int r,g,b;
     ColorRGBList.clear();
     while (fscanf(f, "%d %d %d", &r,&g,&b)!=EOF)
     {
        AddColor(r,g,b);
     };
     fclose(f);
     return 1;
   }
   else
   return 0;
}

double c_ColorDetect::IsSkin(unsigned char r,unsigned char g,unsigned char b)
{
   float c1, c2;
   ConvertColor(r,g,b,c1,c2);
   float a1=c1-avg1;
   float a2=c2-avg2;
   float exponent;                     // exponent
   float prob;                         // pravdepodobnost
   // vypocet exponentu
   exponent=-(a1*(a1*inv11+a2*inv21)+a2*(a1*inv12+a2*inv22))/2;
   // vypocet pravdepodobnosti

   // prob=exp(exponent)/(2*M_PI*sqrt(det));
   prob=exp(exponent);
   return prob;
}

void c_ColorDetect::Compute()
{
 // vypocet stredni hodnoty barvy
 avg1=0;
 avg2=0;
 kov11=0;   kov21=0;
 kov12=0;   kov22=0;
 float a,b;
 if (ColorRGBList.size()==0)
   return;
 t_ColorRGBList::iterator bi = ColorRGBList.begin(), ei = ColorRGBList.end();
 for (; bi != ei; ++bi)
 {
    ConvertColor(bi->r, bi->g, bi->b, a , b);

    avg1=avg1+a;
    avg2=avg2+b;
 }
   avg1=avg1/ColorRGBList.size();
   avg2=avg2/ColorRGBList.size();
 // vypocet kovariancni matice
 bi = ColorRGBList.begin();
 for (; bi != ei; ++bi)
 {
    ConvertColor(bi->r, bi->g, bi->b, a , b);
    kov11=kov11+(a-avg1)*(a-avg1);
    kov12=kov12+(a-avg1)*(b-avg2);
    kov21=kov21+(b-avg2)*(a-avg1);
    kov22=kov22+(b-avg2)*(b-avg2);
 }
    kov11=kov11/ColorRGBList.size();
    kov12=kov12/ColorRGBList.size();
    kov21=kov21/ColorRGBList.size();
    kov22=kov22/ColorRGBList.size();
 // vypocet inverzni matice
 inv11=kov22/(kov11*kov22-kov12*kov21);
 inv12=-kov21/(kov11*kov22-kov12*kov21);
 inv21=-kov12/(kov11*kov22-kov12*kov21);
 inv22=kov11/(kov11*kov22-kov12*kov21);
 // vypocet determinantu
 det=kov11*kov22-kov12*kov21;
}

int c_ColorDetect::SkinDetect(ImageStruct *Input,ImageStruct *Output, double Tres)
{
  int XSize,YSize,SXStep,DXStep;
  int i,j;

  XSize = ImageXSize(Input);
  YSize = ImageYSize(Input);
  SXStep = ImageXOffset(Input);
  DXStep = ImageXOffset(Output);

  for (j = 0;j<YSize;++j)
  {
    unsigned char * Src = &ImageRGBPixelB(Input,0,j);
    unsigned char * Dst = &Image8Pixel(Output,0,j);

    for (i = 0;i<XSize;++i)
    {
      // ulozeno BGR
      unsigned char b=*(Src+0);
      unsigned char g=*(Src+1);
      unsigned char r=*(Src+2);
      Src+=SXStep;
      double prob=IsSkin(r,g,b);
      if (prob>Tres)
        *Dst = (unsigned char)255;
      else
        *Dst = (unsigned char)0;
      Dst += DXStep;
    }
  }
  return 1;
}

int c_ColorDetect::SkinBackDetect(ImageStruct *Input,ImageStruct *Output, double Tres)
{
  int XSize,YSize,SXStep,DXStep;
  int i,j;

  XSize = ImageXSize(Input);
  YSize = ImageYSize(Input);
  SXStep = ImageXOffset(Input);
  DXStep = ImageXOffset(Output);
  Background->solveBackground(Input);
  for (j = 0;j<YSize;++j)
  {
    unsigned char * Src = &ImageRGBPixelB(Input,0,j);
    unsigned char * Dst = &Image8Pixel(Output,0,j);

    for (i = 0;i<XSize;++i)
    {
      // ulozeno BGR
      unsigned char b=*(Src+0);
      unsigned char g=*(Src+1);
      unsigned char r=*(Src+2);
      Src+=SXStep;
      double prob=IsSkin(r,g,b);
      if ((prob>Tres) && (Background->isBackground(i,j)))
        *Dst = (unsigned char)255;
      else
        *Dst = (unsigned char)0;
      Dst += DXStep;
    }
  }
  Count++;
  if (Count%10 == 0)   Background->update(Input);
  return 1;
}
