/*****************************************************************************\
*	                                                                          *
*	   MATRIX LIBRARY - standard methods                                      *
*      Date: 4.2.2003                                                         *
*                                                                             *
\*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include "matrix.h"

#define MATRIX(name,x,y,width)  ( *(name + (width) * (x) + (y)) )

int arMatrixDisp(ARMat *m)
{
	int r, c;

	printf(" === matrix (%d,%d) ===\n", m->row, m->clm);
	for(r = 0; r < m->row; r++) {
		printf(" |");
		for(c = 0; c < m->clm; c++) {
			printf(" %10g", ARELEM0(m, r, c));
		}
		printf(" |\n");
	}
	printf(" ======================\n");

	return 0;
}

int arMatrixDup(ARMat *dest, ARMat *source)
{
	int r,c;

	if(dest->row != source->row || dest->clm != source->clm) {
		return -1;
	}
	for(r = 0; r < source->row; r++) {
		for(c = 0; c < source->clm; c++) {
			ARELEM0(dest, r, c) = ARELEM0(source, r, c);
		}
	}
	return 0;
}

int arMatrixInv(ARMat *dest, ARMat *source)
{
	if(arMatrixDup(dest, source) < 0) return -1;

	return arMatrixSelfInv(dest);
}

int arMatrixMul(ARMat *dest, ARMat *a, ARMat *b)
{
	int r, c, i;

	if(a->clm != b->row || dest->row != a->row || dest->clm != b->clm) return -1;

	for(r = 0; r < dest->row; r++) {
		for(c = 0; c < dest->clm; c++) {
			ARELEM0(dest, r, c) = 0.0;
			for(i = 0; i < a->clm; i++) {
				ARELEM0(dest, r, c) += ARELEM0(a, r, i) * ARELEM0(b, i, c);
			}
		}
	}

	return 0;
}

int arMatrixTrans(ARMat *dest, ARMat *source)
{
	int r, c;

	if(dest->row != source->clm || dest->clm != source->row) return -1;

	for(r = 0; r < dest->row; r++) {
		for(c = 0; c < dest->clm; c++) {
			ARELEM0(dest, r, c) = ARELEM0(source, c, r);
		}
	}

	return 0;
}

int arMatrixUnit(ARMat *unit)
{
	int r, c;

	if(unit->row != unit->clm) return -1;

	for(r = 0; r < unit->row; r++) {
		for(c = 0; c < unit->clm; c++) {
			if(r == c) {
				ARELEM0(unit, r, c) = 1.0;
			}
			else {
				ARELEM0(unit, r, c) = 0.0;
			}
		}
	}

	return 0;
}


static double *minv( double *ap, int dimen, int rowa )
{
        double *wap, *wcp, *wbp;/* work pointer                 */
        int i,j,n,ip=0,nwork;
        int nos[50];
        double epsl;
        double p,pbuf,work;

        epsl = 1.0e-10;         /* Threshold value      */

        switch (dimen) {
                case (0): return(NULL);                 /* check size */
                case (1): *ap = 1.0 / (*ap);
                          return(ap);                   /* 1 dimension */
        }

        for(n = 0; n < dimen ; n++)
                nos[n] = n;

        for(n = 0; n < dimen ; n++) {
                wcp = ap + n * rowa;

                for(i = n, wap = wcp, p = 0.0; i < dimen ; i++, wap += rowa)
                        if( p < ( pbuf = fabs(*wap)) ) {
                                p = pbuf;
                                ip = i;
                        }
                if (p <= epsl)
                        return(NULL);

                nwork = nos[ip];
                nos[ip] = nos[n];
                nos[n] = nwork;

                for(j = 0, wap = ap + ip * rowa, wbp = wcp; j < dimen ; j++) {
                        work = *wap;
                        *wap++ = *wbp;
                        *wbp++ = work;
                }

                for(j = 1, wap = wcp, work = *wcp; j < dimen ; j++, wap++)
                        *wap = *(wap + 1) / work;
                *wap = 1.0 / work;

                for(i = 0; i < dimen ; i++) {
                        if(i != n) {
                                wap = ap + i * rowa;
                                for(j = 1, wbp = wcp, work = *wap;
                                                j < dimen ; j++, wap++, wbp++)
                                        *wap = *(wap + 1) - work * (*wbp);
                                *wap = -work * (*wbp);
                        }
                }
        }

        for(n = 0; n < dimen ; n++) {
                for(j = n; j < dimen ; j++)
                        if( nos[j] == n) break;
                nos[j] = nos[n];
                for(i = 0, wap = ap + j, wbp = ap + n; i < dimen ;
                                        i++, wap += rowa, wbp += rowa) {
                        work = *wap;
                        *wap = *wbp;
                        *wbp = work;
                }
        }
        return(ap);
}

int arMatrixSelfInv(ARMat *m)
{
	if(minv(m->m, m->row, m->row) == NULL) return -1;

	return 0;
}


static double mdet(double *ap, int dimen, int rowa)
/*  double  *ap;          input matrix */
/*  int     dimen;        Dimension of linre and row, those must be equal,
                          that is square matrix.       */
/*  int     rowa;         ROW Dimension of matrix A    */
{
    double det = 1.0;
    double work;
    int    is = 0;
    int    mmax;
    int    i, j, k;

    for(k = 0; k < dimen - 1; k++) {
        mmax = k;
        for(i = k + 1; i < dimen; i++)
            if (fabs(MATRIX(ap, i, k, rowa)) > fabs(MATRIX(ap, mmax, k, rowa)))
                mmax = i;
        if(mmax != k) {
            for (j = k; j < dimen; j++) {
                work = MATRIX(ap, k, j, rowa);
                MATRIX(ap, k, j, rowa) = MATRIX(ap, mmax, j, rowa);
                MATRIX(ap, mmax, j, rowa) = work;
            }
            is++;
        }
        for(i = k + 1; i < dimen; i++) {
            work = MATRIX(ap, i, k, rowa) / MATRIX(ap, k, k, rowa);
            for (j = k + 1; j < dimen; j++)
                MATRIX(ap, i, j, rowa) -= work * MATRIX(ap, k, j, rowa);
        }
    }
    for(i = 0; i < dimen; i++)
        det *= MATRIX(ap, i, i, rowa);
    for(i = 0; i < is; i++) 
        det *= -1.0;
    return(det);
}

double arMatrixDet(ARMat *m)
{

	if(m->row != m->clm) return 0.0;

	return mdet(m->m, m->row, m->row);
}


ARMat *arMatrixAlloc(int row, int clm)
{
	ARMat *m;

	m = (ARMat *)malloc(sizeof(ARMat));
	if( m == NULL ) return NULL;

	m->m = (double *)malloc(sizeof(double) * row * clm);
	if(m->m == NULL) {
		free(m);
		return NULL;
	}
	else {
		m->row = row;
		m->clm = clm;
	}

	return m;
}

int arMatrixFree(ARMat *m)
{
	free(m->m);
	free(m);

	return 0;
}


ARMat *arMatrixAllocDup(ARMat *source)
{
	ARMat *dest;

	dest = arMatrixAlloc(source->row, source->clm);
        if( dest == NULL ) return NULL;

	if( arMatrixDup(dest, source) < 0 ) {
		arMatrixFree(dest);
		return NULL;
	}

	return dest;
}

ARMat *arMatrixAllocInv(ARMat *source)
{
	ARMat *dest;

	dest = arMatrixAlloc(source->row, source->row);
	if( dest == NULL ) return NULL;

	if( arMatrixInv(dest, source) < 0 ) {
		arMatrixFree( dest );
		return NULL;
	}

	return dest;
}

ARMat *arMatrixAllocMul(ARMat *a, ARMat *b)
{
	ARMat *dest;

	dest = arMatrixAlloc(a->row, b->clm);
	if( dest == NULL ) return NULL;

	if( arMatrixMul(dest, a, b) < 0 ) {
		arMatrixFree(dest);
		return NULL;
	}

	return dest;
}

ARMat *arMatrixAllocTrans(ARMat *source)
{
	ARMat *dest;

	dest = arMatrixAlloc(source->clm, source->row);
	if( dest == NULL ) return NULL;

	if( arMatrixTrans(dest, source) < 0 ) {
		arMatrixFree(dest);
		return NULL;
	}

	return dest;
}

ARMat *arMatrixAllocUnit(int dim)
{
	ARMat *m;

	m = arMatrixAlloc(dim, dim);
	if( m == NULL ) return NULL;

	if( arMatrixUnit(m) < 0 ) {
		arMatrixFree(m);
		return NULL;
	}

	return m;
}
