/*****************************************************************************/
/*                       Image processing functions                          */
/*  Source file : hough transform for ellipse detection                      */
/*  Designed by Igor Potucek (potucek@fit.vutbr.cz)                          */
/*  created 2005                                                             */
/*  Description: find ellipses in the image for head tracking                */
/*****************************************************************************/
#include "directioncircle.h"
#define max(a,b) ((a)>(b))?(a):(b)
#include <math>

const int EDGE_THRES = 20;


void SkinHeadDetect::Init(int xsize, int ysize, int Width, int Height, int MinR, int RStep, int NrSteps, float ratio)
{
  XSize = Width;
  YSize = Height;
  XRatio = Width/(float)xsize;
  YRatio = Height/(float)ysize;
  if (Array!=NULL)
     delete [] Array;
  Array = new int[XSize*YSize*NrSteps];
  memset(Array,0,XSize*YSize*NrSteps*sizeof(int));
  MinRadius = MinR;
  Step = RStep;
  Ratio = ratio;
  out = MinR*XRatio;
  Radiuses = NrSteps;
}

void SkinHeadDetect::AddPoint(int x, int y, int ax, int ay, ImageStruct *Skin)
{
   float alpha = 0;
   // precise method
   if (ax!=0)
   {
     // if ((fabs((float)ay/ax))>1)return ;
      alpha=atan((float)ay/ax);
   }
   else
     if (ay>0) alpha=M_PI/2.0;
   else
     if (ay<0) alpha=3*M_PI/2.0;
   float cosa = cos(alpha);
   float sina = sin(alpha);
   for (int i=0; i<Radiuses; i++)
   {
     int r = MinRadius + i*Step;
     int Offset = XSize*YSize*i;
     int y1 = YRatio*(y+r*sina)+0.5;
     int y2 = YRatio*(y-r*sina)+0.5;
     int x1 = XRatio*(x+Ratio*r*cosa)+0.5;
     int x2 = XRatio*(x-Ratio*r*cosa)+0.5;
     // the center can lie on both directions from a border point
     if (((alpha>0) && (alpha<M_PI/2)) || ((alpha>M_PI) && (alpha<3*M_PI/2.0)))
     {   // edge direction
         if ((x2>=0) && (x2<XSize) && (y2>=0) && (y2<YSize))
           Array[Offset+x2+y2*XSize]+=1;
         if ((x1>=0) && (x1<XSize) && (y1>=0) && (y1<YSize))
           Array[Offset+x1+y1*XSize]+=1;
     }
     else
     {   // edge direction /
         if ((x1>=0) && (x1<XSize) && (y2>=0) && (y2<YSize))
           Array[Offset+x1+y2*XSize]+=1;
         if ((x2>=0) && (x2<XSize) && (y1>=0) && (y1<YSize))
           Array[Offset+x2+y1*XSize]+=1;
     }
   }
}


void pdCircleDetect::Init(int xsize, int ysize, int Width, int Height, int Radius, float ratio)
{
  XSize = Width;
  YSize = Height;
  XRatio = Width/(float)xsize;
  YRatio = Height/(float)ysize;
  if (Array!=NULL)
     delete [] Array;
  Array = new int[XSize*YSize];
  memset(Array,0,XSize*YSize*sizeof(int));
  r = Radius;
  Ratio = ratio;
  out = r*XRatio;
}

void pdCircleDetect::AddPoint(int x, int y, int ax, int ay)
{
   float alpha = 0;
   // precise method
   if (ax!=0)
   {
     // if ((fabs((float)ay/ax))>1)return ;
      alpha=atan((float)ay/ax);
   }
   else
     if (ay>0) alpha=M_PI/2.0;
   else
     if (ay<0) alpha=3*M_PI/2.0;
   float cosa = cos(alpha);
   float sina = sin(alpha);
     int y1 = YRatio*(y+r*sina)+0.5;
     int y2 = YRatio*(y-r*sina)+0.5;
     int x1 = XRatio*(x+Ratio*r*cosa)+0.5;
     int x2 = XRatio*(x-Ratio*r*cosa)+0.5;
     // the center can lie on both directions from a border point
     if (((alpha>0) && (alpha<M_PI/2)) || ((alpha>M_PI) && (alpha<3*M_PI/2.0)))
     {   // edge direction
         if ((x2>=0) && (x2<XSize) && (y2>=0) && (y2<YSize))
           Array[x2+y2*XSize]+=1;
         if ((x1>=0) && (x1<XSize) && (y1>=0) && (y1<YSize))
           Array[x1+y1*XSize]+=1;
     }
     else
     {   // edge direction /
         if ((x1>=0) && (x1<XSize) && (y2>=0) && (y2<YSize))
           Array[x1+y2*XSize]+=1;
         if ((x2>=0) && (x2<XSize) && (y1>=0) && (y1<YSize))
           Array[x2+y1*XSize]+=1;
     }
}

void pdCircleDetect::AddPointSkin(int x, int y, int ax, int ay, ImageStruct *Skin)
{
   float alpha = 0;
   // precise method
   if (ax!=0)
   {
     // if ((fabs((float)ay/ax))>1)return ;
      alpha=atan((float)ay/ax);
   }
   else
     if (ay>0) alpha=M_PI/2.0;
   else
     if (ay<0) alpha=3*M_PI/2.0;
   float cosa = cos(alpha);
   float sina = sin(alpha);
     int oy1 = y+r*sina+0.5;    int y1 = YRatio*oy1+0.5;
     int oy2 = y-r*sina+0.5;    int y2 = YRatio*oy2+0.5;
     int ox1 = x+Ratio*r*cosa+0.5;     int x1 = XRatio*ox1+0.5;
     int ox2 = x-Ratio*r*cosa+0.5;     int x2 = XRatio*ox2+0.5;
     // the center can lie on both directions from a border point
     if (((alpha>0) && (alpha<M_PI/2)) || ((alpha>M_PI) && (alpha<3*M_PI/2.0)))
     {   // edge direction
         if ((x2>=0) && (x2<XSize) && (y2>=0) && (y2<YSize))
           Array[x2+y2*XSize]+=Image8Pixel(Skin,ox2,oy2)>0?5:1;
         if ((x1>=0) && (x1<XSize) && (y1>=0) && (y1<YSize))
           Array[x1+y1*XSize]+=Image8Pixel(Skin,ox1,oy1)>0?5:1;
     }
     else
     {   // edge direction /
         if ((x1>=0) && (x1<XSize) && (y2>=0) && (y2<YSize))
           Array[x1+y2*XSize]+=Image8Pixel(Skin,ox1,oy2)>0?5:1;
         if ((x2>=0) && (x2<XSize) && (y1>=0) && (y1<YSize))
           Array[x2+y1*XSize]+=Image8Pixel(Skin,ox2,oy1)>0?5:1;
     }
}



#include "Unit1.h"
#ifdef DIGILIB
void pdCircleDetect::ProcessImage(ImageStruct *Image)
{
    int mask[3] = { -2, 0, 2 };
    for (int j=1; j<Image->YSize-1; j++)
      for (int i=1; i<Image->XSize-1; i++)
      {
         int sumY = 0;
         int sumX = 0;
         for(int J=-1; J<=1; J++)
            sumY = sumY + Image8Pixel(Image,i,j+J) * mask[J+1];
         for(int J=-1; J<=1; J++)
            sumX = sumX + Image8Pixel(Image,i+J,j) * mask[J+1];
         int direct = abs(sumX)+abs(sumY);
         if ((abs(sumX)>EDGE_THRES) || (abs(sumY)>EDGE_THRES))
         {
           AddPoint(i,j,sumX,sumY);
         //  SetImage8Pixel(Form1->ImageOutput->Image,i,j,direct);
         }
      }
}

void pdCircleDetect::ProcessImageSkin(ImageStruct *Image, ImageStruct *Skin)
{
    int mask[3] = { -2, 0, 2 };
    for (int j=1; j<Image->YSize-1; j++)
      for (int i=1; i<Image->XSize-1; i++)
      {
         int sumY = 0;
         int sumX = 0;
         for(int J=-1; J<=1; J++)
            sumY = sumY + Image8Pixel(Image,i,j+J) * mask[J+1];
         for(int J=-1; J<=1; J++)
            sumX = sumX + Image8Pixel(Image,i+J,j) * mask[J+1];
         int direct = abs(sumX)+abs(sumY);
         if ((abs(sumX)>EDGE_THRES) || (abs(sumY)>EDGE_THRES))
         {
           AddPointSkin(i,j,sumX,sumY, Skin);
         //  SetImage8Pixel(Form1->ImageOutput->Image,i,j,direct);
         }
      }
}

void pdCircleDetect::DrawEllipse(ImageStruct *Image, int x0, int y0, float ratio)
{
    for (float a=0; a<2*M_PI; a+=0.01)
    {
       float x = x0+r*Ratio*cos(a);
       float y = y0+r*sin(a);

       if ((x>=0) && (y>=0) && (x<Image->XSize) && (y<Image->YSize))
         SetImageRGBPixel(Image,int(x*ratio),int(y*ratio),50,250,50);
    }
}

int pdCircleDetect::FindMax(int &x, int &y)
{
    int great=1;

    // search maximum
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         if (great<Array[j*XSize+i])
         {
            great = Array[j*XSize+i];
            x = i; y = j;
         }
      }
    int MinX = x-out;                if (MinX<0) MinX = 0;
    int MaxX = x+out;                if (MaxX>XSize) MaxX = XSize;
    int MinY = y-out/Ratio;      if (MinY<0) MinY = 0;
    int MaxY = y+out/Ratio;      if (MaxY>YSize) MaxY = YSize;
    for (int j=MinY; j<MaxY; j++)
      for (int i=MinX; i<MaxX; i++)
        Array[j*XSize+i] = 0;
    x=x/XRatio; y=y/YRatio;
    return great;
}

float pdCircleDetect::FindRatio(ImageStruct *Image, float minRatio, float maxRatio)
{
  int max = 0;
  float FinalRatio;
  for(float rat=minRatio; rat<maxRatio; rat+=0.01)
  {
     memset(Array,0,XSize*YSize*sizeof(int));
     Ratio = rat;
     ProcessImage(Image);
     ArrayBlur();
     int x,y;
     int value = FindMax(x,y);
     if (value>max)
     {
        max = value;
        FinalRatio = rat;
     }
  }
  return FinalRatio;
}

void pdCircleDetect::ArrayBlur()
{
  int         MASK[5][5];
  int         c ,r , I, J;
  float       SUM;
  int *src = new int[XSize*YSize];
  memcpy(src,Array,XSize*YSize*sizeof(int));
  int *dst = Array;
  /* 5x5 Gaussian mask.  Ref: http://www.cee.hw.ac.uk/hipr/html/gsmooth.html */
  MASK[0][0] = 2; MASK[0][1] =  4; MASK[0][2] =  5; MASK[0][3] =  4; MASK[0][4] = 2;
  MASK[1][0] = 4; MASK[1][1] =  9; MASK[1][2] = 12; MASK[1][3] =  9; MASK[1][4] = 4;
  MASK[2][0] = 5; MASK[2][1] = 12; MASK[2][2] = 15; MASK[2][3] = 12; MASK[2][4] = 5;
  MASK[3][0] = 4; MASK[3][1] =  9; MASK[3][2] = 12; MASK[3][3] =  9; MASK[3][4] = 4;
  MASK[4][0] = 2; MASK[4][1] =  4; MASK[4][2] =  5; MASK[4][3] =  4; MASK[4][4] = 2;

   /**********************************************
   *   GAUSSIAN MASK ALGORITHM STARTS HERE
   ***********************************************/

   for(r=0; r<YSize; r++)  {
   for(c=0; c<XSize; c++)  {
        SUM = 0;
         if(r==0 || r==1 || r==YSize-2 || r==YSize-1)
              SUM = src[r*XSize+c];
         else if(c==0 || c==1 || c==XSize-2 || c==XSize-1)
              SUM = src[r*XSize+c];
         else
         {
           for(I=-2; I<=2; I++)
             for(J=-2; J<=2; J++)
             {
              SUM = SUM + (int)( src[c+I+(r+J)*XSize]*MASK[I+2][J+2]);
             }
           SUM=SUM/115;
          }

        *dst=(int)SUM;
        dst+=1;
   }
   }
   delete []src;
}

//------------------------------------------------------------------------------
 long pdCircleDetect::DrawArray(ImageStruct *Image)
 {  // normalize by maximal value
    int great=1;
    // search maximum
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         great=max(great,Array[j*XSize+i]);
      }
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         int value=255*Array[j*XSize+i]/great;
         if (value>2)
           SetImageRGBPixel(Image,i,j,value, value, value)
         else
           SetImageRGBPixel(Image,i,j,0,0,0);
      }
    return great;
 }
#endif




