/*****************************************************************************/
/*                       Image processing functions                          */
/*  Source file : color conversion functions                                 */
/*  Designed by Igor Potucek (potucek@fit.vutbr.cz)                          */
/*  created 2005                                                             */
/*****************************************************************************/
#include <math>
#include <mem>
//#include <stdio>
//#include <mem.h>
#include "colorhsv.h"

#define max(a,b) ((a)>(b))?(a):(b)

// define minimal value of saturation part -> hardly defined color
const int MIN_SATURATION=2;                     // ideal 10% of range
// define minimal value of intensity part -> dark hardly defined colors
const int MIN_VALUE=2;                          // ideal 10% of range

// count of bins in histogram for each part - altogether BINS_COUNT x BINS_COUNT
const int BINS_COUNT = 50;                     // ideal value 20-30

void rgb_to_rg(struct s_ColorBGR *rgb, struct s_ColorHSV &rg)
{
   int sum = rgb->r+rgb->g+rgb->b;
   if (sum>0)
   {
     rg.a=(BINS_COUNT*rgb->r)/sum;
     rg.b=(BINS_COUNT*rgb->g)/sum;
   }
   else
   {
     rg.a=0;
     rg.b=0;
   }
}

void rgb_to_hsv(struct s_ColorBGR *rgb, struct s_ColorHSV &hsv)
{
        int min, max, delta;
        int i, w;

        w = 0;
        min = max = rgb->r;
        if (rgb->g<min) min=rgb->g;
        if (rgb->b<min) min=rgb->b;
        if (rgb->g>max) { max=rgb->g; w=1; }
        if (rgb->b>max) { max=rgb->b; w=2; }

        /* calculate V */
        hsv.c = BINS_COUNT*max/255;
        if (hsv.c<MIN_VALUE)
        {
           hsv.a=0; hsv.b=0; hsv.c=0;
           return;
        }
        /* calculate S */
        hsv.b = (BINS_COUNT*(max-min))/max;

        /* calculate H */
        delta = max-min;
        if (delta==0)
        {
          hsv.a=0; hsv.b=0;
          return ;
        }
        switch(w) {
        case 0: hsv.a = BINS_COUNT*(0.0 + (rgb->g-rgb->b)/(float)delta)/6; break;
        case 1: hsv.a = BINS_COUNT*(2.0 + (rgb->b-rgb->r)/(float)delta)/6; break;
        case 2: hsv.a = BINS_COUNT*(4.0 + (rgb->r-rgb->g)/(float)delta)/6; break;
        }
        while (hsv.a < 0) hsv.a += BINS_COUNT;
     // printf("%d %d %d\n",hsv.a, hsv.b, hsv.c);
}

//------------------------------------------------------------------------------

void hsv_to_rgb(float *rgb, float *hsv) {
        float h,s,v,f,p,q,t;
        int i;

        h = hsv[0];
        s = hsv[1];
        v = hsv[2];

        while(h<0.0) h+=360.0;
        while(h>=360.0) h-=360.0;

        h/=60.0;
        i = floor(h);
        f = h-(float)i;
        p = v*(1.0-s);
        q = v*(1.0-(s*f));
        t = v*(1.0-(s*(1.0-f)));

        switch(i) {
        case 0: rgb[0] = v; rgb[1] = t; rgb[2] = p; break;
        case 1: rgb[0] = q; rgb[1] = v; rgb[2] = p; break;
        case 2: rgb[0] = p; rgb[1] = v; rgb[2] = t; break;
        case 3: rgb[0] = p; rgb[1] = q; rgb[2] = v; break;
        case 4: rgb[0] = t; rgb[1] = p; rgb[2] = v; break;
        case 5: rgb[0] = v; rgb[1] = p; rgb[2] = q; break;
        }
}

//------------------------------------------------------------------------------
// CLASS for creating histogram from given area
//------------------------------------------------------------------------------
void c_ColorDetectHSV::AddIntoMap(int a, int b)
{
  Map[XSize*b+a]=1;
}

void c_ColorDetectHSV::InitHistogram(int XAxis, int YAxis)
{  // allocates memory for histogram and clean it
   if (Histogram!=NULL) delete[] Histogram;
   Histogram=NULL;
   Histogram = new float[XAxis*YAxis];
   if (Map!=NULL) delete[] Map;
   Map=NULL;
   Map = new int[XAxis*YAxis];
   XSize=XAxis;
   YSize=YAxis;
   memset(Histogram, 0, XAxis*YAxis*sizeof(float));
   memset(Map, 0, XAxis*YAxis*sizeof(int));
   Count=0;
}
//------------------------------------------------------------------------------

float c_ColorDetectHSV::GetHistogramValue(int a, int b)
{  // return one value from histogram
   if ((a<XSize) && (b<YSize))
     return Histogram[b*XSize+a];
   else
     return 0;
}

//------------------------------------------------------------------------------
 bool c_ColorDetectHSV::AddColor(unsigned char r,unsigned char g,unsigned char b)
 {  // transform color into HSV model and add to the histogram
    s_ColorHSV rg;
    s_ColorBGR rgb; rgb.r=r; rgb.g=g; rgb.b=b;
    rgb_to_hsv(&rgb,rg);
    Count++;
    if ((rg.b>MIN_SATURATION) && (rg.a<XSize) && (rg.b<YSize))
       Histogram[rg.b*XSize+rg.a]+=1;
    else
      return false;
    return true;
 }

 int c_ColorDetectHSV::FindProminentColor()
 // find part of hue color, which has in histogram greatest number
 // histogram is splitted into 6 parts of hue color
 {  // range of hue is 0 to BINS_COUNT (hue color is normalized)
    float greatest=0, max=0;
    for (int i=0; i<6; i++)
    {
       int low=i*BINS_COUNT/6;                          // low border of hue color
       int high=(i+1)*BINS_COUNT/6;                     // upper border of hue color
       float value=0;
       for (int j=low; j<high; j++)                     // for each of 6 hue colors
       {
          for (int k=0; k<BINS_COUNT; k++)
            value+=Histogram[k*XSize+j%BINS_COUNT];     // modulo to ensure -30 to 30 degree color
       }
       if (value>max)
       {
          max=value;
          greatest=i;
       }
    }
    return greatest;
 }

//------------------------------------------------------------------------------

 bool c_ColorDetectHSV::IsMapSkin(unsigned char r,unsigned char g,unsigned char b)
 {
    s_ColorHSV rg;
    s_ColorBGR rgb; rgb.r=r; rgb.g=g; rgb.b=b;
    rgb_to_hsv(&rgb,rg);
    if (Map[rg.a+rg.b*YSize]==1)
       return true;
    return false;
 }
//------------------------------------------------------------------------------

 bool c_ColorDetectHSV::IsSkin(unsigned char r,unsigned char g,unsigned char b)
 {
    s_ColorHSV rg;
    s_ColorBGR rgb; rgb.r=r; rgb.g=g; rgb.b=b;
    rgb_to_hsv(&rgb,rg);
    int Color=(rg.a<<8)+(BINS_COUNT<<8)/12;
    if ((rg.b>6) && (Color>=256) && (Color<=3*256) && (rg.b<BINS_COUNT/2))
      return true;
    return false;
 }
//------------------------------------------------------------------------------
#ifdef DIGILIB
void c_ColorDetectHSV::SkinDetection(ImageStruct *Input, ImageStruct *Output)
{
  int XSize,YSize,SXStep,DXStep;
  int i,j;
  XSize = ImageXSize(Input);
  YSize = ImageYSize(Input);
  SXStep = ImageXOffset(Input);
  DXStep = ImageXOffset(Output);
  for (j = 0;j<YSize;++j)
  {
    unsigned char * Src = &ImageRGBPixelB(Input,0,j);
    unsigned char * Dst = &Image8Pixel(Output,0,j);
    for (i = 0;i<XSize;++i)
    {
      // ulozeno BGR
      unsigned char b=*(Src+0);
      unsigned char g=*(Src+1);
      unsigned char r=*(Src+2);
      Src+=SXStep;
      if (IsSkin(r,g,b))
        *Dst = (unsigned char)255;
      else
        *Dst = (unsigned char)0;
      Dst += DXStep;
    }
  }

}

void c_ColorDetectHSV::MapSkinDetection(ImageStruct *Input, ImageStruct *Output)
{
  int XSize,YSize,SXStep,DXStep;
  int i,j;
  XSize = ImageXSize(Input);
  YSize = ImageYSize(Input);
  SXStep = ImageXOffset(Input);
  DXStep = ImageXOffset(Output);
  for (j = 0;j<YSize;++j)
  {
    unsigned char * Src = &ImageRGBPixelB(Input,0,j);
    unsigned char * Dst = &Image8Pixel(Output,0,j);
    for (i = 0;i<XSize;++i)
    {
      // ulozeno BGR
      unsigned char b=*(Src+0);
      unsigned char g=*(Src+1);
      unsigned char r=*(Src+2);
      Src+=SXStep;
      if (IsMapSkin(r,g,b))
        *Dst = (unsigned char)255;
      else
        *Dst = (unsigned char)0;
      Dst += DXStep;
    }
  }

}

//------------------------------------------------------------------------------
 long c_ColorDetectHSV::Drawhistogram(ImageStruct *Image)
 {  // normalize by maximal value
    float great=1;
    // search maximum
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         great=max(great,Histogram[j*XSize+i]);
      }
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         SetImage8Pixel(Image,i,j,255-255*Histogram[j*XSize+i]/great);
      }
    return great;
 }
#endif

void c_Smoothing::Init(int Nr, float Value)
{
  Size=Nr;
  Actual=0;
  for (int i=0; i<Nr; i++)
    Arr[i]=Value;
}
float c_Smoothing::Compute(float Value)
{ // uklada hodnoty odchylek od predikce a vraci prumernou hodnotu
  int i;
  float pom=0;
  if (Actual>(Size-1))
    Actual=0;
  Arr[Actual]=Value;
  Actual=Actual+1;

  for (i=0; i<Size; i++)
     pom=pom+Arr[i];
  pom=pom / Size;              // vypocita prumernou hodnotu odchylek
  return pom;
}

float c_Smoothing::Get()
{ // vraci prumernou hodnotu odchylek od predikce
  float pom=0;
  int i;
  for (i=0; i<Size; i++)
     pom=pom+Arr[i];
  pom=pom / Size;              // vypocita prumernou hodnotu odchylek
  return pom;
}
