/*****************************************************************************/
/*                       Function for circle detection                       */
/*  Using cubic edge detection and finding circle parameters                 */
/*  Source file : cubiccircle.cpp                                            */
/*  Designed by Igor Potucek (potucek@fit.vutbr.cz)                          */
/*  created 2005                                                             */
/*****************************************************************************/
#include "cubiccircle.h"
#include "cubicedge.h"
#include "MyImageFun.h"
#include <math.h>
#include <time.h>
#include "definitions.h"
#include "logging.h"
extern float RATIO;

void c_CubicCircle::SpecialCubicDetect(ImageStruct *Input, float &x, float &y, float &r, int TestCount)
{
   // return only center coordinates, which are computed as average
   ImageStruct *ImageGrey=NULL;
   float maxder=0, xpos, ypos,rad;
   double avgX = 0, avgY = 0, avgR = 0;
   long count = 0;
   FastDetection.clear();
   ResizeImage8(&ImageGrey, Input->XSize, Input->YSize);
   GreyScale(Input,ImageGrey);
   for (int i = 0;i<TestCount;++i)
   {
     float angle = 2*M_PI*(float)i/TestCount;
     for (float radius = r-Distance/2.0; radius<r+Distance/2.0;radius+=1)
     {
       float xret, yret;
       float der=DetectEdge(ImageGrey, x, y, radius, angle, xret, yret);
       if (fabs(maxder)<fabs(der))
       {
          maxder=der; xpos=xret; ypos=yret; rad=radius;
       }
     }
     Value[i] = maxder;
     XVal[i]  = xpos;
     YVal[i]  = ypos;
     Radius[i]= rad;
     if (maxder>10)
     {
       avgX += xpos-cos(angle)*rad;
       avgY += ypos-sin(angle)*rad;
       avgR += rad;
       count++;
     }
     maxder=0;
   }
   x = avgX/count;
   y = avgY/count;
   r = avgR/count;
}

void c_CubicCircle::CubicDetect(ImageStruct *Input, float &x, float &y, float &r, int TestCount)
{  // with approximation values around teh circle line
   ImageStruct *ImageGrey=NULL;
   float maxder=0, xpos, ypos,rad;
   FastDetection.clear();
   ResizeImage8(&ImageGrey, Input->XSize, Input->YSize);
   GreyScale(Input,ImageGrey);
   for (int i = 0;i<TestCount;++i)
   {
     for (float radius = r-Distance/2.0; radius<r+Distance/2.0;radius+=1)
     {
       float xret, yret;
       float der=DetectEdge(ImageGrey, x, y, radius, 2*M_PI*(float)i/TestCount, xret, yret);
       if (fabs(maxder)<fabs(der))
       {
          maxder=der; xpos=xret; ypos=yret; rad=radius;
       }
     }
     Value[i] = maxder;
     XVal[i]  = xpos;
     YVal[i]  = ypos;
     Radius[i]= rad;

     maxder=0;
   }
   // compute smoothed average sign values of the 1st derivation
   for (int i = 0;i<TestCount;++i)
   {
      float value=0;
      int k=0;
      for (int w=-c_Window/2; w<c_Window/2; w++)
      {
         value+=fabs(Value[(TestCount+w+i)%(TestCount)]);
         k+=(Value[(TestCount+w+i)%(TestCount)]>=0)?1:-1;
      }
      value=value/c_Window;
      k=(k>=0)?1:-1;
      CompValue[i]=k*fabs(value);
   }
   // choose only the same egde orientations in the longer segment, given by the smoothed average
   for (int i = 0;i<TestCount;++i)
   {
     // determine if the sign agrees with the smoothed average
     bool sgn = (((CompValue[i]>=0) && (Value[i]>=0)) || ((CompValue[i]<0) && (Value[i]<0)));
     // choose only the edges in the range <Distance/3.0 and with the first derivation > 2/3 of the smoothed average
     if ((fabs(Radius[i]-r)<Distance/3.0) && sgn && (fabs(CompValue[i])/1.5<fabs(Value[i])))
     {
        FastDetection.addPoint(XVal[i],YVal[i]);
         int xpos=RATIO*(XVal[i]-x)+x;
         int ypos=YVal[i];
         SetImageRGBPixel(Input,(int)xpos,(int)ypos,240,240,240);
         SetImageRGBPixel(Input,(int)xpos,(int)ypos+1,200,20,20);
         SetImageRGBPixel(Input,(int)xpos-1,(int)ypos,200,20,20);
         SetImageRGBPixel(Input,(int)xpos+1,(int)ypos,200,20,20);
         SetImageRGBPixel(Input,(int)xpos,(int)ypos-1,200,20,20);
     }
   }
   FastDetection.Blockfit();
   x=FastDetection.xcenter();
   y=FastDetection.ycenter();
   r=FastDetection.radius();
}

void c_CubicCircle::CubicDetectOld(ImageStruct *Input, float &x, float &y, float &r, int TestCount)
{  // old cubic detection
   ImageStruct *ImageGrey=NULL;
   // hustota testovani okoli kruznice
   float xret, yret;
   float maxder=0, xpos, ypos,rad;
   t_PointList ResultList;
   s_Point Point;
   ResultList.clear();
   FastDetection.clear();
   /*
   ResizeImage8(&ImageGrey, Input->XSize, Input->YSize);
   GreyScale(Input,ImageGrey);
   */
   ImageGrey = Input;


   for (int i = 0;i<TestCount;++i)
   {
     clock_t Start = clock();
     for (float radius = r-Distance/2.0; radius<=r+Distance/2.0;radius+=0.7)
     {
       //float X = x+RATIO*cos(2*M_PI*(float)i/TestCount)*(r+Height/2.0-j);
       //float Y = y+sin(2*M_PI*(float)i/TestCount)*(r+Height/2.0-j);
       float der=DetectEdge(ImageGrey, x, y, radius, 2*M_PI*(float)i/TestCount, xret, yret);
       if (maxder<fabs(der))
       {
          maxder=fabs(der); xpos=xret; ypos=yret; rad=radius;
       }
     }
//     Log("cubic", "Time: %d Length: %d", int(1000.0 * Start / CLK_TCK), int(1000.0 * (clock() - Start) / CLK_TCK));
     if (maxder>10)
     {
        maxder=0;
        Point.x=xpos; Point.y=ypos; Point.radius=rad;
        ResultList.push_back(Point);
        xpos=RATIO*(xpos-x)+x;
        /*
         SetImageRGBPixel(Input,(int)xpos,(int)ypos,240,240,240);
         SetImageRGBPixel(Input,(int)xpos,(int)ypos+1,200,20,20);
         SetImageRGBPixel(Input,(int)xpos-1,(int)ypos,200,20,20);
         SetImageRGBPixel(Input,(int)xpos+1,(int)ypos,200,20,20);
         SetImageRGBPixel(Input,(int)xpos,(int)ypos-1,200,20,20);
         */
     }
   }



   t_PointList::iterator bi = ResultList.begin(), ei = ResultList.end();
   for (; bi != ei; ++bi)
   {
     if (fabs((*bi).radius-r)<Distance/4.0)
     {
        FastDetection.addPoint((*bi).x,(*bi).y);
     }
   }
   FastDetection.Blockfit();
   x=FastDetection.xcenter();
   y=FastDetection.ycenter();
   r=FastDetection.radius();
   ResultList.clear();


//   DeleteImage(ImageGrey);
}

void c_CubicCircle::AllCubicDetect(ImageStruct *Input, float &x, float &y, float &r)
{
  t_PointList ResultList;
  ImageStruct *ImageGrey=NULL;
  s_Point Point;
  int cnt, w, TestCount=200; int Height=20;
  float xret, yret;
  float maxder=0, xpos, ypos, rad;
  ResizeImage8(&ImageGrey, Input->XSize, Input->YSize);
  GreyScale(Input,ImageGrey);
 for (w=0; w<5; w++)
 {
  FastDetection.clear();
  ResultList.clear();
  cnt=0;
   for (int i = 0;i<TestCount;++i)
   {
     for (float radius = r-Height/2.0; radius<r+Height/2.0;radius++)
     {
       float der=DetectEdge(ImageGrey, x, y, radius, 2*M_PI*(float)i/TestCount, xret, yret);
       if (maxder<der)
       {
          maxder=der; xpos=xret; ypos=yret; rad=radius;
       }
     }
     if (maxder>40)
     {
        maxder=0;
        Point.x=xpos; Point.y=ypos; Point.radius=rad;
        ResultList.push_back(Point);
        xpos=RATIO*(xpos-x)+x;
         SetImageRGBPixel(Input,(int)xpos,(int)ypos,240,240,240);
         SetImageRGBPixel(Input,(int)xpos,(int)ypos+1,200,20,20);
         SetImageRGBPixel(Input,(int)xpos-1,(int)ypos,200,20,20);
         SetImageRGBPixel(Input,(int)xpos+1,(int)ypos,200,20,20);
         SetImageRGBPixel(Input,(int)xpos,(int)ypos-1,200,20,20);

         cnt++;
     }
   }
   if (TestCount/4>cnt)
   {
     if (w<2) { Height=Height*3; } else break;
   }
   cnt=0;
   t_PointList::iterator bi = ResultList.begin(), ei = ResultList.end();
   for (; bi != ei; ++bi)
   {
     if (fabs((*bi).radius-r)<Height/4.0)
     {
        FastDetection.addPoint((*bi).x,(*bi).y);
        cnt++;
     }
   }
   if (TestCount/4>cnt)
   {
     if (w<2) { Height=Height*3; } else break;
   }
     FastDetection.Blockfit();
     x=FastDetection.xcenter();
     y=FastDetection.ycenter();
     r=FastDetection.radius();
     Height=Height/1.35;
 }
 if (ImageGrey!=NULL) DeleteImage(ImageGrey);
}
