/*****************************************************************************/
/*                       Image processing functions                          */
/*  Source file : fasttransform.cpp                                          */
/*  Designed by Igor Potucek (potucek@fit.vutbr.cz)                          */
/*  created 2005                                                             */
/*  Description : fast transformation of the omni-directional image          */
/*                                                                           */
/*  Updates :                                                                */
/*****************************************************************************/
                      
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "fasttransform.h"

c_OmniTransform::c_OmniTransform()
{
  XArray = NULL;
  YArray = NULL;
}

c_OmniTransform::~c_OmniTransform()
{
  if (XArray)
    delete[] XArray;
  if (YArray)
    delete[] YArray;
}

void c_OmniTransform::SetImageParameters(int InputWidth, int InputHeight, int OutputWidth, int OutputHeight, float ratio)
{
   Width = OutputWidth;
   Height = OutputHeight;
   XSize = InputWidth;
   YSize = InputHeight;
   Ratio = ratio;
}

void c_OmniTransform::SetTransformParameters(float XCenter, float YCenter, float radius, float InnerRadius, bool reverse, float offset)
{
   X0 = XCenter;
   Y0 = YCenter;
   Radius = radius;
   R0 = InnerRadius;
   Reverse = reverse;
   Offset = offset;
}

void c_OmniTransform::Init()
// initialize data array for transformation of the whole panoramic image
{
   if (XArray!=NULL)
     delete [] XArray;
   XArray = new float [Width*Height];
   memset(XArray,0,Width*Height*sizeof(float));
   if (YArray!=NULL)
     delete [] YArray;
   YArray = new float [Width*Height];
   memset(YArray,0,Width*Height*sizeof(float));
   float *xarray, *yarray;
   if (Reverse)
   {
     xarray = &XArray[Width*(Height-1)];
     yarray = &YArray[Width*(Height-1)];
   }
   else
   {
     xarray = XArray;
     yarray = YArray;
   }
   for (int j=0; j<Height; j++)
   {
     for (int i=0; i<Width; i++)
     {
       // M_PI offset is for coincidence with the geometrical image transformation
       float X = X0+Ratio*cos(2*M_PI*(float)i/Width+Offset+M_PI)*(Radius-(Radius-R0)*(float)j/Height);
       float Y = Y0+sin(2*M_PI*(float)i/Width+Offset+M_PI)*(Radius-(Radius-R0)*(float)j/Height);
       *xarray = X; xarray+=1;
       *yarray = Y; yarray+=1;
     }
     if (Reverse)
     {
       xarray-=2*Width;
       yarray-=2*Width;
     }
   }
}

#ifdef DIGILIB
void bilinearRGB(const ImageStruct *Input, float X, float Y, int &r, int &g, int &b)
{ // pozor !! osetrit meze, protoze bereme i barvu 4 okolnich pixelu
  int divYY, divXX, divYX, divXY,x1,x2,y1,y2;
  if (X<1) X=1; if (Y<2) Y=1;
  if (Y>(int)(ImageYSize(Input) - 3)) Y=ImageYSize(Input) - 3;
  if (X>(int)(ImageXSize(Input) - 2)) X=ImageXSize(Input) - 2;

    int x=X*256;
    int y=Y*256;
    int centX=(((int)X)<<8)+(128);   // x + 0.5
    int centY=(((int)Y)<<8)+(128);   // y + 0.5
    x1=centX-(x-(128));
    x2=x+(128)-centX;
    y1=centY-(y-(128));
    y2=y+(128)-centY;
    divXX=x1*y1;
    divYY=x2*y2;
    divYX=x1*y2;
    divXY=x2*y1;

    r=(ImageRGBPixelR(Input,(int)(X),(int)(Y)))*divXX+(ImageRGBPixelR(Input,(int)(X+1),(int)(Y+1)))*divYY
    +(ImageRGBPixelR(Input,(int)(X+1),(int)(Y)))*divXY+(ImageRGBPixelR(Input,(int)(X),(int)(Y+1)))*divYX;
    g=(ImageRGBPixelG(Input,(int)(X),(int)(Y)))*divXX+(ImageRGBPixelG(Input,(int)(X+1),(int)(Y+1)))*divYY
    +(ImageRGBPixelG(Input,(int)(X+1),(int)(Y)))*divXY+(ImageRGBPixelG(Input,(int)(X),(int)(Y+1)))*divYX;
    b=(ImageRGBPixelB(Input,(int)(X),(int)(Y)))*divXX+(ImageRGBPixelB(Input,(int)(X+1),(int)(Y+1)))*divYY
    +(ImageRGBPixelB(Input,(int)(X+1),(int)(Y)))*divXY+(ImageRGBPixelB(Input,(int)(X),(int)(Y+1)))*divYX;
    r=r>>16; g=g>>16; b=b>>16;
}

void c_OmniTransform::ImageTransform(ImageStruct *Input, ImageStruct *Output, float X, float Y)
{
   float x=X-X0;        // difference between initial center and new center
   float y=Y-Y0;
   int DXStep = ImageXOffset(Output);
   float *xarray = XArray;
   float *yarray = YArray;
   for (int j=0; j<Height; j++)
   {
     unsigned char * Dst = &ImageRGBPixelB(Output,0,j);
     for (int i=0; i<Width; i++)
     {
        int r,g,b;
        bilinearRGB(Input, x+*xarray, y+*yarray, r,g,b);
        *Dst=b;
        *(Dst+1)=g;
        *(Dst+2)=r;
        Dst+=DXStep;
        xarray+=1;  yarray+=1;
     }
   }
}

#endif
