/*
								+---------------------------------+
								|                                 |
								|  ***   Transform utility   ***  |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|             TrUt.h              |
								|                                 |
								+---------------------------------+
*/

#ifndef __TRANSFORM_UTIL_INCLUDED
#define __TRANSFORM_UTIL_INCLUDED

#include "image.h"

enum {
    trut_Ok,               // no error
    trut_Error,            // generic error (trut kaput)
	trut_BadParams,        // invalid parameters
    trut_NoPaths,          // no supported paths found (init error)
    trut_NoShaders,        // no shaders found (init error)
    trut_NoRenderBuffer,   // no offscreen render buffers supported (init error)
    trut_NoMemory,         // not enough memory
    trut_GLError,          // OpenGL error
	trut_NoGL13,           // need at lest OpenGL 1.3 (init error)
	trut_NoMultitextureExt,// need multitexture extension (init error)
	trut_NoShaderExt,      // need shaders (init error)
	trut_BadPixelType,     // unsupported ImageStruct pixel type
	trut_ShCompileError    // shader compile error
};

enum {
    tt_Simple = 0, tt_Geometrical = 1, tt_Geom_Polynet = 2
};
// transform type

enum {
	ipp_None = 0, ipp_SkinDetect = 1, ipp_CalcGrayscale = 2
};
// immediate-post-processing type (results are stored in alpha-channel)
// (it's not like second pass, it's done still in the first pass. prefix
// pp_ reserved for possible "real" post-processing)

enum {
	sip_None = 0, sip_Gray = 1,
	sip_Gauss5_Mono = 2, // monochromatic gauss filter with kernel of width 5
	sip_Gauss7_Mono = 3, // monochromatic gauss filter with kernel of width 7
	sip_Sobel_RadialCutoffMono = 4,	// monochromatic sobel filter, sensitive to edges,
									// tangential to ellipse. furthermore it processes
									// stripe between min and max radius only (speedup)
	sip_LocalMax_RadialCutoffMono = 5,
	sip_RaytraceFrontline = 6
};
// source image processing (none / grayscale)

class COmniDirectionalTransform;
class CGLRenderBuffer;
struct TPath;
class CShaderInfo;
class COpenGL_Driver;
// don't need to know what's inside of those

enum {
	n_pbo_num = 8	// maybe overkill, 2 should be enough, but hey - ve'we got 256 megs
					// of barely used memory (it makes transfers bit faster, but it definitely
					// depends on CPU workload between them)
};

#define GL_BGRA                           0x80E1
// bcb doesn't quite have recent open-gl headres

class __declspec(dllexport) CTrUt {
protected:
    COmniDirectionalTransform *m_p_transform;
	// transform-handler object

    CGLRenderBuffer *m_p_render_buffer;
	// buffer for offscreen rendering of transformation result

    int m_n_transform_type, m_n_ipp_type;
	// transform / immediate post-process type

    int m_b_ready, m_n_transform_state;
    int m_n_last_error;
	// state flags

    float m_f_texel_w, m_f_texel_h;
	float m_f_image_ratio;
	// source image properties

	unsigned int m_n_src_tex, m_n_dst_tex;
	// source / dest texture handles

	char *m_p_s_sh_info_log;
	// shader info log (compile log, generated on errors / warnings)

	COpenGL_Driver *m_p_gl_driver;
	// gl-driver

	int m_b_async_support;
	int m_b_fence_support;
	// async / state query support flags

	unsigned int m_p_fence_id_list[4];
	// fences for state querys (upload, source processing, transform, download)

	unsigned int m_p_pbo_list[n_pbo_num];
	int m_n_cur_pbo;
	// pixel-buffer objects for downloading image

	unsigned int m_n_uploader_pbo;
	// pixel-buffer object for uploading image

    float m_f_src_texel_w, m_f_src_texel_h;
	int *m_p_source_img_proc, m_n_source_img_proc_num, m_b_multipass_sip;
	COmniDirectionalTransform *m_p_imgproc;
    CGLRenderBuffer *m_p_render_buffer_source_img;
	//unsigned int m_n_sipdown_pbo;
	unsigned int m_n_processed_src_tex, m_n_aux_processed_src_tex;
	// type of processing, buffer for offscreen rendering and downloader pixel-buffer object

	int m_n_ellipse_detector, m_n_ellipse_points;

public:
    CTrUt();
    ~CTrUt();

    int n_GetLastError();
    int b_Ready();
	const char *p_s_ShaderInfoLog(); // here you can try to receive info when trut_ShCompileError occurs

    int Init(int n_transform_type, int n_immediate_postprocessing_type, int b_want_ellipse_detector,
		int n_source_width, int n_source_height, int n_transform_width, int n_transform_height,
		int *p_source_img_proc = 0, int n_source_img_proc_num = 0, int b_multipas_source_img_proc = false,
		int n_texture_components = GL_BGRA, int n_texture_data_type = GL_UNSIGNED_BYTE,
		int n_src_readback_components = GL_BGRA, int n_src_readback_data_type = GL_UNSIGNED_BYTE,
        int (*load_rp_function)(std::vector<TPath*>&) = std_Load_RenderPaths,
        int (*load_sh_function)(std::vector<CShaderInfo*>&, const char*) = std_Load_Shaders);
    // init transformation, when succesful, b_Ready() returns true (otherwise false)

	int n_Transform_Width();
	int n_Transform_Height();

	int n_SourceProcessed_Width();
	int n_SourceProcessed_Height();

    int Upload_ImageStruct(ImageStruct *p_image);
    // p_image should contain image of size n_source_width per n_source_height pixels
    // otherwise OpenGL errors may occur (depends on graphics card, opengl driver, etc)

	int n_MaxEllipsePoint_Num() { return n_SourceProcessed_Width(); }
	// maximal number of points that will be detected on ellipse (higher = slower with O(n))

	int FindEllipse(float f_assumed_inner_r, float f_assumed_outer_r,
		float f_assumed_center_x, float f_assumed_center_y,
		float f_pixel_ratio, float f_noise_treshold = .02f, int n_searched_point_num = 360);
	// must specify b_want_ellipse_detector = true in init() in order for this to work
	// call after Upload_ImageStruct before transformations or after ReleaseFramebuffer()
	// between FindEllipse and GetEllipsePoints there must be no calls but b_FindEllipseFinished()
	// or Wait_UntilFindEllipseFinished()

	int GetEllipsePoints(unsigned short *p_point_list);
	// get found points p_point_list must be allocated to 2 * n_searched_point_num (2 ~ x,y)
	// coordinates are in relative space of source image
	// to get pixel coordinate, you have to calculate:
	// x_pixels = x_short / 65280.0 * n_SourceProcessed_Width() // 65280 ~ 255 * 256
	// y_pixels = (1.0f - y_short / 65280.0) * n_SourceProcessed_Height() // 1 - y ~ OpenGL coords
	// function is synchronous

	int Set_Gauss_ImageProcParams(int n_sip_index, float f_direction_x, float f_direction_y);
	int Set_Radial_ImageProcParams(int n_sip_index, float f_inner_r, float f_outer_r,
		float f_center_x, float f_center_y, float f_pixel_ratio, float f_treshold);

	int Set_SkindetectParams(float f_r_avg, float f_g_avg,
		float f_k11 = 0.62545f, float f_k21 = 0.00357f,
		float f_k12 = 0.00357f, float f_k22 = 0.23543f);
	// set skin-detection params (only when ipp_SkinDetect is specified)

    int Set_SimpleParams(float f_inner_r, float f_outer_r, float f_center_x, float f_center_y,
        float f_angular_offset, float f_pixel_ratio);
    // call for simple transform only, otherwise returns false

    int Set_GeomParams(float f_inner_r, float f_outer_r, float f_center_x, float f_center_y,
        float f_angular_offset, float f_pixel_ratio, float f_a2_mm, float f_b2_mm,
        float f_mirror_radius_mm, float f_delta_mm, float f_bottom_y_mm, float f_top_y_mm);
    // call for geom and geom-polynet transform only, otherwise returns false
    // parameters followed by "_mm" can be passed in milimeters

	int n_ImgProc_Num();
	int n_ImgProc(int n_sip_index);
	// number and types of loaded image operations

	int b_MultipassImgProc();
	// return true if we specified b_multipas_source_img_proc true in Init()

	int ProcessSourceImage(int n_sip_index, int b_upside_down, unsigned int n_source_texture);
	// processes source image (async by nature), result goes to texture, whose opengl-id
	// can is returned by n_SrcProcessedTexture_OpenGL_Id(); results are undefined when
	// use target texture (n_SrcProcessedTexture_OpenGL_Id()) as n_source_texture
	// (basically nothing wrong happens, but image is overwriting itself so for filters
	// where pixel-mapping is single pixel in destination to same single pixel in source
	// this shouldn't cause too much trougle. use aux texture anyway)

	int Download_ProcessedSourceImage(ImageStruct *p_image);
	// downloads processed source image into given imagestruct
	// imagestruct must be allocated to the same size as the source image,
	// for the best performance should have the same pixel-type as specified
	// by n_src_readback_components and n_src_readback_data_type in Init() call

	int Swap_ProcessedSource_Textures();
	// swap processed source texture and auxiliar processed source texture
	// (useful for multi-pass processing)

	int Swap_Source_ProcessedSource();
	// swap source texture and processed source texture (useful for multi-pass processing)

    int Transform(int n_quad_tesselation, int b_upside_down = false);
    // begin transforming image, process is CPU-independent, after the call, you can do some usefull
    // stuff and let graphics card work. function switches drawing context to offscreen buffer

    int Download_ImageStruct(ImageStruct *p_image, float f_angle, int n_y);
    // download image from current bound framebuffer. p_image is already allocated,
    // f_angle is angle in radians (fmod-ed to interval 0 .. 2Pi), specifying angle where mid
    // of the image rectangle should lie, n_y is vertical position (of top border) in raster
    // when angle is near 0 or 2Pi so image rectangle crosses left or right border of unfolded image,
    // it is wrapped over (similar to GL_REPEAT texture clamp mode), when image rectangle crosses
    // bottom or tom border of unfolded image, it's part, lying outside isn't modified

    int ReleaseFramebuffer();
    // if you want to draw to your opengl window, it's necessary to un-bind transformation's
    // off-screen framebuffer. however, it's not necessary before next Transform() so if you're
    // not going to use open-gl between transformations, you don't have to call this

    unsigned int n_TransformedTexture_OpenGL_Id();
    // return OpenGL handle of texture, containing transformed image
    // can be freely used for displaying results on screen, bud
    // must not be deleted

    unsigned int n_SrcTexture_OpenGL_Id();
    unsigned int n_SrcProcessedTexture_OpenGL_Id();
	unsigned int n_SrcProcessedAuxTexture_OpenGL_Id();
	// the same for source, processed and aux processed (if available) source textures

    int ShutDown();
    // free all open-gl resources, b_Ready() returns false

    static int std_Load_RenderPaths(std::vector<TPath*> &r_path_list);
    static int std_Load_Shaders(std::vector<CShaderInfo*> &r_shader_list,
		const char *p_s_path_name);
    // here you can get standard shaders (you can also dig them out this way and rewrite
    // them to satisfy your needs)

	int GL_Init(HWND h_wnd, int n_width, int n_height, int n_bpp = 32,
		int n_depth_bpp = 0, int n_stencil_bpp = 0, int b_fullscreen = false);
	int GL_SetContext();
	int b_GL_Running();
	int GL_PageFlip();
	int GL_ShutDown();
	// new functions for work with Open-GL

	int GL_Wait_UntilTransformed();
	// wait until transformation is done

	void GL_Line(float x1, float y1, float x2, float y2,
		unsigned char r, unsigned char g, unsigned char b);
	// draw 2D line

	void GL_TexturedRect(float x1 = -1, float y1 = -1, float s1 = 0, float t1 = 0,
						 float x2 = 1, float y2 = 1, float s2 = 1, float t2 = 1,
						 unsigned int n_texture_id = 1);
	// draw 2D textured rectangle (x,y are screen coords, s,t are texture coords)

	int b_AsyncSupported();
	// return true if asynchronous transfers are supported (call after init!)

	int b_QueryStateSupported();
	// return true if transfer / transformation completeness query is supported (call after init!)

    int Upload_ImageStruct_Async(ImageStruct *p_image);
    int Download_ImageStruct_Async(ImageStruct *p_image, float f_angle, int n_y); // !!! image contents are from previous call of Download_ImageStruct_Async() !!!
	// should work asynchronously (when supported, otherwise return false),
	// parameters means the same as for their's blocking versions

	int b_UploadFinished();
	int b_SIPFinished();
	int b_EllipseDetectorFinished() { return b_SIPFinished(); }
	int b_TransformFinished();
	int b_DownloadFinished();
	// completeness query functions. in case completeness query is not supported,
	// return always true -> use rather synchronous functions
	// when called before call to Upload_ImageStruct_Async() or Download_ImageStruct_Async() or
	// Transform() (which is asynchronous by its nature), return value is true, but open-gl
	// error GL_INVALID_OPERATION is generated

	void WaitUntil_UploadFinished();
	void WaitUntil_SIPFinished();
	void WaitUntil_EllipseDetectorFinished() { WaitUntil_SIPFinished(); }
	void WaitUntil_TransformFinished();
	void WaitUntil_DownloadFinished();
	// you can also decide to wait until the task is finished. in case state
	// queries aren't supported, returns after ALL gl commands in query finished
	// (contains open-gl code, not "while(!b_*Finished()) ;" - like loop)

protected:
	int Init_OpenGL_Exts();
	int ISPixelType_to_OGLPixelType(int n_pixel_type, int &r_n_fmt, int &r_n_data_type);
	unsigned int CreateTexture_2D(int n_width, int n_height, GLenum n_format,
		int b_create_mipmaps, void *p_data, GLenum n_data_format, GLenum n_data_type);
	CGLRenderBuffer *p_Create_RenderBuffer(int n_width, int n_height, int n_format,
		int n_color_bits, int b_float, int n_texture_target = 0, int b_mipmap = false,
		int b_depth_texture_target = false, int b_double_buffer = false,
		int b_depth_buffer = false, int n_depth_bits = 32, int b_stencil_buffer = false,
		int n_stencil_bits = 8);
	int b_Have_FBO();
	int b_Have_PBuffers();
	int n_Bit_Num(int n_data_type);
	int n_Component_Num(int n_components);
};

#endif //__TRANSFORM_UTIL_INCLUDED

/*
 *		-end-of-file-
 */
