/*
								+---------------------------------+
								|                                 |
								|  ***   Transform utility   ***  |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|             TrUt.h              |
								|                                 |
								+---------------------------------+
*/

#include <malloc.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <windows.h>
#include <gl/gl.h>
#include <vector>
#include <stack>
#include "callstack.h"
#include "glext.h"
#include "wglext.h"
#include "opengl20.h"
#include "glstate.h"
#include "shader.h"
#include "renderbuffer.h"
#include "renderpath.h"
#include "vertexbufferobject.h"
#include "transform.h"
#include "transformdata.h"
#include "opengldrv.h"
#include "trut.h"

#if defined(_MSC_VER) && !defined(__MWERKS__) // msvc
#ifndef for
#define for if(0) {} else for
#endif
// msvc error with variable scoping in for loops
// (for(int i = 0; i < 10; i ++) { } if(i == 10) { } was allright in msvc)
#endif

/*
 *								=== CTrUt ===
 */

/*
 *	CTrUt::CTrUt()
 *		- default constructor
 */
CTrUt::CTrUt()
	:m_p_transform(0),
	m_p_render_buffer(0),
	m_n_transform_type(tt_Simple),
	m_n_ipp_type(ipp_None),
	m_b_ready(false),
	m_n_transform_state(0),
	m_n_last_error(trut_Ok),
	m_n_src_tex(0), m_n_processed_src_tex(0),
	m_n_aux_processed_src_tex(0), m_n_dst_tex(0),
	m_p_s_sh_info_log(0),
	m_p_gl_driver(0),
	m_b_async_support(false),
	m_b_fence_support(false),
	m_p_render_buffer_source_img(0),
	m_p_imgproc(0),
	m_p_source_img_proc(0),
	m_n_source_img_proc_num(0),
	m_b_multipass_sip(false),
	m_n_ellipse_detector(-1),
	m_n_ellipse_points(-1)
{
	__FuncGuard("CTrUt::CTrUt");
}

/*
 *	CTrUt::~CTrUt()
 *		- default destructor
 */
CTrUt::~CTrUt()
{
	__FuncGuard("CTrUt::~CTrUt");

	ShutDown();
}

/*
 *	int CTrUt::n_GetLastError()
 *		- return last error, set last error to no-error
 */
int CTrUt::n_GetLastError()
{
	__FuncGuard("CTrUt::n_GetLastError");

	int n_result = m_n_last_error;
	m_n_last_error = trut_Ok;

	return n_result;
}

/*
 *	int CTrUt::b_Ready()
 *		- return true if transform util was succesfuly initialized and is ready to use,
 *		  otherwise return false
 */
int CTrUt::b_Ready()
{
	__FuncGuard("CTrUt::b_Ready");

	return m_b_ready;
}

/*
 *	const char *CTrUt::p_s_ShaderInfoLog()
 *		- return shader info-log, can be 0
 *		- don't delete
 */
const char *CTrUt::p_s_ShaderInfoLog()
{
	__FuncGuard("CTrUt::p_s_ShaderInfoLog");

	return m_p_s_sh_info_log;
}

/*
 *	int CTrUt::GL_Init(HWND h_wnd, int n_width, int n_height, int n_bpp,
 *		int n_depth_bpp, int n_stencil_bpp, int b_fullscreen)
 *		- init open-gl, i think params are self-explanatory
 */
int CTrUt::GL_Init(HWND h_wnd, int n_width, int n_height, int n_bpp,
	int n_depth_bpp, int n_stencil_bpp, int b_fullscreen)
{
	__FuncGuard("CTrUt::GL_Init");

	_ASSERTE(h_wnd != 0);

	if(b_GL_Running() && !GL_ShutDown()) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// in case it's running already, shutdown first

	if(!m_p_gl_driver && !(m_p_gl_driver = new COpenGL_Driver)) {
		m_n_last_error = trut_NoMemory;
		return false;
	}
	// create a new object instance

	if(!m_p_gl_driver->Init(h_wnd, n_width, n_height, n_bpp, n_depth_bpp,
	   n_stencil_bpp, b_fullscreen)) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// init open-gl

	return true;
}

/*
 *	int CTrUt::GL_SetContext()
 *		- set open-gl context to our context. must be called when
 *		  running multiple open-gl rendering contexts
 */
int CTrUt::GL_SetContext()
{
	__FuncGuard("CTrUt::GL_SetContext");

	if(!b_GL_Running()) {
		m_n_last_error = trut_Error;
		return false;
	}
	// must be running

	m_p_gl_driver->Context_SetCurrent();

	return true;
}

/*
 *	int CTrUt::b_GL_Running()
 *		- returns true if open-gl is running, otherwise false
 */
int CTrUt::b_GL_Running()
{
	__FuncGuard("CTrUt::b_GL_Running");

	return m_p_gl_driver && m_p_gl_driver->b_Status();
}

/*
 *	int CTrUt::GL_PageFlip()
 *		- flip pages of open-gl
 */
int CTrUt::GL_PageFlip()
{
	__FuncGuard("CTrUt::GL_PageFlip");

	if(!b_GL_Running()) {
		m_n_last_error = trut_Error;
		return false;
	}
	// must be running

	m_p_gl_driver->PageFlip();

	return true;
}

/*
 *	CTrUt::GL_ShutDown()
 *		- shut down open-gl instance
 */
int CTrUt::GL_ShutDown()
{
	__FuncGuard("CTrUt::GL_ShutDown");

	if(!b_GL_Running()) {
		m_n_last_error = trut_Error;
		return false;
	}
	// must be running

	if(!m_p_gl_driver->Terminate()) {
		m_n_last_error = trut_GLError;
		return false;
	}

	return true;
}

/*
 *	int CTrUt::GL_Wait_UntilTransformed()
 *		- wait until transformation is done
 */
int CTrUt::GL_Wait_UntilTransformed()
{
	__FuncGuard("CTrUt::GL_Wait_UntilTransformed");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return 0;
	}
	// must be running

	glFlush();
	glFinish();

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	return true;
}

/*
 *	void CTrUt::GL_Line(float x1, float y1, float x2, float y2,
 *		unsigned char r, unsigned char g, unsigned char b)
 *		- draw 2D line
 */
void CTrUt::GL_Line(float x1, float y1, float x2, float y2,
	unsigned char r, unsigned char g, unsigned char b)
{
	__FuncGuard("CTrUt::GL_Line");

	glColor3ub(r, g, b);
	glBegin(GL_LINES);
	glVertex2f(x1, y1);
	glVertex2f(x2, y2);
	glEnd();
	glColor3f(1, 1, 1);
}

/*
 *	void CTrUt::GL_TexturedRect(float x1, float y1, float s1, float t1,
 *		float x2, float y2, float s2, float t2, unsigned int n_texture_id)
 *		- draw 2D textured rectangle (x,y are screen coords, s,t are texture coords)
 */
void CTrUt::GL_TexturedRect(float x1, float y1, float s1, float t1,
	float x2, float y2, float s2, float t2, unsigned int n_texture_id)
{
	__FuncGuard("CTrUt::GL_TexturedRect");

	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, n_texture_id);

	glBegin(GL_QUADS);
	glTexCoord2f(s1, t2);
	glVertex2f(x1, y2);
	glTexCoord2f(s2, t2);
	glVertex2f(x2, y2);
	glTexCoord2f(s2, t1);
	glVertex2f(x2, y1);
	glTexCoord2f(s1, t1);
	glVertex2f(x1, y1);
	glEnd();

	glDisable(GL_TEXTURE_2D);
}

int CTrUt::n_Bit_Num(int n_data_type)
{
	__FuncGuard("CTrUt::n_Bit_Num");

	switch(n_data_type) {
	case GL_BYTE:
	case GL_UNSIGNED_BYTE:
		return 8;
	case GL_SHORT:
	case GL_UNSIGNED_SHORT:
		return 16;
	case GL_INT:
	case GL_UNSIGNED_INT:
	case GL_FLOAT:
		return 32;
	default:
		m_n_last_error = trut_BadPixelType;
		return -1;
	}
}

int CTrUt::n_Component_Num(int n_components)
{
	__FuncGuard("CTrUt::n_Component_Num");

	switch(n_components) {
	case GL_INTENSITY:
	case GL_LUMINANCE:
	case GL_ALPHA:
	case GL_RED:
	case GL_GREEN:
	case GL_BLUE:
		return 1;
		break;
	case GL_LUMINANCE_ALPHA:
		return 2;
		break;
	case GL_RGB:
	case GL_BGR:
		return 3;
		break;
	case GL_RGBA:
	case GL_BGRA:
		return 4;
		break;
	default:
		m_n_last_error = trut_BadPixelType;
		return -1;
	}
}

/*
 *	int CTrUt::Init(int n_transform_type,
 *		int n_immediate_postprocessing_type, int n_source_img_proc,
 *		int n_source_width, int n_source_height, int n_transform_width, int n_transform_height,
 *		int n_texture_components, int n_texture_data_type,
 *		int (*load_rp_function)(std::vector<TPath*>&),
 *		int (*load_sh_function)(std::vector<CShaderInfo*>&, const char*))
 *		- init function, creates a new transform
 */
int CTrUt::Init(int n_transform_type,
	int n_immediate_postprocessing_type,
	int b_want_ellipse_detector,
	int n_source_width, int n_source_height, int n_transform_width, int n_transform_height,
	int *p_source_img_proc,	int n_source_img_proc_num, int b_multipas_source_img_proc,
	int n_texture_components, int n_texture_data_type,
	int n_src_readback_components, int n_src_readback_data_type,
    int (*load_rp_function)(std::vector<TPath*>&),
    int (*load_sh_function)(std::vector<CShaderInfo*>&, const char*))
{
	__FuncGuard("CTrUt::Init");

	if(n_transform_type < tt_Simple || n_transform_type > tt_Geom_Polynet ||
	   n_immediate_postprocessing_type < ipp_None || n_immediate_postprocessing_type > ipp_CalcGrayscale ||
	   n_source_width <= 0 || n_source_height <= 0 || n_transform_width <= 0 ||
	   n_transform_height <= 0 || !load_rp_function || !load_sh_function) {
		m_n_last_error = trut_BadParams;
		return false;
	}
	// check parameters

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(!Init_OpenGL_Exts())
		return false;
	// init open-gl extensions

	ShutDown();
	// free any previously allocated data

	m_n_transform_type = n_transform_type;
	m_n_ipp_type = n_immediate_postprocessing_type;
	if(b_want_ellipse_detector)
		n_source_img_proc_num += 4;
	else
		m_n_ellipse_detector = -1;
	if(m_n_source_img_proc_num = n_source_img_proc_num) {
		_ASSERTE(p_source_img_proc || b_want_ellipse_detector);
		if(!(m_p_source_img_proc = new int[n_source_img_proc_num])) {
			m_n_last_error = trut_NoMemory;
			return false;
		}
		memcpy(m_p_source_img_proc, p_source_img_proc,
			(n_source_img_proc_num - 4 * b_want_ellipse_detector) * sizeof(int));
		m_b_multipass_sip = b_multipas_source_img_proc | b_want_ellipse_detector;
		if(b_want_ellipse_detector) {
			int p_ellipse_img_proc[] = {sip_Gray, sip_Gauss5_Mono,
				sip_Sobel_RadialCutoffMono, sip_RaytraceFrontline};
			// we want grayscale filter, gauss filter, radial sobel filter and raytracer

			m_n_ellipse_detector = (n_source_img_proc_num - 4 * b_want_ellipse_detector);
			memcpy(m_p_source_img_proc + m_n_ellipse_detector,
				p_ellipse_img_proc, 4 * sizeof(int));
		}
	} else
		m_b_multipass_sip = false;
	// copy transform parameters

	m_b_fence_support = CGLExtensionHandler::b_SupportedExtension("GL_NV_fence") &&
		!CGLExtensionHandler::n_GetFenceNVFuncPointers();
	// GL_NV_fence for completeness queries

	if(m_b_fence_support)
		glGenFencesNV(3 + (m_n_source_img_proc_num > 0), m_p_fence_id_list);
	// generate 3 fences for texture upload, transform and texture download query
	// + maybe one more for source image filter and result download

	if(glGetError() != GL_NO_ERROR)
		m_b_fence_support = false;

	m_b_async_support = CGLExtensionHandler::b_SupportedExtension("GL_ARB_pixel_buffer_object") &&
		CGLExtensionHandler::b_SupportedExtension("GL_ARB_vertex_buffer_object") &&
		!CGLExtensionHandler::n_GetVertexBufferObjectARBFuncPointers();
	// GL_ARB_pixel_buffer_object for asynchronous queries,
	// GL_ARB_vertex_buffer_object because of gl*Buffer*() functions

	if(m_b_async_support) {
		glGenBuffersARB(n_pbo_num, m_p_pbo_list);
		glGenBuffersARB(1, &m_n_uploader_pbo);
		/*if(m_n_source_img_proc != sip_None)
			glGenBuffersARB(1, &m_n_sipdown_pbo);*/
		m_n_cur_pbo = 0;
	}
	// create n_pbo_num (2 by default) pixel-buffer objects
	// to support asynchronous image download and a single ona
	// for image upload

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(!(m_p_transform = new COmniDirectionalTransform)) {
		m_n_last_error = trut_NoMemory;
		return false;
	}
	// create a new COmniDirectionalTransform object

	if(!m_p_transform->LoadPaths(load_rp_function) ||
	   !m_p_transform->n_CompatibleRenderPath_Num()) {
		m_n_last_error = trut_NoPaths;
		return false;
	}
	if(!m_p_transform->ChooseRenderPath(0) ||
	   !m_p_transform->LoadShaders(load_sh_function)) {
		m_n_last_error = trut_NoShaders;
		return false;
	}
	// load and choose render path

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	char p_s_shader_prefix[256];
	const char *p_transform_name[] = {"simple_", "geom_", "polynet_"};
	const char *p_ipp_name[] = {"noipp_", "skin_", "gray_"};
	//
	_ASSERTE(n_transform_type >= 0 && n_transform_type < 3);
	_ASSERTE(n_immediate_postprocessing_type >= 0 && n_immediate_postprocessing_type < 3);
	strcpy(p_s_shader_prefix, p_transform_name[n_transform_type]);
	strcat(p_s_shader_prefix, p_ipp_name[n_immediate_postprocessing_type]);
	// generate shader name prefix

	for(int i = 0; i < 3; i ++) {
		if(!i) {
			if(m_p_transform->b_Have_HighLevel_ShadingLanguage()) {
				m_p_transform->UseShadingLanguage(true);
				// try to search trough high-level shaders
			} else
				continue;
		} else if(i == 1) {
			if(m_p_transform->b_Have_LowLevel_ShadingLanguage()) {
				m_p_transform->UseShadingLanguage(false);
				// try to search trough low-level shaders
			} else
				continue;
		} else {
			m_n_last_error = trut_NoShaders;
			return false;
		}

		int b_chose_shader = false;
		for(int j = 0; j < m_p_transform->n_Shader_Num(); j ++) {
			if(!strncmp(m_p_transform->p_s_ShaderName(j),
			   p_s_shader_prefix, strlen(p_s_shader_prefix))) {
				b_chose_shader = true;
				m_p_transform->ChooseShader(j);
				break;
			}
		}

		if(b_chose_shader)
			break;
		// search for shader
	}
	// load and choose shader

	if(m_p_s_sh_info_log)
		delete[] m_p_s_sh_info_log;
	m_p_s_sh_info_log = 0;
	if(!m_p_transform->CompileShader(m_p_s_sh_info_log)) {
		m_n_last_error = trut_ShCompileError;
		return false;
	}
	// compile shader

	//m_p_transform->FreeShadersPaths();
	// free shader & paths information. don't need it when shader is compiled

	if(m_n_source_img_proc_num) {
		if(!(m_p_imgproc = new COmniDirectionalTransform[m_n_source_img_proc_num])) {
			m_n_last_error = trut_NoMemory;
			return false;
		}
		// create a new COmniDirectionalTransform objects
	}
	for(int i = 0; i < m_n_source_img_proc_num; i ++) {
		if(!m_p_imgproc[i].LoadPaths(load_rp_function) ||
		   !m_p_imgproc[i].n_CompatibleRenderPath_Num()) {
			m_n_last_error = trut_NoPaths;
			return false;
		}
		if(!m_p_imgproc[i].ChooseRenderPath(0) ||
		   !m_p_imgproc[i].LoadShaders(load_sh_function)) {
			m_n_last_error = trut_NoShaders;
			return false;
		}
		// load and choose render path

		if(glGetError() != GL_NO_ERROR) {
			m_n_last_error = trut_GLError;
			return false;
		}

		char p_s_shader_prefix[256];
		const char *p_sip_name[] = {"noipp_", "gray_", "gauss_5",
			"gauss_7", "radialsobel_", "localmax_", "frontline_"};
		//
		_ASSERTE(m_p_source_img_proc[i] >= 0 &&
			m_p_source_img_proc[i] < sizeof(p_sip_name) / sizeof(p_sip_name[0]));
		strcpy(p_s_shader_prefix, "imgproc_");
		strcat(p_s_shader_prefix, p_sip_name[m_p_source_img_proc[i]]);
		// generate shader name prefix

		for(int n = 0; n < 3; n ++) {
			if(!n) {
				if(m_p_imgproc[i].b_Have_HighLevel_ShadingLanguage()) {
					m_p_imgproc[i].UseShadingLanguage(true);
					// try to search trough high-level shaders
				} else
					continue;
			} else if(n == 1) {
				if(m_p_imgproc[i].b_Have_LowLevel_ShadingLanguage()) {
					m_p_imgproc[i].UseShadingLanguage(false);
					// try to search trough low-level shaders
				} else
					continue;
			} else {
				m_n_last_error = trut_NoShaders;
				return false;
			}

			int b_chose_shader = false;
			for(int j = 0; j < m_p_imgproc[i].n_Shader_Num(); j ++) {
				if(!strncmp(m_p_imgproc[i].p_s_ShaderName(j),
				   p_s_shader_prefix, strlen(p_s_shader_prefix))) {
					b_chose_shader = true;
					m_p_imgproc[i].ChooseShader(j);
					break;
				}
			}

			if(b_chose_shader)
				break;
			// search for shader
		}
		// load and choose shader

		if(m_p_s_sh_info_log)
			delete[] m_p_s_sh_info_log;
		m_p_s_sh_info_log = 0;
		if(!m_p_imgproc[i].CompileShader(m_p_s_sh_info_log)) {
			m_n_last_error = trut_ShCompileError;
			return false;
		}
		// compile shader

		//m_p_imgproc[i].FreeShadersPaths();
		// free shader & path info strings, don't need them anymore
	}

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	CGLShaderUnBinder::UnBind();

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(!(m_n_src_tex = CreateTexture_2D(n_source_width, n_source_height,
	   GL_RGBA, false, NULL, n_texture_components, n_texture_data_type))) {
		m_n_last_error = trut_GLError;
		return false;
	}
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	// set no repeating for source texture
	if(m_n_source_img_proc_num) {
		int n_tex_comp = n_src_readback_components;
		if(n_tex_comp == GL_BGR) n_tex_comp = GL_RGB;
		if(n_tex_comp == GL_BGRA) n_tex_comp = GL_RGBA;
		if(!(m_n_processed_src_tex = CreateTexture_2D(n_source_width, n_source_height,
		   n_tex_comp, false, NULL, n_src_readback_components,
		   n_src_readback_data_type))) {
			m_n_last_error = trut_GLError;
			return false;
		}
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
		// set no repeating for source texture

		if(m_b_multipass_sip) {
			if(!(m_n_aux_processed_src_tex = CreateTexture_2D(n_source_width, n_source_height,
			   n_tex_comp, false, NULL, n_src_readback_components,
			   n_src_readback_data_type))) {
				m_n_last_error = trut_GLError;
				return false;
			}
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
			// set no repeating for source texture
		}
		// create one more texture for double-buffering for multipass algorithms
	}
	if(!(m_n_dst_tex = CreateTexture_2D(n_transform_width, n_transform_height,
	   GL_RGBA, false, NULL, n_texture_components, n_texture_data_type))) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// create source and destination textures

	m_f_image_ratio = (float)n_source_width / (float)n_source_height;
	m_f_texel_w = 1.0 / (float)n_transform_width;
	m_f_texel_h = 1.0 / (float)n_transform_height;
	m_f_src_texel_w = 1.0 / (float)n_source_width;
	m_f_src_texel_h = 1.0 / (float)n_source_height;
	// calculate transform width and height

	int n_bits, b_float = n_texture_data_type == GL_FLOAT;
	if((n_bits = n_Bit_Num(n_texture_data_type)) == -1)
		return false;
	int n_component_num;
	if((n_component_num = n_Component_Num(n_texture_components)) == -1)
		return false;

	if(!(m_p_render_buffer = p_Create_RenderBuffer(n_transform_width, n_transform_height,
	   n_texture_components, n_bits, b_float, GL_TEXTURE_2D)) &&
	   !(m_p_render_buffer = p_Create_RenderBuffer(n_transform_width, n_transform_height,
	   n_texture_components, n_bits, b_float, 0)))
		return false;
	// create render-buffer (first try direct rendering to texture, then simple render buffer)

	if(m_n_source_img_proc_num > 0) {
		b_float = n_src_readback_data_type == GL_FLOAT;
		if((n_bits = n_Bit_Num(n_src_readback_data_type)) == -1)
			return false;
		if((n_component_num = n_Component_Num(n_src_readback_components)) == -1)
			return false;

		if(!(m_p_render_buffer_source_img = p_Create_RenderBuffer(n_source_width, n_source_height,
		   n_texture_components, n_bits, b_float, GL_TEXTURE_2D)) &&
		   !(m_p_render_buffer_source_img = p_Create_RenderBuffer(n_source_width, n_source_height,
		   n_texture_components, n_bits, b_float, 0)))
			return false;
		// create render-buffer for source read-back
		// (first try direct rendering to texture, then simple render buffer)
	}

	if(m_b_async_support) {
		int n_data_size = n_transform_width * n_transform_height *
			n_Bit_Num(n_texture_data_type) / 8 * n_Component_Num(n_texture_components);
		for(int i = 0; i < n_pbo_num; i ++) {
			glBindBufferARB(GL_PIXEL_PACK_BUFFER_ARB, m_p_pbo_list[i]);
			glBufferDataARB(GL_PIXEL_PACK_BUFFER_ARB, n_data_size, NULL, GL_STATIC_READ);
		}
		glBindBufferARB(GL_PIXEL_PACK_BUFFER_ARB, 0);
		glReadBuffer(GL_FRONT);
		// alloc buffers for asynchronous reading (preferably in computer memory)

		n_data_size = n_source_width * n_source_height *
			n_Bit_Num(n_texture_data_type) / 8 * n_Component_Num(n_texture_components);
		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, m_n_uploader_pbo);
		glBufferDataARB(GL_PIXEL_UNPACK_BUFFER_ARB, n_data_size, NULL, GL_STREAM_DRAW_ARB);
		glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, 0);
		//glDrawBuffer(GL_FRONT);
		// alloc buffer for copying source image

		/*if(m_n_source_img_proc != sip_None) {
			n_data_size = n_source_width * n_source_height *
				n_Bit_Num(n_src_readback_data_type) / 8 *
				n_Component_Num(n_src_readback_components);
			glBindBufferARB(GL_PIXEL_PACK_BUFFER_ARB, m_n_sipdown_pbo);
			glBufferDataARB(GL_PIXEL_PACK_BUFFER_ARB, n_data_size, NULL, GL_STATIC_READ);
			glBindBufferARB(GL_PIXEL_PACK_BUFFER_ARB, 0);
			glReadBuffer(GL_FRONT);
		}*/
		// allov buffer for source image downloading
	}
	// create pbo-s

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	m_b_ready = true;
	m_n_transform_state = 0;

	return true;
}

// return true if asynchronous transfers are supported
int CTrUt::b_AsyncSupported()
{
	__FuncGuard("CTrUt::b_AsyncSupported");

	return m_b_async_support;
}

// return true if transfer / transformation completeness query is supported
int CTrUt::b_QueryStateSupported()
{
	__FuncGuard("CTrUt::b_QueryStateSupported");

	return m_b_fence_support;
}

int CTrUt::n_Transform_Width()
{
	__FuncGuard("CTrUt::n_Transform_Width");

	if(!b_Ready()) {
		return -1;
		m_n_last_error = trut_Error;
	}

	return m_p_render_buffer->n_Width();
}

int CTrUt::n_Transform_Height()
{
	__FuncGuard("CTrUt::n_Transform_Height");

	if(!b_Ready()) {
		return -1;
		m_n_last_error = trut_Error;
	}

	return m_p_render_buffer->n_Height();
}

int CTrUt::n_SourceProcessed_Width()
{
	__FuncGuard("CTrUt::n_Transform_Width");

	if(!b_Ready() || !m_n_source_img_proc_num) {
		return -1;
		m_n_last_error = trut_Error;
	}

	return m_p_render_buffer_source_img->n_Width();
}

int CTrUt::n_SourceProcessed_Height()
{
	__FuncGuard("CTrUt::n_Transform_Height");

	if(!b_Ready() || !m_n_source_img_proc_num) {
		return -1;
		m_n_last_error = trut_Error;
	}

	return m_p_render_buffer_source_img->n_Height();
}

int CTrUt::Upload_ImageStruct(ImageStruct *p_image)
{
	__FuncGuard("CTrUt::Upload_ImageStruct");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	int n_pixel_fmt, n_pixel_datatype;

	if(!ISPixelType_to_OGLPixelType(p_image->PixelType, n_pixel_fmt, n_pixel_datatype))
		return false;

	glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize); // fixme - do i have to do this at here?
	glPixelStorei(GL_PACK_SKIP_PIXELS, 0);

	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, m_n_src_tex);

	if(p_image->YOffset < 0) {
		glTexSubImage2D(GL_TEXTURE_2D, // target type
						0, // LOD pyramid level
						0, // x-offset
						0, // y-offset
						p_image->XSize, // width
						p_image->YSize, // height
						n_pixel_fmt,
						n_pixel_datatype,
						p_image->Raster + p_image->YOffset * (p_image->YSize - 1));
	} else {
		glTexSubImage2D(GL_TEXTURE_2D, // target type
						0, // LOD pyramid level
						0, // x-offset
						0, // y-offset
						p_image->XSize, // width
						p_image->YSize, // height
						n_pixel_fmt,
						n_pixel_datatype,
						p_image->Raster);
	}

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	return true;
}
// p_image should contain image of size n_source_width per n_source_height pixels
// otherwise OpenGL errors may occur (depends on graphics card, opengl driver, etc)

int CTrUt::FindEllipse(float f_assumed_inner_r, float f_assumed_outer_r,
	float f_assumed_center_x, float f_assumed_center_y,
	float f_pixel_ratio, float f_noise_treshold, int n_searched_point_num)
{
	__FuncGuard("CTrUt::FindEllipse");

	m_n_ellipse_points = -1;

	if(!m_b_ready || m_n_ellipse_detector < 0 || n_searched_point_num > n_MaxEllipsePoint_Num()) {
		m_n_last_error = trut_Error;
		return false;
	}

	float f_image_ratio = (float)n_SourceProcessed_Width() / (float)n_SourceProcessed_Height();
	// assumed position of ellipse (center, radius range)

	float f_width_divisor = (float)n_SourceProcessed_Width() /
		(float)(n_searched_point_num / 2 + 2);

	Set_Radial_ImageProcParams(m_n_ellipse_detector + 2, f_assumed_inner_r,
		f_assumed_outer_r, f_assumed_center_x, f_assumed_center_y,
		f_pixel_ratio, f_noise_treshold);
	// treshold f_noise_treshold (everything under f_noise_treshold will be black)
	Set_Radial_ImageProcParams(m_n_ellipse_detector + 3, f_assumed_inner_r + .01f,
		f_assumed_outer_r - .01f, f_assumed_center_x,
		f_assumed_center_y, f_pixel_ratio, f_width_divisor *
		3.1415926535897932384626433832795028841971693993f);
	// +- .01f because there's an edge we need to skip
	// set filters

	glScissor(-5 + (int)((f_assumed_center_x - f_assumed_outer_r / f_pixel_ratio) * n_SourceProcessed_Width()),
			  -5 + (int)((f_assumed_center_y - f_assumed_outer_r * f_image_ratio) * n_SourceProcessed_Height()),
			  10 + (int)(f_assumed_outer_r * 2 / f_pixel_ratio * n_SourceProcessed_Width()),
			  10 + (int)(f_assumed_outer_r * 2 * f_image_ratio * n_SourceProcessed_Height()));
	// -5, 10 ... 5 more pixes at both sides
	glEnable(GL_SCISSOR_TEST);
	for(int i = 0; i < 5; i ++) {
		if(i == 1)
			Set_Gauss_ImageProcParams(m_n_ellipse_detector + 1, 1.0f / (float)n_SourceProcessed_Width(), 0.0);
		else if(i == 2)
			Set_Gauss_ImageProcParams(m_n_ellipse_detector + 1, 0.0, 1.0f / (float)n_SourceProcessed_Width());
		else if(i == n_ImgProc_Num()) {
			glDisable(GL_SCISSOR_TEST);
			glScissor(0, 0, n_searched_point_num, 1);
			glEnable(GL_SCISSOR_TEST);
			// the last operation is the raytracer, we want it to plot
			// a line of few fragments with found intersections
		}

		if(!Swap_ProcessedSource_Textures())
			return false;
		if(!ProcessSourceImage(m_n_ellipse_detector + i - (i > 1), true, (!i)?
		   n_SrcTexture_OpenGL_Id() : n_SrcProcessedAuxTexture_OpenGL_Id()))
			return false;

		if(glGetError() != GL_NO_ERROR) {
			m_n_last_error = trut_GLError;
			return false;
		}
	}
	glDisable(GL_SCISSOR_TEST);
	// process image trough all imgproc passes

	m_n_ellipse_points = n_searched_point_num;

	return true;
}

int CTrUt::GetEllipsePoints(unsigned short *p_point_list)
{
	__FuncGuard("CTrUt::GetEllipsePoints");

	if(!m_b_ready || m_n_ellipse_detector < 0 || m_n_ellipse_points < 0) {
		m_n_last_error = trut_Error;
		return false;
	}

	ImageStruct t_frontline_image = {0};

	t_frontline_image.PixelType = ImageRGB;
	t_frontline_image.XSize = m_n_ellipse_points;
	t_frontline_image.Raster = (unsigned char*)p_point_list;
	t_frontline_image.YSize = 1;
	t_frontline_image.XOffset = 4;
	t_frontline_image.YOffset = t_frontline_image.XSize * 4;
	// create bitmap for storing "frontline" - output from edge raytracer

	return Download_ProcessedSourceImage(&t_frontline_image);
}

int CTrUt::Upload_ImageStruct_Async(ImageStruct *p_image)
{
	__FuncGuard("CTrUt::Upload_ImageStruct_Async");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	int n_pixel_fmt, n_pixel_datatype;

	if(!ISPixelType_to_OGLPixelType(p_image->PixelType, n_pixel_fmt, n_pixel_datatype))
		return false;

	glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, m_n_uploader_pbo);

	/*void *p_data = glMapBufferARB(GL_PIXEL_UNPACK_BUFFER_NV, GL_WRITE_ONLY);
	if(!p_data) {
		m_n_last_error = trut_GLError;
        return false;
    }*/

	glBufferData(GL_PIXEL_UNPACK_BUFFER_NV,
		p_image->XSize * p_image->YSize * p_image->XOffset,
		p_image->Raster + ((p_image->YOffset < 0)?
		p_image->YOffset * (p_image->YSize - 1) : 0), GL_STREAM_DRAW_ARB);

	/*memcpy(p_data, p_image->Raster + ((p_image->YOffset < 0)?
		p_image->YOffset * (p_image->YSize - 1) : 0),
		p_image->XSize * p_image->YSize * p_image->XOffset);
	// copy data into the buffer first ...

    if(!glUnmapBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB)) {
		m_n_last_error = trut_GLError;
        return false;
    }*/

	glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize); // fixme - do i have to do this at here?
	glPixelStorei(GL_PACK_SKIP_PIXELS, 0);

	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, m_n_src_tex);

	glTexSubImage2D(GL_TEXTURE_2D, // target type
					0, // LOD pyramid level
					0, // x-offset
					0, // y-offset
					p_image->XSize, // width
					p_image->YSize, // height
					n_pixel_fmt,
					n_pixel_datatype,
					(char*)NULL);
	// copy in graphics card then ...

    glBindBufferARB(GL_PIXEL_UNPACK_BUFFER_ARB, 0);
	//glDrawBuffer(GL_FRONT);

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(m_b_fence_support)
		glSetFenceNV(m_p_fence_id_list[0], GL_ALL_COMPLETED_NV);
	// set marker ("fence") so we can query how far are we with completion of preceding commands

	return true;
}
// p_image should contain image of size n_source_width per n_source_height pixels
// otherwise OpenGL errors may occur (depends on graphics card, opengl driver, etc)

int CTrUt::Set_Gauss_ImageProcParams(int n_sip_index, float f_direction_x, float f_direction_y)
{
	__FuncGuard("CTrUt::Set_Gauss_ImageProcParams");

	if(!m_b_ready || n_sip_index < 0 || n_sip_index >= m_n_source_img_proc_num ||
	   m_p_source_img_proc[n_sip_index] < sip_Gauss5_Mono ||
	   m_p_source_img_proc[n_sip_index] > sip_Gauss7_Mono) {
		m_n_last_error = trut_Error;
		return false;
	}

	m_p_imgproc[n_sip_index].f_GaussDir(0) = f_direction_x;
	m_p_imgproc[n_sip_index].f_GaussDir(1) = f_direction_y;
	// set gauss params

	return true;
}

int CTrUt::Set_Radial_ImageProcParams(int n_sip_index, float f_inner_r, float f_outer_r,
	float f_center_x, float f_center_y, float f_pixel_ratio, float f_treshold)
{
	__FuncGuard("CTrUt::Set_SobelRadialCutoff_ImageProcParams");

	if(!m_b_ready || n_sip_index < 0 || n_sip_index >= m_n_source_img_proc_num ||
	   m_p_source_img_proc[n_sip_index] < sip_Sobel_RadialCutoffMono ||
	   m_p_source_img_proc[n_sip_index] > sip_RaytraceFrontline) {
		m_n_last_error = trut_Error;
		return false;
	}

	m_p_imgproc[n_sip_index].f_InnerRadius() = f_inner_r / f_pixel_ratio;
	m_p_imgproc[n_sip_index].f_OuterRadius() = f_outer_r / f_pixel_ratio;
	m_p_imgproc[n_sip_index].f_MirrorCenter_x() = f_center_x;
	m_p_imgproc[n_sip_index].f_MirrorCenter_y() = f_center_y;
	m_p_imgproc[n_sip_index].f_AngularOffset() = f_treshold;
	m_p_imgproc[n_sip_index].f_SideRatio() = m_f_image_ratio * f_pixel_ratio;
	m_p_imgproc[n_sip_index].f_Pixel_Width() = m_f_src_texel_w;
	m_p_imgproc[n_sip_index].f_Pixel_Height() = m_f_src_texel_h;
	// set radial sobel params

	return true;
}

int CTrUt::Set_SkindetectParams(float f_r_avg, float f_g_avg,
		float f_k11/* = 0.62545f*/, float f_k21/* = 0.00357f*/,
		float f_k12/* = 0.00357f*/, float f_k22/* = 0.23543f*/)
{
	__FuncGuard("CTrUt::Set_SkindetectParams");

	if(!m_b_ready || m_n_ipp_type != ipp_SkinDetect) {
		m_n_last_error = trut_Error;
		return false;
	}

	m_p_transform->f_Covariant(0, 0) = f_k11;
	m_p_transform->f_Covariant(1, 0) = f_k21;
	m_p_transform->f_Covariant(0, 1) = f_k12;
	m_p_transform->f_Covariant(1, 1) = f_k22;
	m_p_transform->f_AvgComponents(0) = f_r_avg;
	m_p_transform->f_AvgComponents(1) = f_g_avg;
	// set skindetect params

	return true;
}

int CTrUt::Set_SimpleParams(float f_inner_r, float f_outer_r, float f_center_x, float f_center_y,
    float f_angular_offset, float f_pixel_ratio)
{
	__FuncGuard("CTrUt::Set_SimpleParams");

	if(!m_b_ready || m_n_transform_type != tt_Simple) {
		m_n_last_error = trut_Error;
		return false;
	}

	m_p_transform->f_InnerRadius() = f_inner_r / f_pixel_ratio;
	m_p_transform->f_OuterRadius() = f_outer_r / f_pixel_ratio;
	m_p_transform->f_MirrorCenter_x() = f_center_x;
	m_p_transform->f_MirrorCenter_y() = f_center_y;
	m_p_transform->f_AngularOffset() = f_angular_offset;
	m_p_transform->f_SideRatio() = m_f_image_ratio * f_pixel_ratio;
	m_p_transform->f_Pixel_Width() = m_f_texel_w;
	m_p_transform->f_Pixel_Height() = m_f_texel_h;
	// set simple params

	return true;
}
// call for simple transform only, otherwise returns false

int CTrUt::Set_GeomParams(float f_inner_r, float f_outer_r, float f_center_x, float f_center_y,
    float f_angular_offset, float f_pixel_ratio, float f_a2_mm, float f_b2_mm,
    float f_mirror_radius_mm, float f_delta_mm, float f_bottom_y_mm, float f_top_y_mm)
{
	__FuncGuard("CTrUt::Set_GeomParams");

	if(!m_b_ready || (m_n_transform_type != tt_Geometrical && m_n_transform_type != tt_Geom_Polynet)) {
		m_n_last_error = trut_Error;
		return false;
	}

	m_p_transform->f_InnerRadius() = f_inner_r / f_pixel_ratio;
	m_p_transform->f_OuterRadius() = f_outer_r / f_pixel_ratio;
	m_p_transform->f_MirrorCenter_x() = f_center_x;
	m_p_transform->f_MirrorCenter_y() = f_center_y;
	m_p_transform->f_AngularOffset() = f_angular_offset;
	m_p_transform->f_SideRatio() = m_f_image_ratio * f_pixel_ratio;
	m_p_transform->f_Pixel_Width() = m_f_texel_w;
	m_p_transform->f_Pixel_Height() = m_f_texel_h;
	// set simple params

	float f_scale = f_outer_r / f_mirror_radius_mm;
	// r_outer refers to outer radius in texture space
	// (second component of "mirror_area" vector)

	f_delta_mm *= f_scale, f_bottom_y_mm *= f_scale, f_top_y_mm *= f_scale;
	// scale delta and bottom and top projection cylinder bounds

	f_scale *= f_scale;
	// a and b are their respective 2nd powers so we have to scale them
	// accordingly with 2nd power of scaling coefficient

	f_a2_mm *= f_scale, f_b2_mm *= f_scale;
	// scale a and b

#if defined(_MSC_VER) && !defined(__MWERKS__) // msvc
	float F = sqrtf(f_a2_mm + f_b2_mm);
#else
	float F = sqrt(f_a2_mm + f_b2_mm);
#endif
	// calc F (mirror eccentricity, y-coordinate of mirrors focus)

	// convert from milimeters to texture-space, exactly as in documentation

	m_p_transform->f_MirrorGeom(0) = f_a2_mm / f_b2_mm;
	m_p_transform->f_MirrorGeom(1) = 1.0f / (f_delta_mm * f_delta_mm);
	m_p_transform->f_MirrorGeom(2) = 2.0f * F / f_delta_mm;
	m_p_transform->f_MirrorGeom(3) = -f_a2_mm - f_b2_mm;
	//
	m_p_transform->f_MirrorGeom2(0) = f_a2_mm;
	m_p_transform->f_MirrorGeom2(1) = F;
	m_p_transform->f_MirrorGeom2(2) = f_b2_mm / f_a2_mm;
	m_p_transform->f_MirrorGeom2(3) = f_b2_mm * f_b2_mm / f_a2_mm;
	//
	m_p_transform->f_Bottom_Y() = f_bottom_y_mm;
	m_p_transform->f_Cyl_Height() = f_top_y_mm - f_bottom_y_mm;
	// set extra geometric transformation params

	return true;
}
// call for geom transform only, otherwise returns false
// parameters followed by "_mm" can be passed in milimeters

int CTrUt::n_ImgProc_Num()
{
	__FuncGuard("CTrUt::n_ImgProc_Num");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	return m_n_source_img_proc_num;
}
// number of loaded image operations

int CTrUt::n_ImgProc(int n_sip_index)
{
	__FuncGuard("CTrUt::n_ImgProc");

	if(!m_b_ready || n_sip_index < 0 || n_sip_index >= m_n_source_img_proc_num) {
		m_n_last_error = trut_Error;
		return false;
	}

	return m_p_source_img_proc[n_sip_index];
}
// types of loaded image operations

int CTrUt::b_MultipassImgProc()
{
	__FuncGuard("CTrUt::b_MultipassImgProc");

	if(!m_b_ready || !m_n_source_img_proc_num) {
		m_n_last_error = trut_Error;
		return false;
	}

	return m_b_multipass_sip;
}

// return true if we specified b_multipas_source_img_proc true in Init()

int CTrUt::ProcessSourceImage(int n_sip_index, int b_upside_down, unsigned int n_source_texture)
{
	__FuncGuard("CTrUt::ProcessSourceImage");

	if(!m_b_ready || n_sip_index < 0 || n_sip_index >= m_n_source_img_proc_num) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(!m_p_render_buffer_source_img->Bind() ||
	   !m_p_render_buffer_source_img->BindRenderTexture(m_n_processed_src_tex) ||
	   !m_p_render_buffer_source_img->b_Status()) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// bind render buffer and target texture

	glViewport(0, 0, m_p_render_buffer_source_img->n_Width(),
		m_p_render_buffer_source_img->n_Height());
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	if(b_upside_down)
		glScalef(1, -1, 1);

	m_p_imgproc[n_sip_index].DrawUniformFullscreenQuad(n_source_texture, m_p_render_buffer_source_img->p_State());
	// do the transform

	if(b_upside_down)
		glLoadIdentity();

	if((!m_p_render_buffer_source_img->b_DirectRenderToTexture() &&
	   !m_p_render_buffer_source_img->CopyTextureData(GL_TEXTURE_2D))) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// release offscreen render-buffer ... by now filtered source image is in the texure

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(m_b_fence_support)
		glSetFenceNV(m_p_fence_id_list[3], GL_ALL_COMPLETED_NV);
	// set marker ("fence") so we can query how far are we with completion of preceding commands

	return true;
}

int CTrUt::Swap_ProcessedSource_Textures()
{
	__FuncGuard("CTrUt::Swap_ProcessedSource_Textures");

	if(!m_b_ready || !m_n_source_img_proc_num || !m_b_multipass_sip) {
		m_n_last_error = trut_Error;
		return false;
	}

	unsigned int n_tmp = m_n_processed_src_tex;
	m_n_processed_src_tex = m_n_aux_processed_src_tex;
	m_n_aux_processed_src_tex = n_tmp;

	return true;
}

int CTrUt::Swap_Source_ProcessedSource()
{
	__FuncGuard("CTrUt::Swap_ProcessedSource_Textures");

	if(!m_b_ready || !m_n_source_img_proc_num || !m_b_multipass_sip) {
		m_n_last_error = trut_Error;
		return false;
	}

	unsigned int n_tmp = m_n_processed_src_tex;
	m_n_processed_src_tex = m_n_src_tex;
	m_n_src_tex = n_tmp;

	return true;
}

int CTrUt::Download_ProcessedSourceImage(ImageStruct *p_image)
{
	__FuncGuard("CTrUt::Download_ProcessedSourceImage");

	if(!m_b_ready || !m_n_source_img_proc_num) {
		m_n_last_error = trut_Error;
		return false;
	}

	WaitUntil_SIPFinished();

	int n_pixel_fmt, n_pixel_datatype;
	if(!ISPixelType_to_OGLPixelType(p_image->PixelType, n_pixel_fmt, n_pixel_datatype))
		return false;
	// get pixel format

	// simply copy our rectangle ...
	glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
	glPixelStorei(GL_PACK_SKIP_PIXELS, 0);

	glReadPixels(0, // x-offset
				 0, // y-offset
				 p_image->XSize, // width
				 p_image->YSize, // height
				 n_pixel_fmt,
				 n_pixel_datatype,
				 p_image->Raster + ((p_image->YOffset < 0)? p_image->YOffset *
				 (p_image->YSize - 1) : 0));
	// read pixels into the pixel-buffer (effectively only relocate buffer
	// onto their address, since buffer is read-only)

	if(!m_p_render_buffer_source_img->ReleaseRenderTexture() ||
	   !m_p_render_buffer_source_img->Release()) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	return true;
}

int CTrUt::Transform(int n_quad_tesselation, int b_upside_down)
{
	__FuncGuard("CTrUt::Transform");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	m_n_transform_state = 0;
	// transformation is running. as soon as we need the results,
	// we call glFinish(); and set transform state to 1

	if(!m_p_render_buffer->Bind() ||
	   !m_p_render_buffer->BindRenderTexture(m_n_dst_tex)) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// bind render buffer and target texture

	glViewport(0, 0, m_p_render_buffer->n_Width(), m_p_render_buffer->n_Height());
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	if(b_upside_down) {
		//glMatrixMode(GL_TEXTURE);
		glScalef(1, -1, 1);
	}

	if(n_quad_tesselation > 1) {
		if(!m_p_transform->DrawFullscreenQuad(m_n_src_tex,
		   n_quad_tesselation, m_p_render_buffer->p_State())) {
			m_n_last_error = trut_GLError;
		    return false;
		}
	} else
		m_p_transform->DrawFullscreenQuad(m_n_src_tex, m_p_render_buffer->p_State());
	// do the transform

	if(b_upside_down) {
		glLoadIdentity();
		//glMatrixMode(GL_MODELVIEW);
	}

	if(m_b_fence_support)
		glSetFenceNV(m_p_fence_id_list[1], GL_ALL_COMPLETED_NV);
	// set marker ("fence") so we can query how far are we with completion of preceding commands

	return true;
}
// begin transforming image, process is CPU-independent, after the call, you can do some usefull
// stuff and let graphics card work. function switches drawing context to offscreen buffer

int CTrUt::Download_ImageStruct_Async(ImageStruct *p_image, float f_angle, int n_y)
{
	__FuncGuard("CTrUt::Download_ImageStruct_Async");

	if(!m_b_ready || !m_b_async_support) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(m_b_fence_support && !m_n_transform_state)
		WaitUntil_TransformFinished(); // set transform state automatically
	else if(!m_n_transform_state) {
		glFlush(); // force all operations to be finished in finite time
		glFinish(); // wait untill all oparations had finished
		m_n_transform_state = 1;
	}
	// force opengl to finish transform (wait until it's finished)

	f_angle -= m_p_transform->f_AngularOffset();
#if defined(_MSC_VER) && !defined(__MWERKS__) // msvc
	f_angle = fmodf(f_angle / (2.0f * 3.1415926535897932384626433832795028841971f), 1.0f);
#else
	f_angle = (float)fmod(f_angle / (2.0 * 3.1415926535897932384626433832795028841971), 1.0);
#endif
	// fmod f_angle, set to be number between 0 and 1

	int n_x = m_p_render_buffer->n_Width() * f_angle - p_image->XSize / 2;
	// calculate coordinates of left border of required rectangle

	int n_pixel_fmt, n_pixel_datatype;
	if(!ISPixelType_to_OGLPixelType(p_image->PixelType, n_pixel_fmt, n_pixel_datatype))
		return false;
	// get pixel format

	int n_bytes;
	n_bytes = p_image->XOffset; // here should be the same number just away
	// determine how much bytes per pixel image takes

	if(p_image->XSize > m_p_render_buffer->n_Width()) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(n_x + p_image->XSize >= m_p_render_buffer->n_Width())
		n_x = -(m_p_render_buffer->n_Width() - n_x + 1);
	// wrap arround image so when it crosses any boundary, n_x is always negative

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	//// PBO stuff

	unsigned int n_next_buffer = (m_n_cur_pbo + 1) % n_pbo_num;
	// kick of readback of current front-buffer into the next buffer

    glBindBufferARB(GL_PIXEL_PACK_BUFFER_EXT, m_p_pbo_list[n_next_buffer]);

	//// ~PBO stuff

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	if(n_x < 0) {
		glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
		glPixelStorei(GL_PACK_SKIP_PIXELS, -n_x - 1); // skip 3 first pixels on each image scanline

		glReadPixels(0, // x-offset
					 n_y, // y-offset
					 p_image->XSize + n_x + 1,
					 p_image->YSize, // height
					 n_pixel_fmt,
					 n_pixel_datatype,
					 (char*)NULL); // copy 2 pixels on each scanline

		glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
		glPixelStorei(GL_PACK_SKIP_PIXELS, 0);
		// we're copying into the left part of the image -> skip 0 pixels

		glReadPixels(m_p_render_buffer->n_Width() + n_x + 1, // x-offset
					 n_y, // y-offset
					 -n_x, // width
					 p_image->YSize, // height
					 n_pixel_fmt,
					 n_pixel_datatype,
					 (char*)NULL); // copy 3 pixels from the end of the image

		//glPixelStorei(GL_PACK_ROW_LENGTH, -1);
		// fixme - for how long does GL_PACK_ROW_LENGTH apply? // f_ixme - until now; nope
	} else {
		// simply copy our rectangle ...
		glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
		glPixelStorei(GL_PACK_SKIP_PIXELS, 0);

		glReadPixels(n_x, // x-offset
					 n_y, // y-offset
					 p_image->XSize, // width
					 p_image->YSize, // height
					 n_pixel_fmt,
					 n_pixel_datatype,
					 (char*)NULL);
	}
	// read pixels into the pixel-buffer (effectively only relocate buffer
	// onto their address, since buffer is read-only)

	//// PBO stuff

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

    glBindBufferARB(GL_PIXEL_PACK_BUFFER_EXT, m_p_pbo_list[m_n_cur_pbo]);
	// map the current buffer containing the image read back the previous time arround

	void *p_buffer = glMapBufferARB(GL_PIXEL_PACK_BUFFER_EXT, GL_READ_ONLY_ARB);
	if(p_buffer == NULL) {
		m_n_last_error = trut_GLError;
		return false;
	}
    memcpy(p_image->Raster + ((p_image->YOffset < 0)? p_image->YOffset *
		(p_image->YSize - 1) : 0), p_buffer, p_image->XSize * p_image->YSize * p_image->XOffset);
	/*glGetBufferSubDataARB(GL_PIXEL_PACK_BUFFER_EXT, 0,
		p_image->XSize * p_image->YSize * p_image->XOffset, p_image->Raster +
		((p_image->YOffset < 0)? p_image->YOffset * (p_image->YSize - 1) : 0));*/
	// copy the image from mapped buffer (data were transferd from card memory into the system
	// memory, now only memcpy from one place into another; not sure how fast is this going to
	// be, better would be to use this area right away. but not sure if it wouldn't block gpu)

    if(!glUnmapBufferARB(GL_PIXEL_PACK_BUFFER_EXT)) {
        m_n_last_error = trut_GLError;
		return false;
    }
    // unmap the buffer

    glBindBufferARB(GL_PIXEL_PACK_BUFFER_EXT, 0);
	// unbind readback buffer to not interfere with any other (traditional) readbacks.

    m_n_cur_pbo = n_next_buffer;
	// make next-buffer the current buffer

	//// ~PBO stuff

	if(m_b_fence_support)
		glSetFenceNV(m_p_fence_id_list[2], GL_ALL_COMPLETED_NV);
	// set marker ("fence") so we can query how far are we with completion of preceding commands

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	return true;
}

int CTrUt::b_UploadFinished()
{
	__FuncGuard("CTrUt::b_UploadFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	return (m_b_fence_support)? glTestFenceNV(m_p_fence_id_list[0]) : true;
}

int CTrUt::b_SIPFinished()
{
	__FuncGuard("CTrUt::b_SIPFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	return (m_b_fence_support)? glTestFenceNV(m_p_fence_id_list[3]) : true;
}

int CTrUt::b_TransformFinished()
{
	__FuncGuard("CTrUt::b_TransformFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	if((m_b_fence_support)? glTestFenceNV(m_p_fence_id_list[1]) : true) {
		m_n_transform_state = 1; // transform has finished
		return true;
	}
	return false;
}

int CTrUt::b_DownloadFinished()
{
	__FuncGuard("CTrUt::b_DownloadFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	return (m_b_fence_support)? glTestFenceNV(m_p_fence_id_list[2]) : true;
}

void CTrUt::WaitUntil_UploadFinished()
{
	__FuncGuard("CTrUt::WaitUntil_UploadFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return;
	}

	if(m_b_fence_support)
		glFinishFenceNV(m_p_fence_id_list[0]);
	else {
		glFlush();
		glFinish();
	}
}

void CTrUt::WaitUntil_SIPFinished()
{
	__FuncGuard("CTrUt::WaitUntil_SIPFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return;
	}

	if(m_b_fence_support)
		glFinishFenceNV(m_p_fence_id_list[3]);
	else {
		glFlush();
		glFinish();
	}
}

void CTrUt::WaitUntil_TransformFinished()
{
	__FuncGuard("CTrUt::WaitUntil_TransformFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return;
	}

	if(m_b_fence_support)
		glFinishFenceNV(m_p_fence_id_list[1]);
	else {
		glFlush();
		glFinish();
	}

	m_n_transform_state = 1; // transform finished
}

void CTrUt::WaitUntil_DownloadFinished()
{
	__FuncGuard("CTrUt::WaitUntil_DownloadFinished");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return;
	}

	if(m_b_fence_support)
		glFinishFenceNV(m_p_fence_id_list[2]);
	else {
		glFlush();
		glFinish();
	}
}

int CTrUt::Download_ImageStruct(ImageStruct *p_image, float f_angle, int n_y)
{
	__FuncGuard("CTrUt::Download_ImageStruct");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(m_b_fence_support && !m_n_transform_state)
		WaitUntil_TransformFinished(); // set transform state automatically
	else if(!m_n_transform_state) {
		glFlush(); // force all operations to be finished in finite time
		glFinish(); // wait untill all oparations had finished
		m_n_transform_state = 1;
	}
	// force opengl to finish transform (wait until it's finished)

	f_angle -= m_p_transform->f_AngularOffset();
#if defined(_MSC_VER) && !defined(__MWERKS__) // msvc
	f_angle = fmodf(f_angle / (2.0f * 3.1415926535897932384626433832795028841971f), 1.0f);
#else
	f_angle = (float)fmod(f_angle / (2.0 * 3.1415926535897932384626433832795028841971), 1.0);
#endif
	// fmod f_angle, set to be number between 0 and 1

	int n_x = m_p_render_buffer->n_Width() * f_angle - p_image->XSize / 2;
	// calculate coordinates of left border of required rectangle

	int n_pixel_fmt, n_pixel_datatype;
	if(!ISPixelType_to_OGLPixelType(p_image->PixelType, n_pixel_fmt, n_pixel_datatype))
		return false;
	// get pixel format

	int n_bytes;
	/*switch(n_pixel_datatype) {
	case GL_BYTE:
	case GL_UNSIGNED_BYTE:
		n_bytes = 1;
		break;
	case GL_SHORT:
	case GL_UNSIGNED_SHORT:
		n_bytes = 2;
		break;
	case GL_INT:
	case GL_UNSIGNED_INT:
		n_bytes = 4;
		break;
	case GL_FLOAT:
		n_bytes = 4;
		break;
	default:
		m_n_last_error = trut_BadPixelType;
		return false;
	}
	if(p_image->PixelType == ImageRGB || p_image->PixelType == ImageRGBLinear)
		n_bytes *= 4;*/
	n_bytes = p_image->XOffset; // here should be the same number just away
	// determine how much bytes per pixel image takes

	if(p_image->XSize > m_p_render_buffer->n_Width()) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(n_x + p_image->XSize >= m_p_render_buffer->n_Width()) {
		n_x = -(m_p_render_buffer->n_Width() - n_x + 1);
        /*
                01234567890123 <- m_p_render_buffer->n_Width() = 13
                           012 34 ... n_x = -3, XSize = 5
               +--------------+
               +-+         +--|-+
               | |         |  | |
               | |         |  | |
               | |         |  | |
               +-+         +--|-+
               +--------------+
                           ^- n_x = 11

            -(13 - 11 + 1) = -3!
        */
	}
	// wrap arround image so when it crosses any boundary, n_x is always negative

	if(n_x < 0) {
		// copy pixels 0 .. n_x + p_image->XSize from the beggining of frame

        /*
                01234567890123
            012 34 ... n_x = -3, XSize = 5
               +--------------+
            +--+-+         +--|
            |  | |         |  |
            |  | |         |  |
            |  | |         |  |
            +--+-+         +--|
             ^ +--------------+
             |  ^     ^- m_p_render_buffer->n_Width() = 13
             |  |
             |  +- p_image->XSize + n_x = 2
             +- -n_x = 3
        */

		glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
		glPixelStorei(GL_PACK_SKIP_PIXELS, -n_x - 1); // skip 3 first pixels on each image scanline

		if(p_image->YOffset < 0) {
			glReadPixels(0, // x-offset
						 n_y, // y-offset
						 p_image->XSize + n_x + 1,
						 p_image->YSize, // height
						 n_pixel_fmt,
						 n_pixel_datatype,
						 p_image->Raster + p_image->YOffset * (p_image->YSize - 1)); // copy 2 pixels on each scanline
		} else {
			glReadPixels(0, // x-offset
						 n_y, // y-offset
						 p_image->XSize + n_x + 1,
						 p_image->YSize, // height
						 n_pixel_fmt,
						 n_pixel_datatype,
						 p_image->Raster); // copy 2 pixels on each scanline
		}

		glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
		glPixelStorei(GL_PACK_SKIP_PIXELS, 0);
		// we're copying into the left part of the image -> skip 0 pixels

		if(p_image->YOffset < 0) {
			glReadPixels(m_p_render_buffer->n_Width() + n_x + 1, // x-offset - "+1" is correct
						 n_y, // y-offset
						 -n_x, // width
						 p_image->YSize, // height
						 n_pixel_fmt,
						 n_pixel_datatype,
						 p_image->Raster + p_image->YOffset * (p_image->YSize - 1)); // copy 3 pixels from the end of the image
		} else {
			glReadPixels(m_p_render_buffer->n_Width() + n_x + 1, // x-offset
						 n_y, // y-offset
						 -n_x, // width
						 p_image->YSize, // height
						 n_pixel_fmt,
						 n_pixel_datatype,
						 p_image->Raster); // copy 3 pixels from the end of the image
		}

		//glPixelStorei(GL_PACK_ROW_LENGTH, -1);
		// fixme - for how long does GL_PACK_ROW_LENGTH apply? // f_ixme - until now; nope
	} else {
		// simply copy our rectangle ...
		glPixelStorei(GL_PACK_ROW_LENGTH, p_image->XSize);
		glPixelStorei(GL_PACK_SKIP_PIXELS, 0);

		if(p_image->YOffset < 0) {
			glReadPixels(n_x, // x-offset
						 n_y, // y-offset
						 p_image->XSize, // width
						 p_image->YSize, // height
						 n_pixel_fmt,
						 n_pixel_datatype,
						 p_image->Raster + p_image->YOffset * (p_image->YSize - 1));
		} else {
			glReadPixels(n_x, // x-offset
						 n_y, // y-offset
						 p_image->XSize, // width
						 p_image->YSize, // height
						 n_pixel_fmt,
						 n_pixel_datatype,
						 p_image->Raster);
		}
	}

	if(glGetError() != GL_NO_ERROR) {
		m_n_last_error = trut_GLError;
		return false;
	}

	return true;
}
// download image from current bound framebuffer. p_image is already allocated,
// f_angle is angle in radians (fmod-ed to interval 0 .. 2Pi), specifying angle where mid
// of the image rectangle should lie, n_y is vertical position (of top border) in raster
// when angle is near 0 or 2Pi so image rectangle crosses left or right border of unfolded image,
// it is wrapped over (similar to GL_REPEAT texture clamp mode), when image rectangle crosses
// bottom or tom border of unfolded image, it's part, lying outside isn't modified

int CTrUt::ReleaseFramebuffer()
{
	__FuncGuard("CTrUt::ReleaseFramebuffer");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return false;
	}

	if(!m_n_transform_state)
		WaitUntil_TransformFinished();
	// force opengl to finish transform (wait until it's finished)

	if(!m_p_render_buffer->b_DirectRenderToTexture() &&
	   !m_p_render_buffer->CopyTextureData(GL_TEXTURE_2D)) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// copy texture data if necessary

	if(!m_p_render_buffer->ReleaseRenderTexture() ||
	   !m_p_render_buffer->Release()) {
		m_n_last_error = trut_GLError;
		return false;
	}
	// release texture and buffer

	return true;
}
// if you want to draw to your opengl window, it's necessary to un-bind transformation's
// off-screen framebuffer. however, it's not necessary before next Transform() so if you're
// not going to use open-gl between transformations, you don't have to call this

unsigned int CTrUt::n_TransformedTexture_OpenGL_Id()
{
	__FuncGuard("CTrUt::n_TransformedTexture_OpenGL_Id");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return 0;
	}

	return m_n_dst_tex;
}
// return OpenGL handle of texture, containing transformed image
// can be freely used for displaying results on screen, bud
// must not be deleted

unsigned int CTrUt::n_SrcTexture_OpenGL_Id()
{
	__FuncGuard("CTrUt::n_TransformedTexture_OpenGL_Id");

	if(!m_b_ready) {
		m_n_last_error = trut_Error;
		return 0;
	}

	return m_n_src_tex;
}

unsigned int CTrUt::n_SrcProcessedTexture_OpenGL_Id()
{
	__FuncGuard("CTrUt::n_TransformedTexture_OpenGL_Id");

	if(!m_b_ready || !m_n_source_img_proc_num) {
		m_n_last_error = trut_Error;
		return 0;
	}

	return m_n_processed_src_tex;
}

unsigned int CTrUt::n_SrcProcessedAuxTexture_OpenGL_Id()
{
	__FuncGuard("CTrUt::n_TransformedTexture_OpenGL_Id");

	if(!m_b_ready || !m_n_source_img_proc_num || !m_b_multipass_sip) {
		m_n_last_error = trut_Error;
		return 0;
	}

	return m_n_aux_processed_src_tex;
}

int CTrUt::ShutDown()
{
	__FuncGuard("CTrUt::ShutDown");

	m_b_ready = false;
	if(m_n_src_tex) {
		glDeleteTextures(1, &m_n_src_tex);
		m_n_src_tex = 0;
	}
	if(m_n_processed_src_tex) {
		glDeleteTextures(1, &m_n_processed_src_tex);
		m_n_processed_src_tex = 0;
	}
	if(m_n_aux_processed_src_tex) {
		glDeleteTextures(1, &m_n_aux_processed_src_tex);
		m_n_aux_processed_src_tex = 0;
	}
	if(m_n_dst_tex) {
		glDeleteTextures(1, &m_n_dst_tex);
		m_n_dst_tex = 0;
	}
	if(m_p_transform) {
		delete m_p_transform;
		m_p_transform = 0;
	}
	if(m_p_imgproc) {
		delete m_p_imgproc;
		m_p_imgproc = 0;
	}
	if(m_p_render_buffer_source_img) {
		delete m_p_render_buffer_source_img;
		m_p_render_buffer_source_img = 0;
	}
	if(m_p_render_buffer) {
		delete m_p_render_buffer;
		m_p_render_buffer = 0;
	}
	if(m_p_s_sh_info_log) {
		delete m_p_s_sh_info_log;
		m_p_s_sh_info_log = 0;
	}
	if(m_p_gl_driver) {
		delete m_p_gl_driver;
		m_p_gl_driver = 0;
	}
	if(m_b_fence_support && m_p_fence_id_list[0] != 0) {
		glDeleteFencesNV(3 + (m_n_source_img_proc_num > 0), m_p_fence_id_list);
		m_p_fence_id_list[0] = 0;
	}
	if(m_b_async_support && m_p_pbo_list[0] != 0) {
		glDeleteBuffersARB(3, m_p_pbo_list);
		glDeleteBuffersARB(1, &m_n_uploader_pbo);
		/*if(m_n_source_img_proc != sip_None)
			glDeleteBuffersARB(1, &m_n_sipdown_pbo);*/
		m_p_pbo_list[0] = 0;
	}
	if(m_p_source_img_proc) {
		delete[] m_p_source_img_proc;
		m_p_source_img_proc = 0;
	}

	return true;
}
// free all open-gl resources, b_Ready() returns false,

/*
 *	int CTrUt::std_Load_RenderPaths(std::vector<TPath*> &r_path_list)
 *		- loat std render path list
 *		- you can use this function in your own Load_RenderPaths function,
 *		  then add your own paths
 */
int CTrUt::std_Load_RenderPaths(std::vector<TPath*> &r_path_list)
{
	__FuncGuard("CTrUt::std_Load_RenderPaths");

	return __Load_RenderPaths(r_path_list);
}

/*
 *	int CTrUt::std_Load_Shaders(std::vector<CShaderInfo*> &r_shader_list,
 *		const char *p_s_path_name)
 *		- load std shaders, falling under the path p_s_path_name
 *		- you can use this function in your own Load_Shaders function,
 *		  then add your own paths
 */
int CTrUt::std_Load_Shaders(std::vector<CShaderInfo*> &r_shader_list,
	const char *p_s_path_name)
{
	__FuncGuard("CTrUt::std_Load_Shaders");

	return __Load_ShaderInfo(r_shader_list, p_s_path_name);
}

// here you can get standard shaders (you can also dig them out this way and rewrite
// them to satisfy your needs)

/*
 *	int CTrUt::Init_OpenGL_Exts()
 *		- get open-gl extension pointers, show message box on errors
 *		- return true if succeeded, false if not
 */
int CTrUt::Init_OpenGL_Exts()
{
	__FuncGuard("CTrUt::Init_OpenGL_Exts");

	if(!CGLExtensionHandler::b_Support_OpenGL(1, 3)) {
		m_n_last_error = trut_NoGL13;
		return false;
	}
	if(CGLExtensionHandler::n_GetMultitextureFuncPointers()) {
		m_n_last_error = trut_NoMultitextureExt;
		return false;
	}
	CGLExtensionHandler::n_GetFragmentProgramARBFuncPointers();
	CGLExtensionHandler::n_GetFragmentShaderARBFuncPointers();
	CGLExtensionHandler::n_GetFramebufferObjectEXTFuncPointers();
	CGLExtensionHandler::n_GetShaderObjectARBFuncPointers();
	CGLExtensionHandler::n_GetVertexProgramARBFuncPointers();
	CGLExtensionHandler::n_GetVertexShaderARBFuncPointers();
	CGLExtensionHandler::n_GetGL12FuncPointers();
	CGLExtensionHandler::n_GetGL13FuncPointers();
	CGLExtensionHandler::n_GetGL14FuncPointers();
	CGLExtensionHandler::n_GetGL15FuncPointers();
	CGLExtensionHandler::n_GetGL20FuncPointers();
	if(!CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_program") &&
	   !(CGLExtensionHandler::b_SupportedExtension("GL_ARB_fragment_shader") &&
	   CGLExtensionHandler::b_SupportedExtension("GL_ARB_shading_language_100"))) {
		m_n_last_error = trut_NoShaderExt;
		return false;
	}
	// get open-gl function pointers

	return true;
}

/*
 *	int CTrUt::ISPixelType_to_OGLPixelType(int n_pixel_type, int &r_n_fmt, int &r_n_data_type)
 *		- conversion from ImageStruct pixel type to OpenGL pixel format and data type
 *		- return false for unsupported ImageStruct formats, otherwise true
 */
int CTrUt::ISPixelType_to_OGLPixelType(int n_pixel_type, int &r_n_fmt, int &r_n_data_type)
{
	r_n_fmt = GL_LUMINANCE; // 1 channel only, converted to red, 0, 0 when displayed
	r_n_data_type = GL_UNSIGNED_BYTE;

	switch(n_pixel_type) {
	case ImageRGB:
	case ImageRGBLinear:
		r_n_fmt = GL_BGRA; // DigiLib uses "RGB" name, but actual values are stored in byte quadruples
		// this case intentionaly falls trough
	case Image8:
	case Image8Linear:
		r_n_data_type = GL_UNSIGNED_BYTE;
		break;
	case Image16:
	case Image16Linear:
		r_n_data_type = GL_SHORT;
		// fixme - hope this is the same as for ripac data structs (byte is unsigned, bigger numbers are signed)
		break;
	case Image32:
	case Image32Linear:
		r_n_data_type = GL_INT;
		break;
	case ImageFloat:
	case ImageFloatLinear:
		r_n_data_type = GL_FLOAT;
		break;
	case ImageComplex:
	case ImageComplexLinear:
		m_n_last_error = trut_BadPixelType;
		return false; // no support for complex numbers (would have to pass as RG float)
	default:
		m_n_last_error = trut_BadPixelType;
		return false;
	}

	return true;
}

/*
 *	unsigned int CTrUt::CreateTexture_2D(int n_width, int n_height, GLenum n_format,
 *		int b_create_mipmaps, void *p_data, GLenum n_data_format, GLenum n_data_type)
 *		- create open-gl 2D texture (hope params are self-explanatory enough)
 *		- return 0 on failure
 *		- otherwise return opengl texture id
 */
unsigned int CTrUt::CreateTexture_2D(int n_width, int n_height, GLenum n_format,
	int b_create_mipmaps, void *p_data, GLenum n_data_format, GLenum n_data_type)
{
	unsigned int n_open_gl_id = 0;

	glEnable(GL_TEXTURE_2D);
	glGenTextures(1, &n_open_gl_id);
	glBindTexture(GL_TEXTURE_2D, n_open_gl_id);
	// create a new texture and activate it

	if(b_create_mipmaps)
		glTexParameteri(GL_TEXTURE_2D, GL_GENERATE_MIPMAP, GL_TRUE);
	//
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
		(b_create_mipmaps)? GL_LINEAR_MIPMAP_LINEAR : GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	// we don't have mip-maps yet

	glTexImage2D(GL_TEXTURE_2D, 0, n_format,
		n_width, n_height, 0, n_data_format, n_data_type, p_data);
	// load texture data

	if(glGetError() != GL_NO_ERROR) {
		glDeleteTextures(1, &n_open_gl_id);
		m_n_last_error = trut_GLError;
		return 0;
	}
	// see if we had enough memory

	return n_open_gl_id;
}

/*
 *	int CTrUt::b_Have_FBO()
 *		- return true if current instance of open-gl window was given framebuffer object support
 */
int CTrUt::b_Have_FBO()
{
	return !CGLExtensionHandler::n_GetFramebufferObjectEXTFuncPointers() &&
		CGLExtensionHandler::b_SupportedExtension("GL_EXT_framebuffer_object");
}

/*
 *	int CTrUt::b_Have_PBuffers()
 *		- return true if current instance of open-gl window was given pixel-buffer support
 *		- note pixel-buffer is wgl extension and thus is available under windows only
 *		  (fbo is multiplatform, but not so wide supported)
 */
int CTrUt::b_Have_PBuffers()
{
	return CGLExtensionHandler::b_SupportedWGLExtension("WGL_ARB_pbuffer") &&
		   CGLExtensionHandler::b_SupportedWGLExtension("WGL_ARB_pixel_format") &&
		   wglGetProcAddress("wglGetPixelFormatAttribivARB") &&
		   wglGetProcAddress("wglChoosePixelFormatARB") &&
		   wglGetProcAddress("wglCreatePbufferARB") &&
		   wglGetProcAddress("wglGetPbufferDCARB") &&
		   wglGetProcAddress("wglReleasePbufferDCARB") &&
		   wglGetProcAddress("wglDestroyPbufferARB") &&
		   wglGetProcAddress("wglQueryPbufferARB");
}

/*
 *	CGLRenderBuffer *CTrUt::p_Create_RenderBuffer(int n_width, int n_height, int n_format,
 *		int n_color_bits, int b_float, int n_texture_target = 0, int b_mipmap = false,
 *		int b_depth_texture_target = false, int b_double_buffer = false, int b_depth_buffer = false,
 *		int n_depth_bits = 32, int b_stencil_buffer = false, int n_stencil_bits = 8)
 *		- create open-gl buffer object handler for offscreen rendering
 *		- also - parameters should be self-explanatory
 *		- return 0 when not enough memory or requested buffer isn't supported
 *		- otherwise return pointer to the buffer handler class object
 */
CGLRenderBuffer *CTrUt::p_Create_RenderBuffer(int n_width, int n_height, int n_format,
	int n_color_bits, int b_float, int n_texture_target, int b_mipmap,
	int b_depth_texture_target, int b_double_buffer, int b_depth_buffer, int n_depth_bits,
	int b_stencil_buffer, int n_stencil_bits)
{
	for(int i = 0; i < 2; i ++) {
		CGLRenderBuffer *p_render_buffer;

		if(i) {
			if(b_Have_PBuffers()) {
				p_render_buffer = new CGLRenderBuffer_PBuffer(n_width, n_height, n_format,
					n_color_bits, b_float, n_texture_target, b_mipmap, b_depth_texture_target,
					b_double_buffer, b_depth_buffer, n_depth_bits,
					b_stencil_buffer, n_stencil_bits);
			} else
				continue;
		} else {
			if(b_Have_FBO()) {
				p_render_buffer = new CGLRenderBuffer_FBO(n_width, n_height, n_format,
					n_color_bits, b_float, n_texture_target, b_mipmap, b_depth_texture_target,
					b_double_buffer, b_depth_buffer, n_depth_bits,
					b_stencil_buffer, n_stencil_bits);
			} else
				continue;
		}

		if(!p_render_buffer) {
			m_n_last_error = trut_NoMemory;
		} else if(!p_render_buffer->b_Supported() || !p_render_buffer->b_Status())
			delete p_render_buffer;
		else
			return p_render_buffer;
	}

	m_n_last_error = trut_NoRenderBuffer;
	return 0;
}

/*
 *								=== ~CTrUt ===
 */

/*
 *		-end-of-file-
 */
