/*
								+---------------------------------+
								|                                 |
								|   ***   OGL state guard   ***   |
								|                                 |
								|  Copyright   -tHE SWINe- 2005  |
								|                                 |
								|           GlState.cpp           |
								|                                 |
								+---------------------------------+
*/

#include <windows.h>
#include <crtdbg.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <gl/gl.h>
#include <vector>
//
#include "glext.h"
#include "wglext.h"
//
//#include "OpenGlDrv.h"
#include "OpenGl20.h"

#include "glstate.h"
#include "callstack.h"

/*
 *								=== ~CGLState ===
 */

#define __MULTITEXTURE_ENABLED

std::vector<TGLState*> *CGLState::m_p_state = 0;
int CGLState::m_n_state_refs = 0;

CGLState::CGLState(HDC h_dc, HGLRC h_glrc)
{
	__FuncGuard("CGLState::CGLState");

	TGLState *p_my_state;
	HGLRC h_prev_glrc;
	HDC h_prev_dc;
	int i;

	m_h_dc = 0;
	m_h_glrc = 0;

	if(!m_p_state) {
		if(!(m_p_state = new std::vector<TGLState*>))
			return;
	}
	m_n_state_refs ++;
	// first class

	m_h_dc = h_dc;
	m_h_glrc = h_glrc;

	for(i = 0; i < m_p_state->size(); i ++) {
		if((*m_p_state)[i]->h_glrc == h_glrc) {
			m_p_cur_state = (*m_p_state)[i];
			m_p_cur_state->n_references ++;

			break;
		}
	}
	if(i == m_p_state->size()) {
		if(!(p_my_state = new TGLState)) {
			m_b_init = false;
			return;
		}

		p_my_state->h_glrc = m_h_glrc;
		p_my_state->n_references = 1;
		//
		m_p_state->push_back(p_my_state);
		_ASSERTE(m_p_state->size());
		m_p_cur_state = (*m_p_state)[m_p_state->size() - 1];
		//
		h_prev_dc = wglGetCurrentDC();
		h_prev_glrc = wglGetCurrentContext();
		wglMakeCurrent(m_h_dc, m_h_glrc);
		//
		Update_States();
		//
		wglMakeCurrent(h_prev_dc, h_prev_glrc);
		// create my state and store current opengl state isnide
	}
	// try to find state that was already set for this context

	m_b_init = true;
}

CGLState::~CGLState()
{
	__FuncGuard("CGLState::~CGLState");

	if(m_b_init && !(-- m_n_state_refs) && m_p_state) {
		for(int i = 0; i < m_p_state->size(); i ++)
			delete (*m_p_state)[i];
		delete m_p_state;
		m_p_state = 0;
	}
	m_b_init = false;
}

void CGLState::Update_States()
{
	__FuncGuard("CGLState::Update_States");

	char *p_s_ext;
	int i;

	p_s_ext = (char*)glGetString(GL_EXTENSIONS);

	glGetFloatv(GL_CURRENT_COLOR, m_p_cur_state->f_color);
	for(i = 0; i < 4; i ++)
		m_p_cur_state->n_color[i] = (unsigned char)(m_p_cur_state->f_color[i] / 255.0f);
	// color

	glGetIntegerv(GL_DEPTH_WRITEMASK, &m_p_cur_state->b_depth_write);
	glGetIntegerv(GL_COLOR_WRITEMASK, m_p_cur_state->b_color_write);
	// write masks

	glGetFloatv(GL_COLOR_CLEAR_VALUE, m_p_cur_state->f_clr_color);
	glGetFloatv(GL_DEPTH_CLEAR_VALUE, &m_p_cur_state->f_clr_depth);
	// clear color

	m_p_cur_state->b_blend = glIsEnabled(GL_BLEND);
	glGetIntegerv(GL_BLEND_SRC, &m_p_cur_state->n_blend_func[0]);
	glGetIntegerv(GL_BLEND_DST, &m_p_cur_state->n_blend_func[1]);
	// blending

	glGetIntegerv(GL_FRONT_FACE, &m_p_cur_state->n_front_face);
	m_p_cur_state->b_cull_face = glIsEnabled(GL_CULL_FACE);
	glGetIntegerv(GL_CULL_FACE_MODE, &m_p_cur_state->n_cull_face);
	// face culling

	m_p_cur_state->b_alpha_test = glIsEnabled(GL_ALPHA_TEST);
	glGetIntegerv(GL_ALPHA_TEST_FUNC, &m_p_cur_state->n_alpha_func);
	glGetFloatv(GL_ALPHA_TEST_REF, &m_p_cur_state->f_alpha_tresh);
	// alpha test

	m_p_cur_state->b_depth_test = glIsEnabled(GL_DEPTH_TEST);
	glGetIntegerv(GL_DEPTH_FUNC, &m_p_cur_state->n_depth_func);
	// depth test

#ifdef __MULTITEXTURE_ENABLED
	glGetIntegerv(GL_MAX_TEXTURE_UNITS, &m_p_cur_state->n_max_texture_units);
	glGetIntegerv(GL_ACTIVE_TEXTURE, &m_p_cur_state->n_active_tex_unit);
	m_p_cur_state->n_active_tex_unit -= GL_TEXTURE0;
#ifdef _DEBUG
	if(m_p_cur_state->n_max_texture_units > 32)
		_ASSERTE(0);
#endif
	//
	for(i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
		glActiveTexture(GL_TEXTURE0 + i);
		//
		m_p_cur_state->b_texture1d_enabled[i] = glIsEnabled(GL_TEXTURE_1D);
		m_p_cur_state->b_texture2d_enabled[i] = glIsEnabled(GL_TEXTURE_2D);
		m_p_cur_state->b_texture3d_enabled[i] = false;
		m_p_cur_state->n_texture_cubemap[i] = false; // nkterej hw tohle nemus podporovat
		m_p_cur_state->b_texture3d_enabled[i] = glIsEnabled(GL_TEXTURE_3D);
		m_p_cur_state->b_texture_cubemap_enabled[i] = glIsEnabled(GL_TEXTURE_CUBE_MAP);
		//
		glGetIntegerv(GL_TEXTURE_1D, &m_p_cur_state->n_texture1d[i]);
		glGetIntegerv(GL_TEXTURE_2D, &m_p_cur_state->n_texture2d[i]);
		glGetIntegerv(GL_TEXTURE_3D, &m_p_cur_state->n_texture3d[i]);
		glGetIntegerv(GL_TEXTURE_CUBE_MAP, &m_p_cur_state->n_texture_cubemap[i]);
	}
	glActiveTexture(m_p_cur_state->n_active_tex_unit + GL_TEXTURE0);
	// texture units
#else
	m_p_cur_state->n_max_texture_units = 1;
	m_p_cur_state->n_active_tex_unit = 0;
	//
	m_p_cur_state->b_texture1d_enabled[0] = glIsEnabled(GL_TEXTURE_1D);
	m_p_cur_state->b_texture2d_enabled[0] = glIsEnabled(GL_TEXTURE_2D);
	m_p_cur_state->b_texture3d_enabled[0] = false;
	m_p_cur_state->n_texture_cubemap[0] = false; // nkterej hw tohle nemus podporovat
	m_p_cur_state->b_texture3d_enabled[0] = glIsEnabled(GL_TEXTURE_3D);
	m_p_cur_state->n_texture_cubemap[0] = glIsEnabled(GL_TEXTURE_CUBE_MAP);
	//
	glGetIntegerv(GL_TEXTURE_1D, &m_p_cur_state->n_texture1d[0]);
	glGetIntegerv(GL_TEXTURE_2D, &m_p_cur_state->n_texture2d[0]);
	glGetIntegerv(GL_TEXTURE_3D, &m_p_cur_state->n_texture3d[0]);
	glGetIntegerv(GL_TEXTURE_CUBE_MAP, &m_p_cur_state->n_texture_cubemap[0]);
	// texture units
#endif

	m_p_cur_state->b_lighting = glIsEnabled(GL_LIGHTING);
	glGetIntegerv(GL_MAX_LIGHTS, &m_p_cur_state->n_max_lights);
#ifdef _DEBUG
	if(m_p_cur_state->n_max_lights > 32)
		_ASSERTE(0);
#endif
	for(i = 0; i < m_p_cur_state->n_max_lights; i ++) {
		m_p_cur_state->b_light[i] = glIsEnabled(GL_LIGHT0 + i);
		//
		glGetLightfv(GL_LIGHT0 + i, GL_AMBIENT, m_p_cur_state->t_light_data[i].f_ambient);
		glGetLightfv(GL_LIGHT0 + i, GL_DIFFUSE, m_p_cur_state->t_light_data[i].f_diffuse);
		glGetLightfv(GL_LIGHT0 + i, GL_SPECULAR, m_p_cur_state->t_light_data[i].f_specular);
		glGetLightfv(GL_LIGHT0 + i, GL_POSITION, m_p_cur_state->t_light_data[i].f_position);
		glGetLightfv(GL_LIGHT0 + i, GL_SPOT_DIRECTION,
			m_p_cur_state->t_light_data[i].f_spot_direction);
		glGetLightfv(GL_LIGHT0 + i, GL_SPOT_EXPONENT,
			&m_p_cur_state->t_light_data[i].f_spot_exponent);
		glGetLightfv(GL_LIGHT0 + i, GL_SPOT_CUTOFF,
			&m_p_cur_state->t_light_data[i].f_spot_cutoff);
		glGetLightfv(GL_LIGHT0 + i, GL_CONSTANT_ATTENUATION,
			&m_p_cur_state->t_light_data[i].f_constant_attenuation);
		glGetLightfv(GL_LIGHT0 + i, GL_LINEAR_ATTENUATION,
			&m_p_cur_state->t_light_data[i].f_linear_attenuation);
		glGetLightfv(GL_LIGHT0 + i, GL_QUADRATIC_ATTENUATION,
			&m_p_cur_state->t_light_data[i].f_quadratic_attenuation);
	}
	// lights

	m_p_cur_state->b_fog = glIsEnabled(GL_FOG);
	glGetIntegerv(GL_FOG_MODE, &m_p_cur_state->n_fog_type);
	glGetFloatv(GL_FOG_COLOR, m_p_cur_state->f_fog_color);
	glGetFloatv(GL_FOG_START, &m_p_cur_state->f_fog_start);
	glGetFloatv(GL_FOG_DENSITY, &m_p_cur_state->f_fog_density);
	glGetFloatv(GL_FOG_END, &m_p_cur_state->f_fog_end);
	// fog

	glGetIntegerv(GL_SHADE_MODEL, &m_p_cur_state->n_shade_model);
	// shade model

	glGetIntegerv(GL_POLYGON_MODE, m_p_cur_state->n_polygon_mode);
	glGetFloatv(GL_LINE_WIDTH, &m_p_cur_state->f_line_width);
	glGetFloatv(GL_POINT_SIZE, &m_p_cur_state->f_point_size);
	// rasterizer props

	for(int j = 0; j < m_p_cur_state->n_max_texture_units; j ++) {
		glActiveTexture(j + GL_TEXTURE0);
		for(i = 0; i < 4; i ++) {
			m_p_cur_state->b_texgen_enabled[j][i] = glIsEnabled(GL_TEXTURE_GEN_S + i);
			//
			glGetTexGeniv(GL_S + i, GL_TEXTURE_GEN_MODE, &m_p_cur_state->n_texgen_mode[j][i]);
			glGetTexGenfv(GL_S + i, GL_EYE_PLANE, m_p_cur_state->f_texgen_eye_plane[j][i]);
			glGetTexGenfv(GL_S + i, GL_OBJECT_PLANE, m_p_cur_state->f_texgen_object_plane[j][i]);
		}
	}
	glActiveTexture(m_p_cur_state->n_active_tex_unit + GL_TEXTURE0);
	// texgen props

	m_p_cur_state->b_polygon_offset_support =
		CGLExtensionHandler::b_SupportedExtension("GL_EXT_polygon_offset") &&
		!CGLExtensionHandler::n_GetPolygonOffsetEXTFuncPointers();
	//
	if(m_p_cur_state->b_polygon_offset_support) {
		m_p_cur_state->b_polygon_offset = glIsEnabled(GL_POLYGON_OFFSET_EXT);
		glGetFloatv(GL_POLYGON_OFFSET_FACTOR_EXT, &m_p_cur_state->f_polygon_offset_factor);
		glGetFloatv(GL_POLYGON_OFFSET_BIAS_EXT, &m_p_cur_state->f_polygon_offset_bias);
	} else
		m_p_cur_state->b_polygon_offset = false;
}

HDC CGLState::h_DeviceContext()
{
	__FuncGuard("CGLState::h_DeviceContext");

	return m_h_dc;
}

HGLRC CGLState::h_RenderingContext()
{
	__FuncGuard("CGLState::h_RenderingContext");

	return m_h_glrc;
}

void __fastcall CGLState::LineWidth(float f_width)
{
	__FuncGuard("CGLState::LineWidth");

	if(m_p_cur_state->f_line_width != f_width)
		glLineWidth(m_p_cur_state->f_line_width = f_width);
}

void __fastcall CGLState::PointSize(float f_size)
{
	__FuncGuard("CGLState::PointSize");

	if(m_p_cur_state->f_point_size != f_size)
		glPointSize(m_p_cur_state->f_point_size = f_size);
}

void __fastcall CGLState::PolygonMode(int n_side, int n_mode)
{
	__FuncGuard("CGLState::PolygonMode");

	if(n_side == GL_FRONT_AND_BACK &&
	   (n_mode != m_p_cur_state->n_polygon_mode[0] ||
	   n_mode != m_p_cur_state->n_polygon_mode[1])) {
		glPolygonMode(GL_FRONT_AND_BACK, m_p_cur_state->n_polygon_mode[0] =
			m_p_cur_state->n_polygon_mode[1] = n_mode);
	} else if(n_side == GL_FRONT &&
	   n_mode != m_p_cur_state->n_polygon_mode[0])
		glPolygonMode(GL_FRONT, m_p_cur_state->n_polygon_mode[0] = n_mode);
	else if(n_side == GL_BACK &&
	   n_mode != m_p_cur_state->n_polygon_mode[1])
		glPolygonMode(GL_BACK, m_p_cur_state->n_polygon_mode[1] = n_mode);	
}

void __fastcall CGLState::EnableFog()
{
	__FuncGuard("CGLState::EnableFog");

	if(!m_p_cur_state->b_fog) {
		m_p_cur_state->b_fog = true;
		glEnable(GL_FOG);
	}
}

void __fastcall CGLState::DisableFog()
{
	__FuncGuard("CGLState::DisableFog");

	if(m_p_cur_state->b_fog) {
		m_p_cur_state->b_fog = false;
		glDisable(GL_FOG);
	}
}

void __fastcall CGLState::FogMode(int n_value)
{
	__FuncGuard("CGLState::FogMode");

	if(m_p_cur_state->n_fog_type != n_value)
		glFogi(GL_FOG_MODE, m_p_cur_state->n_fog_type = n_value);
}

void __fastcall CGLState::FogDensity(float f_value)
{
	__FuncGuard("CGLState::FogDensity");

	if(m_p_cur_state->f_fog_density != f_value)
		glFogf(GL_FOG_DENSITY, m_p_cur_state->f_fog_density = f_value);
}

void __fastcall CGLState::FogColor(const float *p_color)
{
	__FuncGuard("CGLState::FogColor");

	if(memcmp(m_p_cur_state->f_fog_color, p_color, 4 * sizeof(float))) {
		memcpy(m_p_cur_state->f_fog_color, p_color, 4 * sizeof(float));
		glFogfv(GL_FOG_COLOR, p_color);
	}
}

void __fastcall CGLState::FogStart(float f_value)
{
	__FuncGuard("CGLState::FogStart");

	if(m_p_cur_state->f_fog_start != f_value)
		glFogf(GL_FOG_START, m_p_cur_state->f_fog_start = f_value);
}

void __fastcall CGLState::FogEnd(float f_value)
{
	__FuncGuard("CGLState::FogEnd");

	if(m_p_cur_state->f_fog_end != f_value)
		glFogf(GL_FOG_END, m_p_cur_state->f_fog_end = f_value);
}

void __fastcall CGLState::EnableLighting()
{
	__FuncGuard("CGLState::EnableLighting");

	if(!m_p_cur_state->b_lighting) {
		glEnable(GL_LIGHTING);
		m_p_cur_state->b_lighting = true;
	}
}

void __fastcall CGLState::DisableLighting()
{
	__FuncGuard("CGLState::DisableLighting");

	if(m_p_cur_state->b_lighting) {
		glDisable(GL_LIGHTING);
		m_p_cur_state->b_lighting = false;
	}
}

void __fastcall CGLState::EnableLight(int n_light)
{
	__FuncGuard("CGLState::EnableLight");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(!m_p_cur_state->b_light[n_light]) {
		glEnable(GL_LIGHT0 + n_light);
		m_p_cur_state->b_light[n_light] = true;
	}
}

void __fastcall CGLState::DisableLight(int n_light)
{
	__FuncGuard("CGLState::DisableLight");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(m_p_cur_state->b_light[n_light]) {
		glDisable(GL_LIGHT0 + n_light);
		m_p_cur_state->b_light[n_light] = false;
	}
}

void __fastcall CGLState::ShadeModel(int n_model)
{
	__FuncGuard("CGLState::ShadeModel");

	if(m_p_cur_state->n_shade_model != n_model)
		glShadeModel(m_p_cur_state->n_shade_model = n_model);
}

void __fastcall CGLState::LightAmbient(int n_light, const float *p_color)
{
	__FuncGuard("CGLState::LightAmbient");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(memcmp(m_p_cur_state->t_light_data[n_light].f_ambient, p_color, 4 * sizeof(float))) {
		memcpy(m_p_cur_state->t_light_data[n_light].f_ambient, p_color, 4 * sizeof(float));
		glLightfv(GL_LIGHT0 + n_light, GL_AMBIENT, p_color);
	}
}

void __fastcall CGLState::LightDiffuse(int n_light, const float *p_color)
{
	__FuncGuard("CGLState::LightDiffuse");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(memcmp(m_p_cur_state->t_light_data[n_light].f_diffuse, p_color, 4 * sizeof(float))) {
		memcpy(m_p_cur_state->t_light_data[n_light].f_diffuse, p_color, 4 * sizeof(float));
		glLightfv(GL_LIGHT0 + n_light, GL_DIFFUSE, p_color);
	}
}

void __fastcall CGLState::LightSpecular(int n_light, const float *p_color)
{
	__FuncGuard("CGLState::LightSpecular");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(memcmp(m_p_cur_state->t_light_data[n_light].f_specular, p_color, 4 * sizeof(float))) {
		memcpy(m_p_cur_state->t_light_data[n_light].f_specular, p_color, 4 * sizeof(float));
		glLightfv(GL_LIGHT0 + n_light, GL_SPECULAR, p_color);
	}
}

void __fastcall CGLState::LightPosition(int n_light, const float *p_position)
{
	__FuncGuard("CGLState::LightPosition");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(memcmp(m_p_cur_state->t_light_data[n_light].f_position, p_position, 4 * sizeof(float))) {
		memcpy(m_p_cur_state->t_light_data[n_light].f_position, p_position, 4 * sizeof(float));
		glLightfv(GL_LIGHT0 + n_light, GL_POSITION, p_position);
	}
}

void __fastcall CGLState::LightSpotDirection(int n_light, const float *p_direction)
{
	__FuncGuard("CGLState::LightSpotDirection");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(memcmp(m_p_cur_state->t_light_data[n_light].f_spot_direction, p_direction, 3 * sizeof(float))) {
		memcpy(m_p_cur_state->t_light_data[n_light].f_spot_direction, p_direction, 3 * sizeof(float));
		glLightfv(GL_LIGHT0 + n_light, GL_SPOT_DIRECTION, p_direction);
	}
}

void __fastcall CGLState::LightSpotExponent(int n_light, float f_value)
{
	__FuncGuard("CGLState::LightSpotExponent");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(m_p_cur_state->t_light_data[n_light].f_spot_exponent != f_value) {
		glLightf(GL_LIGHT0 + n_light, GL_SPOT_EXPONENT,
			m_p_cur_state->t_light_data[n_light].f_spot_exponent = f_value);
	}
}

void __fastcall CGLState::LightSpotCutoff(int n_light, float f_value)
{
	__FuncGuard("CGLState::LightSpotCutoff");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(m_p_cur_state->t_light_data[n_light].f_spot_cutoff != f_value) {
		glLightf(GL_LIGHT0 + n_light, GL_SPOT_CUTOFF,
			m_p_cur_state->t_light_data[n_light].f_spot_cutoff = f_value);
	}
}

void __fastcall CGLState::LightConstantAttenuation(int n_light, float f_value)
{
	__FuncGuard("CGLState::LightConstantAttenuation");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(m_p_cur_state->t_light_data[n_light].f_constant_attenuation != f_value) {
		glLightf(GL_LIGHT0 + n_light, GL_CONSTANT_ATTENUATION,
			m_p_cur_state->t_light_data[n_light].f_constant_attenuation = f_value);
	}
}

void __fastcall CGLState::LightLinearAttenuation(int n_light, float f_value)
{
	__FuncGuard("CGLState::LightLinearAttenuation");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(m_p_cur_state->t_light_data[n_light].f_linear_attenuation != f_value) {
		glLightf(GL_LIGHT0 + n_light, GL_LINEAR_ATTENUATION,
			m_p_cur_state->t_light_data[n_light].f_linear_attenuation = f_value);
	}
}

void __fastcall CGLState::LightQuadraticAttenuation(int n_light, float f_value)
{
	__FuncGuard("CGLState::LightQuadraticAttenuation");

	_ASSERTE(n_light >= 0 && n_light < m_p_cur_state->n_max_lights);

	if(m_p_cur_state->t_light_data[n_light].f_quadratic_attenuation != f_value) {
		glLightf(GL_LIGHT0 + n_light, GL_QUADRATIC_ATTENUATION,
			m_p_cur_state->t_light_data[n_light].f_quadratic_attenuation = f_value);
	}
}

void __fastcall CGLState::EnableBlend()
{
	__FuncGuard("CGLState::EnableBlend");

	if(!m_p_cur_state->b_blend) {
		glEnable(GL_BLEND);
		m_p_cur_state->b_blend = true;
	}
}

void __fastcall CGLState::DisableBlend()
{
	__FuncGuard("CGLState::DisableBlend");

	if(m_p_cur_state->b_blend) {
		glDisable(GL_BLEND);
		m_p_cur_state->b_blend = false;
	}
}

void __fastcall CGLState::BlendFunc(int n_src_op, int n_dest_op)
{
	__FuncGuard("CGLState::BlendFunc");

	if(m_p_cur_state->n_blend_func[0] != n_src_op ||
	   m_p_cur_state->n_blend_func[1] != n_dest_op) {
		glBlendFunc(m_p_cur_state->n_blend_func[0] = n_src_op,
			m_p_cur_state->n_blend_func[1] = n_dest_op);
	}
}

void __fastcall CGLState::EnableDepthTest()
{
	__FuncGuard("CGLState::EnableDepthTest");

	if(!m_p_cur_state->b_depth_test) {
		glEnable(GL_DEPTH_TEST);
		m_p_cur_state->b_depth_test = true;
	}
}

void __fastcall CGLState::DisableDepthTest()
{
	__FuncGuard("CGLState::DisableDepthTest");

	if(m_p_cur_state->b_depth_test) {
		glDisable(GL_DEPTH_TEST);
		m_p_cur_state->b_depth_test = false;
	}
}

void __fastcall CGLState::DepthFunc(int n_op)
{
	__FuncGuard("CGLState::DepthFunc");

	if(m_p_cur_state->n_depth_func != n_op)
		glDepthFunc(m_p_cur_state->n_depth_func = n_op);
}

void __fastcall CGLState::EnableAlphaTest()
{
	__FuncGuard("CGLState::EnableAlphaTest");

	if(!m_p_cur_state->b_alpha_test) {
		glEnable(GL_ALPHA_TEST);
		m_p_cur_state->b_alpha_test = true;
	}
}

void __fastcall CGLState::DisableAlphaTest()
{
	__FuncGuard("CGLState::DisableAlphaTest");

	if(m_p_cur_state->b_alpha_test) {
		glDisable(GL_ALPHA_TEST);
		m_p_cur_state->b_alpha_test = false;
	}
}

void __fastcall CGLState::AlphaFunc(int n_op, float f_tresh)
{
	__FuncGuard("CGLState::AlphaFunc");

	if(m_p_cur_state->n_depth_func != n_op ||
	   m_p_cur_state->f_alpha_tresh != f_tresh) {
		glAlphaFunc(m_p_cur_state->n_depth_func = n_op,
			m_p_cur_state->f_alpha_tresh = f_tresh);
	}
}

void __fastcall CGLState::DisableAllTexture1D()
{
	__FuncGuard("CGLState::DisableAllTexture1D");

	int i;

	for(i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
#ifdef __MULTITEXTURE_ENABLED
		glActiveTexture(GL_TEXTURE0 + i);
#endif
		if(m_p_cur_state->b_texture1d_enabled[i]) {
			glDisable(GL_TEXTURE_1D);
			m_p_cur_state->b_texture1d_enabled[i] = false;
		}
	}
#ifdef __MULTITEXTURE_ENABLED
	glActiveTexture(GL_TEXTURE0 + m_p_cur_state->n_active_tex_unit);
#endif
}

void __fastcall CGLState::DisableAllTextures()
{
	int i;

	for(i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
#ifdef __MULTITEXTURE_ENABLED
		glActiveTexture(GL_TEXTURE0 + i);
#endif
		if(m_p_cur_state->b_texture1d_enabled[i]) {
			glDisable(GL_TEXTURE_1D);
			m_p_cur_state->b_texture1d_enabled[i] = false;
		}
		if(m_p_cur_state->b_texture2d_enabled[i]) {
			glDisable(GL_TEXTURE_2D);
			m_p_cur_state->b_texture2d_enabled[i] = false;
		}
		if(m_p_cur_state->b_texture3d_enabled[i]) {
			glDisable(GL_TEXTURE_3D);
			m_p_cur_state->b_texture3d_enabled[i] = false;
		}
		if(m_p_cur_state->b_texture_cubemap_enabled[i]) {
			glDisable(GL_TEXTURE_CUBE_MAP);
			m_p_cur_state->b_texture_cubemap_enabled[i] = false;
		}
	}
#ifdef __MULTITEXTURE_ENABLED
	glActiveTexture(GL_TEXTURE0 + m_p_cur_state->n_active_tex_unit);
#endif
}

void __fastcall CGLState::DisableAllTexture2D()
{
	__FuncGuard("CGLState::DisableAllTexture2D");

	int i;

	for(i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
#ifdef __MULTITEXTURE_ENABLED
		glActiveTexture(GL_TEXTURE0 + i);
#endif
		if(m_p_cur_state->b_texture2d_enabled[i]) {
			glDisable(GL_TEXTURE_2D);
			m_p_cur_state->b_texture2d_enabled[i] = false;
		}
	}
#ifdef __MULTITEXTURE_ENABLED
	glActiveTexture(GL_TEXTURE0 + m_p_cur_state->n_active_tex_unit);
#endif
}

void __fastcall CGLState::DisableAllTexture3D()
{
	__FuncGuard("CGLState::DisableAllTexture3D");

	int i;

	for(i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
#ifdef __MULTITEXTURE_ENABLED
		glActiveTexture(GL_TEXTURE0 + i);
#endif
		if(m_p_cur_state->b_texture3d_enabled[i]) {
			glDisable(GL_TEXTURE_3D);
			m_p_cur_state->b_texture3d_enabled[i] = false;
		}
	}
#ifdef __MULTITEXTURE_ENABLED
	glActiveTexture(GL_TEXTURE0 + m_p_cur_state->n_active_tex_unit);
#endif
}

void __fastcall CGLState::DisableAllTextureCubeMap()
{
	__FuncGuard("CGLState::DisableAllTextureCubeMap");

	int i;

	for(i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
#ifdef __MULTITEXTURE_ENABLED
		glActiveTexture(GL_TEXTURE0 + i);
#endif
		if(m_p_cur_state->b_texture_cubemap_enabled[i]) {
			glDisable(GL_TEXTURE_CUBE_MAP);
			m_p_cur_state->b_texture_cubemap_enabled[i] = false;
		}
	}
#ifdef __MULTITEXTURE_ENABLED
	glActiveTexture(GL_TEXTURE0 + m_p_cur_state->n_active_tex_unit);
#endif
}

void __fastcall CGLState::DisableAllTexture(int n_type)
{
	__FuncGuard("CGLState::DisableAllTexture");

	switch(n_type) {
	case GL_TEXTURE_1D:
		DisableAllTexture1D();
		return;
	case GL_TEXTURE_2D:
		DisableAllTexture2D();
		return;
	case GL_TEXTURE_3D:
		DisableAllTexture3D();
		return;
	case GL_TEXTURE_CUBE_MAP:
		DisableAllTextureCubeMap();
		return;
	default:
		for(int i = 0; i < m_p_cur_state->n_max_texture_units; i ++) {
#ifdef __MULTITEXTURE_ENABLED
			glActiveTexture(GL_TEXTURE0 + i);
#endif
			glDisable(n_type);
		}
#ifdef __MULTITEXTURE_ENABLED
		glActiveTexture(m_p_cur_state->n_active_tex_unit);
#endif
	}
}

void __fastcall CGLState::EnableTexture1D()
{
	__FuncGuard("CGLState::EnableTexture1D");

	if(!m_p_cur_state->b_texture1d_enabled[m_p_cur_state->n_active_tex_unit]) {
		glEnable(GL_TEXTURE_1D);
		m_p_cur_state->b_texture1d_enabled[m_p_cur_state->n_active_tex_unit] = true;
	}
}

void __fastcall CGLState::EnableTexture2D()
{
	__FuncGuard("CGLState::EnableTexture2D");

	if(!m_p_cur_state->b_texture2d_enabled[m_p_cur_state->n_active_tex_unit]) {
		glEnable(GL_TEXTURE_2D);
		m_p_cur_state->b_texture2d_enabled[m_p_cur_state->n_active_tex_unit] = true;
	}
}

void __fastcall CGLState::EnableTexture3D()
{
	__FuncGuard("CGLState::EnableTexture3D");

	if(!m_p_cur_state->b_texture3d_enabled[m_p_cur_state->n_active_tex_unit]) {
		glEnable(GL_TEXTURE_3D);
		m_p_cur_state->b_texture3d_enabled[m_p_cur_state->n_active_tex_unit] = true;
	}
}

void __fastcall CGLState::EnableTextureCubeMap()
{
	__FuncGuard("CGLState::EnableTextureCubeMap");

	if(!m_p_cur_state->b_texture_cubemap_enabled[m_p_cur_state->n_active_tex_unit]) {
		glEnable(GL_TEXTURE_CUBE_MAP);
		m_p_cur_state->b_texture_cubemap_enabled[m_p_cur_state->n_active_tex_unit] = true;
	}
}

void __fastcall CGLState::EnableTexture(int n_type)
{
	__FuncGuard("CGLState::EnableTexture");

	switch(n_type) {
	case GL_TEXTURE_1D:
		EnableTexture1D();
		return;
	case GL_TEXTURE_2D:
		EnableTexture2D();
		return;
	case GL_TEXTURE_3D:
		EnableTexture3D();
		return;
	case GL_TEXTURE_CUBE_MAP:
		EnableTextureCubeMap();
		return;
	default:
		glEnable(n_type);
	}
}

void __fastcall CGLState::DisableTexture1D()
{
	__FuncGuard("CGLState::DisableTexture1D");

	if(m_p_cur_state->b_texture1d_enabled[m_p_cur_state->n_active_tex_unit]) {
		glDisable(GL_TEXTURE_1D);
		m_p_cur_state->b_texture1d_enabled[m_p_cur_state->n_active_tex_unit] = false;
	}
}

void __fastcall CGLState::DisableTexture2D()
{
	__FuncGuard("CGLState::DisableTexture2D");

	if(m_p_cur_state->b_texture2d_enabled[m_p_cur_state->n_active_tex_unit]) {
		glDisable(GL_TEXTURE_2D);
		m_p_cur_state->b_texture2d_enabled[m_p_cur_state->n_active_tex_unit] = false;
	}
}

void __fastcall CGLState::DisableTexture3D()
{
	__FuncGuard("CGLState::DisableTexture3D");

	if(m_p_cur_state->b_texture3d_enabled[m_p_cur_state->n_active_tex_unit]) {
		glDisable(GL_TEXTURE_3D);
		m_p_cur_state->b_texture3d_enabled[m_p_cur_state->n_active_tex_unit] = false;
	}
}

void __fastcall CGLState::DisableTextureCubeMap()
{
	__FuncGuard("CGLState::DisableTextureCubeMap");

	if(m_p_cur_state->b_texture_cubemap_enabled[m_p_cur_state->n_active_tex_unit]) {
		glDisable(GL_TEXTURE_CUBE_MAP);
		m_p_cur_state->b_texture_cubemap_enabled[m_p_cur_state->n_active_tex_unit] = false;
	}
}

void __fastcall CGLState::DisableTexture(int n_type)
{
	__FuncGuard("CGLState::DisableTexture");

	switch(n_type) {
	case GL_TEXTURE_1D:
		DisableTexture1D();
		return;
	case GL_TEXTURE_2D:
		DisableTexture2D();
		return;
	case GL_TEXTURE_3D:
		DisableTexture3D();
		return;
	case GL_TEXTURE_CUBE_MAP:
		DisableTextureCubeMap();
		return;
	default:
		glDisable(n_type);
	}
}

void __fastcall CGLState::ActiveTextureUnit(int n_unit)
{
	__FuncGuard("CGLState::ActiveTextureUnit");

	_ASSERTE(n_unit >= 0 && n_unit < m_p_cur_state->n_max_texture_units);

#ifdef __MULTITEXTURE_ENABLED
	if(m_p_cur_state->n_active_tex_unit != n_unit)
		glActiveTexture(GL_TEXTURE0 + (m_p_cur_state->n_active_tex_unit = n_unit));
#endif
}

int __fastcall CGLState::n_TexUnit_Num()
{
	__FuncGuard("CGLState::n_TexUnit_Num");

	return m_p_cur_state->n_max_texture_units;
}

void __fastcall CGLState::BindTexture1D(int n_texture)
{
	__FuncGuard("CGLState::BindTexture1D");

	if(m_p_cur_state->n_texture1d[m_p_cur_state->n_active_tex_unit] != n_texture) {
		glBindTexture(GL_TEXTURE_1D,
			m_p_cur_state->n_texture1d[m_p_cur_state->n_active_tex_unit] = n_texture);
	}
}

void __fastcall CGLState::BindTexture2D(int n_texture)
{
	__FuncGuard("CGLState::BindTexture2D");

	if(m_p_cur_state->n_texture2d[m_p_cur_state->n_active_tex_unit] != n_texture) {
		glBindTexture(GL_TEXTURE_2D,
			m_p_cur_state->n_texture2d[m_p_cur_state->n_active_tex_unit] = n_texture);
	}
}

void __fastcall CGLState::BindTexture3D(int n_texture)
{
	__FuncGuard("CGLState::BindTexture3D");

	if(m_p_cur_state->n_texture3d[m_p_cur_state->n_active_tex_unit] != n_texture) {
		glBindTexture(GL_TEXTURE_3D,
			m_p_cur_state->n_texture3d[m_p_cur_state->n_active_tex_unit] = n_texture);
	}
}

void __fastcall CGLState::BindTextureCubeMap(int n_texture)
{
	__FuncGuard("CGLState::BindTextureCubeMap");

	if(m_p_cur_state->n_texture_cubemap[m_p_cur_state->n_active_tex_unit] != n_texture) {
		glBindTexture(GL_TEXTURE_CUBE_MAP,
			m_p_cur_state->n_texture_cubemap[m_p_cur_state->n_active_tex_unit] = n_texture);
	}
}

void __fastcall CGLState::BindTexture(int n_type, int n_texture)
{
	__FuncGuard("CGLState::BindTexture");

	switch(n_type) {
	case GL_TEXTURE_1D:
		BindTexture1D(n_texture);
		return;
	case GL_TEXTURE_2D:
		BindTexture2D(n_texture);
		return;
	case GL_TEXTURE_3D:
		BindTexture3D(n_texture);
		return;
	case GL_TEXTURE_CUBE_MAP:
		BindTextureCubeMap(n_texture);
		return;
	default:
		glBindTexture(n_type, n_texture);
	}
}

void __fastcall CGLState::Color3f(float r, float g, float b)
{
	__FuncGuard("CGLState::Color3f");

	if(m_p_cur_state->f_color[0] != r ||
	   m_p_cur_state->f_color[1] != g ||
	   m_p_cur_state->f_color[2] != b) {
		glColor3f(m_p_cur_state->f_color[0] = r,
				  m_p_cur_state->f_color[1] = g,
				  m_p_cur_state->f_color[2] = b);
		//
		m_p_cur_state->n_color[0] = (unsigned char)(m_p_cur_state->f_color[0] * 0xff);
		m_p_cur_state->n_color[1] = (unsigned char)(m_p_cur_state->f_color[1] * 0xff);
		m_p_cur_state->n_color[2] = (unsigned char)(m_p_cur_state->f_color[2] * 0xff);
	}
}

void __fastcall CGLState::Color4f(float r, float g, float b, float a)
{
	__FuncGuard("CGLState::Color4f");

	if(m_p_cur_state->f_color[0] != r ||
	   m_p_cur_state->f_color[1] != g ||
	   m_p_cur_state->f_color[2] != b ||
	   m_p_cur_state->f_color[3] != a) {
		glColor4f(m_p_cur_state->f_color[0] = r,
				  m_p_cur_state->f_color[1] = g,
				  m_p_cur_state->f_color[2] = b,
				  m_p_cur_state->f_color[3] = a);
		//
		m_p_cur_state->n_color[0] = (unsigned char)(m_p_cur_state->f_color[0] * 0xff);
		m_p_cur_state->n_color[1] = (unsigned char)(m_p_cur_state->f_color[1] * 0xff);
		m_p_cur_state->n_color[2] = (unsigned char)(m_p_cur_state->f_color[2] * 0xff);
		m_p_cur_state->n_color[3] = (unsigned char)(m_p_cur_state->f_color[3] * 0xff);
	}
}

void __fastcall CGLState::Color3b(unsigned char r, unsigned char g, unsigned char b)
{
	__FuncGuard("CGLState::Color3b");

	if(m_p_cur_state->n_color[0] != r ||
	   m_p_cur_state->n_color[1] != g ||
	   m_p_cur_state->n_color[2] != b) {
		glColor3b(m_p_cur_state->n_color[0] = r,
				  m_p_cur_state->n_color[1] = g,
				  m_p_cur_state->n_color[2] = b);
		//
		m_p_cur_state->f_color[0] = m_p_cur_state->f_color[0] / 255.0f;
		m_p_cur_state->f_color[1] = m_p_cur_state->f_color[1] / 255.0f;
		m_p_cur_state->f_color[2] = m_p_cur_state->f_color[2] / 255.0f;
	}
}

void __fastcall CGLState::Color4b(unsigned char r, unsigned char g,
									   unsigned char b, unsigned char a)
{
	__FuncGuard("CGLState::Color4b");

	if(m_p_cur_state->n_color[0] != r ||
	   m_p_cur_state->n_color[1] != g ||
	   m_p_cur_state->n_color[2] != b ||
	   m_p_cur_state->n_color[3] != a) {
		glColor4b(m_p_cur_state->n_color[0] = r,
				  m_p_cur_state->n_color[1] = g,
				  m_p_cur_state->n_color[2] = b,
				  m_p_cur_state->n_color[3] = a);
		//
		m_p_cur_state->f_color[0] = m_p_cur_state->f_color[0] / 255.0f;
		m_p_cur_state->f_color[1] = m_p_cur_state->f_color[1] / 255.0f;
		m_p_cur_state->f_color[2] = m_p_cur_state->f_color[2] / 255.0f;
		m_p_cur_state->f_color[3] = m_p_cur_state->f_color[3] / 255.0f;
	}
}

void __fastcall CGLState::ClearColor4f(float r, float g, float b, float a)
{
	__FuncGuard("CGLState::ClearColor4f");

	if(m_p_cur_state->f_clr_color[0] != r ||
	   m_p_cur_state->f_clr_color[1] != g ||
	   m_p_cur_state->f_clr_color[2] != b ||
	   m_p_cur_state->f_clr_color[3] != a) {
		glClearColor(m_p_cur_state->f_clr_color[0] = r,
					 m_p_cur_state->f_clr_color[1] = g,
					 m_p_cur_state->f_clr_color[2] = b,
					 m_p_cur_state->f_clr_color[3] = a);
	}
}

void __fastcall CGLState::ClearDepth(float d)
{
	__FuncGuard("CGLState::ClearDepth");

	if(m_p_cur_state->f_clr_depth != d)
		glClearDepth(m_p_cur_state->f_clr_depth = d);
}

void __fastcall CGLState::EnableCullFace()
{
	__FuncGuard("CGLState::EnableCullFace");

	if(!m_p_cur_state->b_cull_face) {
		glEnable(GL_CULL_FACE);
		m_p_cur_state->b_cull_face = true;
	}
}

void __fastcall CGLState::DisableCullFace()
{
	__FuncGuard("CGLState::DisableCullFace");

	if(m_p_cur_state->b_cull_face) {
		glDisable(GL_CULL_FACE);
		m_p_cur_state->b_cull_face = false;
	}
}

void __fastcall CGLState::CullFace(int n_mode)
{
	__FuncGuard("CGLState::CullFace");

	if(m_p_cur_state->n_cull_face != n_mode)
		glCullFace(m_p_cur_state->n_cull_face = n_mode);
}

void __fastcall CGLState::FrontFace(int n_mode)
{
	__FuncGuard("CGLState::FrontFace");

	if(m_p_cur_state->n_front_face != n_mode)
		glFrontFace(m_p_cur_state->n_front_face = n_mode);
}

void __fastcall CGLState::Enable_STRQ_TexGen()
{
	__FuncGuard("CGLState::Enable_STRQ_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0])
		glEnable(GL_TEXTURE_GEN_S);
	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1])
		glEnable(GL_TEXTURE_GEN_T);
	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2])
		glEnable(GL_TEXTURE_GEN_R);
	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3])
		glEnable(GL_TEXTURE_GEN_Q);
	//
	m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] =
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] =
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2] =
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3] = true;
}

void __fastcall CGLState::Disable_STRQ_Texgen()
{
	__FuncGuard("CGLState::Disable_STRQ_Texgen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0])
		glDisable(GL_TEXTURE_GEN_S);
	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1])
		glDisable(GL_TEXTURE_GEN_T);
	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2])
		glDisable(GL_TEXTURE_GEN_R);
	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3])
		glDisable(GL_TEXTURE_GEN_Q);
	//
	m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] =
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] =
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2] =
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3] = false;
}

void __fastcall CGLState::Enable_S_TexGen()
{
	__FuncGuard("CGLState::Enable_S_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0]) {
		glEnable(GL_TEXTURE_GEN_S);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] = true;
	}
}

void __fastcall CGLState::Enable_T_TexGen()
{
	__FuncGuard("CGLState::Enable_T_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1]) {
		glEnable(GL_TEXTURE_GEN_T);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] = true;
	}
}

void __fastcall CGLState::Enable_R_TexGen()
{
	__FuncGuard("CGLState::Enable_R_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2]) {
		glEnable(GL_TEXTURE_GEN_R);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2] = true;
	}
}

void __fastcall CGLState::Enable_Q_TexGen()
{
	__FuncGuard("CGLState::Enable_Q_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3]) {
		glEnable(GL_TEXTURE_GEN_Q);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3] = true;
	}
}

void __fastcall CGLState::Disable_S_TexGen()
{
	__FuncGuard("CGLState::Disable_S_TexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0]) {
		glDisable(GL_TEXTURE_GEN_S);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] = false;
	}
}

void __fastcall CGLState::Disable_T_TexGen()
{
	__FuncGuard("CGLState::Disable_T_TexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1]) {
		glDisable(GL_TEXTURE_GEN_T);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] = false;
	}
}

void __fastcall CGLState::Disable_R_TexGen()
{
	__FuncGuard("CGLState::Disable_R_TexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2]) {
		glDisable(GL_TEXTURE_GEN_R);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2] = false;
	}
}

void __fastcall CGLState::Disable_Q_TexGen()
{
	__FuncGuard("CGLState::Disable_Q_TexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3]) {
		glDisable(GL_TEXTURE_GEN_Q);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][3] = false;
	}
}

void __fastcall CGLState::Enable_ST_TexGen()
{
	__FuncGuard("CGLState::Enable_ST_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0]) {
		glEnable(GL_TEXTURE_GEN_S);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] = true;
	}
	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1]) {
		glEnable(GL_TEXTURE_GEN_T);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] = true;
	}
}

void __fastcall CGLState::Enable_STR_TexGen()
{
	__FuncGuard("CGLState::Enable_STR_TexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0]) {
		glEnable(GL_TEXTURE_GEN_S);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] = true;
	}
	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1]) {
		glEnable(GL_TEXTURE_GEN_T);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] = true;
	}
	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2]) {
		glEnable(GL_TEXTURE_GEN_R);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2] = true;
	}
}

void __fastcall CGLState::EnableTexGen(int n_coord)
{
	__FuncGuard("CGLState::EnableTexGen");

	if(!m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][n_coord - GL_TEXTURE_GEN_S]) {
		glEnable(n_coord);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][n_coord - GL_TEXTURE_GEN_S] = true;
	}
}

void __fastcall CGLState::Disable_ST_TexGen()
{
	__FuncGuard("CGLState::Disable_ST_TexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0]) {
		glDisable(GL_TEXTURE_GEN_S);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] = false;
	}
	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1]) {
		glDisable(GL_TEXTURE_GEN_T);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] = false;
	}
}

void __fastcall CGLState::Disable_STR_TexGen()
{
	__FuncGuard("CGLState::Disable_STR_TexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0]) {
		glDisable(GL_TEXTURE_GEN_S);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][0] = false;
	}
	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1]) {
		glDisable(GL_TEXTURE_GEN_T);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][1] = false;
	}
	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2]) {
		glDisable(GL_TEXTURE_GEN_R);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][2] = false;
	}
}

void __fastcall CGLState::DisableTexGen(int n_coord)
{
	__FuncGuard("CGLState::DisableTexGen");

	if(m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][n_coord - GL_TEXTURE_GEN_S]) {
		glDisable(n_coord);
		m_p_cur_state->b_texgen_enabled[m_p_cur_state->n_active_tex_unit][n_coord - GL_TEXTURE_GEN_S] = false;
	}
}

void __fastcall CGLState::TexGen_STRQ_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_STRQ_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_OBJECT_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_OBJECT_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float));
			glTexGenfv(GL_R, GL_OBJECT_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float));
			glTexGenfv(GL_Q, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_EYE_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_EYE_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float));
			glTexGenfv(GL_R, GL_EYE_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float));
			glTexGenfv(GL_Q, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_S_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_S_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_T_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_T_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_R_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_R_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float));
			glTexGenfv(GL_R, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float));
			glTexGenfv(GL_R, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_Q_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_Q_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float));
			glTexGenfv(GL_Q, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][3], p_param, 4 * sizeof(float));
			glTexGenfv(GL_Q, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_ST_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_ST_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_OBJECT_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_EYE_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_STR_fv(int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGen_STR_fv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_OBJECT_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_OBJECT_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float));
			glTexGenfv(GL_R, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][0], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S, GL_EYE_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][1], p_param, 4 * sizeof(float));
			glTexGenfv(GL_T, GL_EYE_PLANE, p_param);
		}
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][2], p_param, 4 * sizeof(float));
			glTexGenfv(GL_R, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGenfv(int n_coord, int n_name, float *p_param)
{
	__FuncGuard("CGLState::TexGenfv");

	if(n_name == GL_OBJECT_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][n_coord], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_object_plane[m_p_cur_state->n_active_tex_unit][n_coord], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S + n_coord, GL_OBJECT_PLANE, p_param);
		}
	} else if(n_name == GL_EYE_PLANE) {
		if(memcmp(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][n_coord], p_param, 4 * sizeof(float))) {
			memcpy(m_p_cur_state->f_texgen_eye_plane[m_p_cur_state->n_active_tex_unit][n_coord], p_param, 4 * sizeof(float));
			glTexGenfv(GL_S + n_coord, GL_EYE_PLANE, p_param);
		}
	} else {
		_ASSERTE(0);
	}
}

void __fastcall CGLState::TexGen_STRQ_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_STRQ_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] != n_param)
		glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] = n_param);
	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] != n_param)
		glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] = n_param);
	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][2] != n_param)
		glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][2] = n_param);
	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][3] != n_param)
		glTexGeni(GL_Q, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][3] = n_param);
}

void __fastcall CGLState::TexGen_STR_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_STR_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] != n_param)
		glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] = n_param);
	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] != n_param)
		glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] = n_param);
	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][2] != n_param)
		glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][2] = n_param);
}

void __fastcall CGLState::TexGen_S_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_S_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] != n_param)
		glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] = n_param);
}

void __fastcall CGLState::TexGen_T_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_T_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] != n_param)
		glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] = n_param);
}

void __fastcall CGLState::TexGen_R_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_R_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][2] != n_param)
		glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][2] = n_param);
}

void __fastcall CGLState::TexGen_Q_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_Q_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][3] != n_param)
		glTexGeni(GL_Q, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][3] = n_param);
}

void __fastcall CGLState::TexGen_ST_Mode(int n_name, int n_param)
{
	__FuncGuard("CGLState::TexGen_ST_Mode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] != n_param)
		glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][0] = n_param);
	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] != n_param)
		glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][1] = n_param);
}

void __fastcall CGLState::TexGenMode(int n_coord, int n_param)
{
	__FuncGuard("CGLState::TexGenMode");

	if(m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][n_coord] != n_param)
		glTexGeni(GL_S + n_coord, GL_TEXTURE_GEN_MODE, m_p_cur_state->n_texgen_mode[m_p_cur_state->n_active_tex_unit][n_coord] = n_param);
}

void __fastcall CGLState::ColorMask(int b_r, int b_g, int b_b, int b_a)
{
	__FuncGuard("CGLState::ColorMask");

	if(m_p_cur_state->b_color_write[0] != b_r ||
	   m_p_cur_state->b_color_write[1] != b_g ||
	   m_p_cur_state->b_color_write[2] != b_b ||
	   m_p_cur_state->b_color_write[3] != b_a)
		glColorMask(m_p_cur_state->b_color_write[0] = b_r,
					m_p_cur_state->b_color_write[1] = b_g,
					m_p_cur_state->b_color_write[2] = b_b,
					m_p_cur_state->b_color_write[3] = b_a);
}

void __fastcall CGLState::DepthMask(int b_d)
{
	__FuncGuard("CGLState::DepthMask");

	if(m_p_cur_state->b_depth_write != b_d)
		glDepthMask(m_p_cur_state->b_depth_write = b_d);
}

void __fastcall CGLState::Enable(int n_value)
{
	__FuncGuard("CGLState::Enable");

	switch(n_value) {
	case GL_ALPHA_TEST:
		EnableAlphaTest();
		return;
	case GL_DEPTH_TEST:
		EnableDepthTest();
		return;
	case GL_CULL_FACE:
		EnableCullFace();
		return;
	case GL_TEXTURE_1D:
		EnableTexture1D();
		return;
	case GL_TEXTURE_2D:
		EnableTexture2D();
		return;
	case GL_TEXTURE_3D:
		EnableTexture3D();
		return;
	case GL_TEXTURE_CUBE_MAP:
		EnableTextureCubeMap();
		return;
	case GL_BLEND:
		EnableBlend();
		return;
	case GL_LIGHTING:
		EnableLighting();
		return;
	case GL_LIGHT0:
	case GL_LIGHT1:
	case GL_LIGHT2:
	case GL_LIGHT3:
	case GL_LIGHT4:
	case GL_LIGHT5:
	case GL_LIGHT6:
	case GL_LIGHT7:
		EnableLight(n_value - GL_LIGHT0);
		return;
	case GL_FOG:
		EnableFog();
		return;
	case GL_TEXTURE_GEN_S:
		Enable_S_TexGen();
		return;
	case GL_TEXTURE_GEN_T:
		Enable_T_TexGen();
		return;
	case GL_TEXTURE_GEN_R:
		Enable_R_TexGen();
		return;
	case GL_TEXTURE_GEN_Q:
		Enable_Q_TexGen();
		return;
	case GL_POLYGON_OFFSET_EXT:
		EnablePolygonOffset();
		return;

	default:
		glEnable(n_value);
	}
}

void __fastcall CGLState::Disable(int n_value)
{
	__FuncGuard("CGLState::Disable");

	switch(n_value) {
	case GL_ALPHA_TEST:
		DisableAlphaTest();
		return;
	case GL_DEPTH_TEST:
		DisableDepthTest();
		return;
	case GL_CULL_FACE:
		DisableCullFace();
		return;
	case GL_TEXTURE_1D:
		DisableTexture1D();
		return;
	case GL_TEXTURE_2D:
		DisableTexture2D();
		return;
	case GL_TEXTURE_3D:
		DisableTexture3D();
		return;
	case GL_TEXTURE_CUBE_MAP:
		DisableTextureCubeMap();
		return;
	case GL_BLEND:
		DisableBlend();
		return;
	case GL_LIGHTING:
		DisableLighting();
		return;
	case GL_LIGHT0:
	case GL_LIGHT1:
	case GL_LIGHT2:
	case GL_LIGHT3:
	case GL_LIGHT4:
	case GL_LIGHT5:
	case GL_LIGHT6:
	case GL_LIGHT7:
		DisableLight(n_value - GL_LIGHT0);
		return;
	case GL_FOG:
		DisableFog();
		return;
	case GL_TEXTURE_GEN_S:
		Disable_S_TexGen();
		return;
	case GL_TEXTURE_GEN_T:
		Disable_T_TexGen();
		return;
	case GL_TEXTURE_GEN_R:
		Disable_R_TexGen();
		return;
	case GL_TEXTURE_GEN_Q:
		Disable_Q_TexGen();
		return;
	case GL_POLYGON_OFFSET_EXT:
		DisablePolygonOffset();
		return;

	default:
		glDisable(n_value);
	}
}

void __fastcall CGLState::EnablePolygonOffset()
{
	__FuncGuard("CGLState::EnablePolygonOffset");

	if(m_p_cur_state->b_polygon_offset_support && !m_p_cur_state->b_polygon_offset) {
		glEnable(GL_POLYGON_OFFSET_EXT);
		m_p_cur_state->b_polygon_offset = true;
	}
}

void __fastcall CGLState::DisablePolygonOffset()
{
	__FuncGuard("CGLState::DisablePolygonOffset");

	if(m_p_cur_state->b_polygon_offset_support && m_p_cur_state->b_polygon_offset) {
		glDisable(GL_POLYGON_OFFSET_EXT);
		m_p_cur_state->b_polygon_offset = false;
	}
}

void __fastcall CGLState::PolygonOffset(float f_factor, float f_bias)
{
	__FuncGuard("CGLState::PolygonOffset");

	if(m_p_cur_state->b_polygon_offset_support) {
		if(m_p_cur_state->f_polygon_offset_factor != f_factor ||
		   m_p_cur_state->f_polygon_offset_bias != f_bias) {
			glPolygonOffsetEXT(m_p_cur_state->f_polygon_offset_factor != f_factor,
							   m_p_cur_state->f_polygon_offset_bias != f_bias);
		}
	}
}

/*
 *								=== ~CGLState ===
 */

/*
 *		-end-of-file-
 */
