/*
 *  headDetection.h
 */

#ifndef headDetectionH
#define headDetectionH

#include <math.h>
#include <vector>
#include <assert.h>
#include "Image.h"
#include "Vector2D.h"
#include "Background.h"
#include "skinColor.h"
#include "directionCircle.h"

class THeadDetector
{
  // Components in binary image
  struct Component
  {
    vector<iVector2D> pixels;
    int value; // Hodnotam jakou ma komp. v obraze!
    int MinX, MinY, MaxX, MaxY;

    Component(int v):MinX(1000), MinY(1000), MaxX(-1), MaxY(-1), value(v)
    {
      pixels.clear();
    }

     // Stred bounding boxu
    iVector2D center()
    {
      return iVector2D((MaxX+MinX)/2, (MaxY+MinY)/2);
    }

    // Vrati teziste komponenty
    iVector2D focus()
    {
      iVector2D t;
      for (unsigned i = 0; i < pixels.size(); i++)
      {
        t.x() += pixels[i].x();
        t.y() += pixels[i].y();
      }
      t = t / pixels.size();
      return t;
    }

    void moments(float & m0, float & m1, float & m2)
    {
      m0 = m1 = m2 = 0;
      iVector2D t = focus();
      for (unsigned i = 0; i < pixels.size(); i++)
      {
//        m0 += (pixels[i].x() - t.x())*(pixels[i].y() - t.y());
        m1 += pow(pixels[i].x() - t.x(), 2);
        m2 += pow(pixels[i].y() - t.y(), 2);
      }
//      m0 /= pixels.size();
      m1 /= pixels.size();
      m2 /= pixels.size();
    }

    float elipseAspect() {
      float m0, m1, m2;
      moments(m0, m1, m2);
      return sqrt(m1)/sqrt(m2);
    }
  }; // Component

  bool isHead(Component & c);
  void fillComponent(ImageStruct * img, int x, int y, int v, Component & c, int);
  vector<iVector2D> getComponentPositions(ImageStruct * img);

  int width;
  int height;
  int scale;

public:
  TBackground * bg;
  SkinColor * skin;
  pdCircleDetect * ellipse;
  ImageStruct * skinimg;

  float minCompactness;
  unsigned minComponentSize;
  float minAspectRatio;
  float maxAspectRatio;
  int skinProbabilityTreshold;
  int ellipseThreshold;
  bool useImageAlphaAsSkin;

  THeadDetector(int w, int h, int s, string modelFile): width(w), height(h), scale(s)
  {
    bg = new TBackground(w, h, scale);
    skin = new SkinColor(modelFile);
     // we do not use circle detection
    ellipse = new pdCircleDetect();
    ellipse->Init(w/s, h/s, w/s, h/s, 5, 12, 0.7);

    skinimg = NewImage8(w/s, h/s);

    // Set default values
    minCompactness = 0.5;
    minComponentSize = 280;
    minAspectRatio = 0.35;
    maxAspectRatio = 1.2;
    skinProbabilityTreshold = 60;
    ellipseThreshold = 1000;
    useImageAlphaAsSkin = false;
  }

  ~THeadDetector()
  {
    delete skin;
    delete bg;
    delete ellipse;
    DeleteImage(skinimg);
  }

  // Only need when using process1() to detect heads
  void update(ImageStruct * img)
  {
    bg->update(img);
  }

  void reloadColorModel(string modelFile)
  {
    if (skin) delete skin;
    skin = new SkinColor(modelFile);
  }

  // Processing methods
  vector<iVector2D> process(ImageStruct * img);
  vector<iVector2D> process1(ImageStruct * img);
  vector<iVector2D> process2(ImageStruct * img);

  ImageStruct * backgroundModel()
  {
    return bg->getBackground();
  }

  int isBackground(int x, int y)
  {
    return bg->isBackground(x, y);
  }

  void drawCross(ImageStruct * img, iVector2D p, Image8Value v = 254);
  void drawCrossRGB(ImageStruct * img, iVector2D p, ImageRGBValueRGB);

}; // THeadDetector

ImageStruct * getResizedImage(ImageStruct * img, int scale);

//---------------------------------------------------------------------------
#endif
