/*****************************************************************************/
/*                       Image processing functions                          */
/*  Source file : hough transform for ellipse detection                      */
/*  Designed by Igor Potucek (potucek@fit.vutbr.cz)                          */
/*  created 2005                                                             */
/*  Description: find ellipses in the image for head tracking                */
/*  Contributors:                                                            */
/*    Roman Jurnek (xjuran07@stud.fit.vutbr.cz)                             */
/*****************************************************************************/
#include "directioncircle.h"
#define max(a,b) ((a)>(b))?(a):(b)
#include <math.h>

const int EDGE_THRES = 20;


void pdCircleDetect::Init(int xsize, int ysize, int Width, int Height, int MinRadius, int MaxRadius, float ratio)
{
  XSize = Width;
  YSize = Height;
  XRatio = Width/(float)xsize;
  YRatio = Height/(float)ysize;
  if (Array!=NULL)
     delete [] Array;
  Array = new int[XSize*YSize];
  memset(Array,0,XSize*YSize*sizeof(int));
  MaxR = MaxRadius;
  MinR = MinRadius;
  Ratio = ratio;
  incR = 1/YRatio;
}

void pdCircleDetect::AddPoint(int x, int y, int ax, int ay, int weight)
{
   float alpha = 0;
   // precise method
   if (ax!=0)
   {
     // if ((fabs((float)ay/ax))>1)return ;
      alpha=atan((float)ay/ax);
   }
   else
     if (ay>0) alpha=M_PI/2.0;
   else
     if (ay<0) alpha=3*M_PI/2.0;
   float cosa = cos(alpha);
   float sina = sin(alpha);
   for (float r=MinR; r<MaxR; r+=incR)
   {
     int y1 = int( YRatio*(y+r*sina)+0.5);
     int y2 = int( YRatio*(y-r*sina)+0.5);
     int x1 = int( XRatio*(x+Ratio*r*cosa)+0.5);
     int x2 = int( XRatio*(x-Ratio*r*cosa)+0.5);
     // the center can lie on both directions from a border point
     if (((alpha>0) && (alpha<M_PI/2)) || ((alpha>M_PI) && (alpha<3*M_PI/2.0)))
     {   // edge direction
         if ((x2>=0) && (x2<XSize) && (y2>=0) && (y2<YSize))
           Array[x2+y2*XSize]+=weight;
         if ((x1>=0) && (x1<XSize) && (y1>=0) && (y1<YSize))
           Array[x1+y1*XSize]+=weight;
     }
     else
     {   // edge direction /
         if ((x1>=0) && (x1<XSize) && (y2>=0) && (y2<YSize))
           Array[x1+y2*XSize]+=weight;
         if ((x2>=0) && (x2<XSize) && (y1>=0) && (y1<YSize))
           Array[x2+y1*XSize]+=weight;
     }
   }
}

#ifdef DIGILIB
void pdCircleDetect::ProcessImage(ImageStruct *Image)
{
    int mask[3] = { -2, 0, 2 };
    for (int j=1; j<Image->YSize-1; j++)
      for (int i=1; i<Image->XSize-1; i++)
      {
         int sumY = 0;
         int sumX = 0;
         for(int J=-1; J<=1; J++)
            sumY = sumY + Image8Pixel(Image,i,j+J) * mask[J+1];
         for(int J=-1; J<=1; J++)
            sumX = sumX + Image8Pixel(Image,i+J,j) * mask[J+1];
         int direct = abs(sumX)+abs(sumY);
         if ((abs(sumX)>EDGE_THRES) || (abs(sumY)>EDGE_THRES))
         {
           AddPoint(i,j,sumX,sumY,1);
         }
      }
}

void pdCircleDetect::ProcessImageSkin(ImageStruct *Image, ImageStruct * skin, int x1, int y1, int x2, int y2)
{
    for (int j=y1+1; j<y2-1; j++)
      for (int i=x1+1; i<x2-1; i++)
      {
         int sumY = 0;
         int sumX = 0;
         sumY = -Image8Pixel(Image,i,j-1)+Image8Pixel(Image,i,j+1);
         sumX = -Image8Pixel(Image,i-1,j)+Image8Pixel(Image,i+1,j);
         int direct = abs(sumX)+abs(sumY);
         if ((abs(sumX)>EDGE_THRES) || (abs(sumY)>EDGE_THRES))
         {
//           int w = (Image8LinearPixel(skin, i, j) > 80) ? 50 : 1 ;
           int w = 1;
           for (int v = -1; v < 1; v++)
             for (int u = -1; u < 1; u++)
               if (Image8LinearPixel(skin, i+u, j+v) > 100) w = 50;
           AddPoint(i,j,sumX,sumY, w);
           //Array[XSize*j+i] = 255;
         }
      }
}

void pdCircleDetect::DrawEllipse(ImageStruct *Image, int x0, int y0, float ratio)
{
    float r = (MaxR-MinR)/2.0+MinR;
    for (float a=0; a<2*M_PI; a+=0.01)
    {
       float x = x0+r*Ratio*cos(a);
       float y = y0+r*sin(a);

       if ((x>=0) && (y>=0) && (x<Image->XSize) && (y<Image->YSize))
         SetImageRGBPixel(Image,int(x*ratio),int(y*ratio),50,250,50);
    }
}

int pdCircleDetect::FindMax(int &x, int &y)
{
    int great=1;

    // search maximum
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         if (great<Array[j*XSize+i])
         {
            great = Array[j*XSize+i];
            x = i; y = j;
         }
      }
    int MinX = x-MaxR/2-10;                if (MinX<0) MinX = 0;
    int MaxX = x+MaxR/2+10;                if (MaxX>XSize) MaxX = XSize;
    int MinY = y-MaxR/(2.0*Ratio)-10;      if (MinY<0) MinY = 0;
    int MaxY = y+MaxR/(2.0*Ratio)+10;      if (MaxY>YSize) MaxY = YSize;

    for (int j=MinY; j<MaxY; j++)
      for (int i=MinX; i<MaxX; i++)
        Array[j*XSize+i] = 0;
    x=x/XRatio; y=y/YRatio;

    return great;
}

//------------------------------------------------------------------------------
 long pdCircleDetect::DrawArray(ImageStruct *Image)
 {  // normalize by maximal value
    int great=1;
    // search maximum
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         great=max(great,Array[j*XSize+i]);
      }
    for (int j=0; j<YSize; j++)
      for (int i=0; i<XSize; i++)
      {
         int value=255*Array[j*XSize+i]/great;
         if (value>2)
           SetImageRGBPixel(Image,i,j,value, value, value)
         else
           SetImageRGBPixel(Image,i,j,0,0,0);
      }
    return great;
 }

#endif




