/*
*
*	This file contains structures that store pointer to image
*	and also information about used area in rectangle.
*	Structures has reference counting.
*
*	There is also manager for this objects.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <vector>
using namespace std;

#include <windows.h>

//FORWARD DECLARATIONS
namespace NSSimpleUnit
{
	class TImageRGBResult;
	class TImageRGBResultManager;
};

#ifndef _SIMPLEUNIT_IMAGERESSTRUCT_HH_
#define _SIMPLEUNIT_IMAGERESSTRUCT_HH_

#include "pipeline/UnitTypes.h"
#include "pipeline/Image.h"

namespace NSSimpleUnit
{

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageRGBResult

/** Tato trida definuje objekt, ktery obsahuje ukazatel na obrazek
*	typu TImageRGB a dale informace o platne casti obrazku typu TRT_Rect.
*
*	Trida obsahuje pocitadlo referenci.
*
*	Metody AddRefs() a Release() jsou synchronizovany pro vicenasobny pristup.
*
*	Objekt ze tridy lze vytvorit pouze pres manager TImageRGBResultManager.
*
*	Po dosazeni poctu referenci na 0 je objekt vracen zpet do manageru kde byl vytvoren.
*
*	Tato trida neni virtualni - je pouzivana pouze v ramci DLL jednotek a ty mohou
*	exportovat z DLL napr. ukazatelen na TImage a TRT_Rect.
*/
class TImageRGBResult
{
friend TImageRGBResultManager;

//PUBLIC METHODS
public:
	void				AddRefs(void);
	void				Release(void);
	DWORD				GetRefs(void) const;

	const TImageRGB*	GetImage(void) const;
	const TRT_Rect*		GetRectangle(void) const;

	TImageRGB*			GetImageWrite(void);
	void				SetRectangle( const TRT_Rect & rect );

//PRIVATE METHODS
private:
						TImageRGBResult( TImageRGBResultManager * manager, DWORD width, DWORD height );
						~TImageRGBResult( void );

	void				Reset(void);

//PRIVATE FAKE METHODS
private:
						TImageRGBResult( const TImageRGBResult & orig );	///< falesny kopirovaci konstruktor
	void				operator=( const TImageRGBResult & orig );			///< falesny prirazovaci operator

//PRIVATE COMPONENTS
private:
	DWORD						m_refs;		///< pocet referenci na objekt
	TImageRGBResultManager*		m_manager;	///< manager kde byl objekt vytvoren

	TCriticalSection			m_cs;		///< criticka sekce pro synchronizaci pristupu k poctu referenci

	TRT_Rect					m_rect;		///< informace o platne casti obrazku
	TImageRGB*					m_image;	///< ukazatel na obrazek	
};
//OK 2007-08-25 23:39:08 B04-315B\Tom

/** Konstruktor.
*
*	Vytvori uvnitr obrazek se zadanymi rozmery.
*
*	Obdelnik je nastaven na celou velikost obrazku - tj. [0,0, width - 1, height - 1].
*
*	Pocet referenci je nastaven na 0.
*
*	\param	manager		[in] manager kde byl objekt vytvoren
*	\param	width		[in] sirka obrazku v pixelech
*	\param	height		[in] vyska obrazku v pixelech
*/
inline TImageRGBResult::TImageRGBResult( 
	TImageRGBResultManager * manager,
	DWORD width,
	DWORD height 
	)
{
	m_manager = manager;

	m_refs = 0;

	m_image = new TImageRGB( width, height );

	m_rect.left		= 0;
	m_rect.top		= 0;
	m_rect.right	= width - 1;
	m_rect.bottom	= height - 1;
}
//OK 2007-08-25 23:39:11 B04-315B\Tom

/** Soukromy destruktor.
*
*	Muze byt volan pouze z manageru techto objektu
*/
inline TImageRGBResult::~TImageRGBResult( void )
{
	delete m_image;
	m_image = NULL;	//TODO
}
//OK 2007-08-25 23:39:13 B04-315B\Tom

/** Nastavi pocet referenci na 1 a velikost platneho
*	obdelniku nn cely obraz.
*
*	Metoda je volana v manageru objektu pri vraceni objektu
*	aplikaci. Tim je objekt uveden do stavu po vytvoreni
*/
inline void TImageRGBResult::Reset(void)
{
	m_refs = 1;

	m_rect.left		= 0;
	m_rect.top		= 0;
	m_rect.right	= m_image->GetWidth() - 1;
	m_rect.bottom	= m_image->GetHeight() - 1;
}
//OK 2007-08-25 23:39:17 B04-315B\Tom

/** Vrati ukazatel na obrazek ulozeny v tomto objektu.
*/
inline const TImageRGB* TImageRGBResult::GetImage(void) const
{
	return m_image;
}
//OK 2007-08-25 23:39:20 B04-315B\Tom

/** Vrati ukazatel na obdelnik urcujici platna data v tomto obrazku.
*/
inline const TRT_Rect* TImageRGBResult::GetRectangle(void) const
{
	return &m_rect;
}
//OK 2007-08-25 23:39:22 B04-315B\Tom

/** Vrati ukazatel na obrazek ulozeny v tomto objektu.
*
*	Obrazek je mozne modifikovat.
*/
inline TImageRGB* TImageRGBResult::GetImageWrite(void)
{
	return m_image;
}
//OK 2007-08-25 23:39:24 B04-315B\Tom

/** Nastavi obdelnik urcujici pouzitou cast obrazku.
*
*	\param	rect	[in] novy obdelnik
*/
inline void TImageRGBResult::SetRectangle( const TRT_Rect & rect )
{
	m_rect = rect;
}
//OK 2007-08-25 23:39:25 B04-315B\Tom

/** Zvysi pocet referenci na objekt.
*
*	Metoda je synchronizovana pro vicenasobny pristup.
*/
inline void TImageRGBResult::AddRefs(void)
{
	m_cs.Enter();
		m_refs++;
	m_cs.Leave();
}
//OK 2007-08-25 23:39:34 B04-315B\Tom

/** Vrati pocet referenci na tento objekt.
*/
inline DWORD TImageRGBResult::GetRefs(void) const
{	
	return m_refs;
}
//OK 2007-08-25 23:39:37 B04-315B\Tom

// TImageRGBResult
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TImageRGBResultManager

/** Manager objektu typu TImageRGBResult.
*
*	Pred jeho znicenim musi byt z aplikace odstraneny vsechny objekty vytvorene
*	timto managerem.
*
*	Metody GetObject() a InsertObject() jsou synchronizovany pro vicenasobny pristup.
*
*	Soukroma metoda InsertObject() vraci snimek zpatky do manageru. Je volana primo ze snimku
*	v okmaziku, kdy dojde k uvolneni posledni reference snimku.
*/
class TImageRGBResultManager
{
friend TImageRGBResult;

//PUBLIC METHODS
public:
							TImageRGBResultManager( DWORD width, DWORD height, DWORD initSize);
							~TImageRGBResultManager( void );

	TImageRGBResult*		GetObject( void );

//PRIVATE METHODS
private:
	void					InsertObject( TImageRGBResult* image );

//PRIVATE FAKE METHODS
private:
							TImageRGBResultManager( const TImageRGBResultManager & orig);	///< falesny kopirovaci konstruktor
	void					operator=( const TImageRGBResultManager & orig );				///< falesny operator prirazeni

//PRIVATE COMPONENTS
private:
	DWORD							m_width;		///< sirka vytvarenych obrazku
	DWORD							m_height;		///< vyska vytvarenych obrazku

	int								m_exportCount;	///< pocet objektu vydanych timto managerem
													///< do aplikace

	vector<TImageRGBResult*>		m_object;		///< pole s objekty

	TCriticalSection				m_cs;			///< criticka sekce pro soubezny pristup k
													///< objektum
};
//OK 2007-08-25 23:41:30 B04-315B\Tom

// TImageRGBResultManager
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

}; //END of NSSimpleUnit
using namespace NSSimpleUnit;

#endif