/*
*	This file contains data types for working units.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <windows.h>

namespace NSPipeline
{
	class TImage;
};

#ifndef _PIPELINE_UNITTYPES_HH_
#define _PIPELINE_UNITTYPES_HH_

namespace NSPipeline
{
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tento enumerator definuje typ vypocetni jednotky.
*
*	Typ jednotky urcuje, jake vysledky jednotka vraci.
*/
enum EnumUnitType
{
	ENUM_UNITTYPE_NODATA = 0,				///< tento typ nevraci zadna data
											///< jednotka tohoto typu vraci vzdy NULL

	ENUM_UNITTYPE_INTEGER = 1,				///< navratova hodnota je integer
											///< vysledky jsou typu TUnitRetType_integeres

	ENUM_UNITTYPE_RECTANGLES = 2,			///< typ vraci pole obdelniku typu TRT_Rect
											///< vysledky jsou typu TUnitRetType_rectangles

	ENUM_UNITTYPE_IMAGES_RGB = 3,			///< typ vraci pole obrazku typu RGB ( TImageRGB)
											///< s obdelniky urcujicimi platnou cast obrazku
											///< vysledky jsou typu TUnitRetType_images	

	ENUM_UNITTYPE_IMAGES_RGB_RECTANGLES = 4,///< typ vraci pole obrazku typu RGB (TImageRGB) a zaroven 
											///< nezavisle pole obdelniku TRT_Rect
											///< vysledky jsou typu TUnitRetType_images_rectangles

	//ZDE MOHOU BYT PRIDAVANY DALSI TYPY

	ENUM_UNITTYPE_FORCED32 = 0x7fffffff,	///< zarovnani na 32-bitu
};

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// datove typy pro vraceni vysledku z jednotek
#pragma pack( 1 )

/** Datovy typ reprezentujici obdelnik.
*
*	X-souradnice roste zleva doprava.
*	Y-souradnice roste odshora dolu.
*/
struct TRT_Rect
{
//PUBLIC METHODS
public:
			TRT_Rect(void);
			TRT_Rect( int l, int t, int r, int b);

	int		GetWidth(void) const;
	int		GetHeight(void) const;

//PUBLIC COMPONENTS
public:
	int		left;
	int		top;
	int		right;
	int		bottom;
};
//OK 2007-08-25 15:48:29 B04-315B\Tom

/** Defaultni konstruktor.
*
*	Nastavi vsechny polozky na 0.
*/
inline TRT_Rect::TRT_Rect(void)
{
	left = 
	right = 
	top = 
	bottom = 0;
}
//OK 2007-08-25 18:12:46 B04-315B\Tom

/** Konstruktor.
*
*	\param	l		[in] x-souradnice leveho horniho rohu
*	\param	t		[in] y-souradnice leveho horniho rohu
*	\param	r		[in] x-souradnice leveho horniho rohu
*	\param	b		[in] y-souradnice leveho horniho rohu
*/
inline TRT_Rect::TRT_Rect( int l, int t, int r, int b)
{
	left	=	l;
	top		=	t;
	right	=	r;
	bottom	=	b;
}
//OK 2007-08-25 15:48:31 B04-315B\Tom

/** Vrati sirku obdelniku.
*/
inline int TRT_Rect::GetWidth(void) const
{
	return right - left + 1;
}
//OK 2007-08-25 15:48:33 B04-315B\Tom
//OK 2007-09-08 12:45:38 B04-315B\Tom

/** Vrati vysku obdelniku.
*/
inline int TRT_Rect::GetHeight(void) const
{
	return bottom - top + 1;
}
//OK 2007-08-25 15:48:35 B04-315B\Tom
//OK 2007-09-08 12:45:40 B04-315B\Tom

#pragma pack()
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// abstraktni tridy pro navrat hodnot z vypocetnich jednotek

/** Rozhrani pro vsechny tridy, pomoci nichz se predavaji vysledky.
*
*	Jednotlive navratove datove typy jednotek se odvozuji z tohoto rozhrani.
*
*	Vzdy musi byt zachovana pouze jednoducha dedicnost - kvuli snadnemu pretypovani bez RTTI
*	z potomka na predka!!!
*/
class TUnitRetTypeInterface
{
//PUBLIC METHODS
public:
	virtual EnumUnitType	GetType(void) const = 0;		///< vrati typ dat,
															///< mel by byt shodny s typem jednotky
	virtual void			Release(void) = 0;				///< uvolneni vysledku zpet jednotce

//PROTECTED METHODS
protected:
							TUnitRetTypeInterface(void){};	///< konstruktor
							~TUnitRetTypeInterface(void){};	///< destruktor je pristupny pouze 
															///< pro odvozene tridy

//PROTECTED FAKE METHODS
protected:
							TUnitRetTypeInterface( const TUnitRetTypeInterface & orig );///< falesny kopirovaci konstruktor
	void					operator=( const TUnitRetTypeInterface & orig );			///< falesny prirazovaci operator

};
//OK 2007-08-25 15:49:54 B04-315B\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tato trida reprezentuje navratovy typ vypocetni jednotky reprezentujici cele cislo.
*
*	Odvozena trida musi reimplementovat metody Release() a GetValue().
*/
class TUnitRetType_integer : public TUnitRetTypeInterface
{
//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumUnitType	GetType(void) const { return ENUM_UNITTYPE_INTEGER; };
	virtual void			Release(void) = 0;

//PUBLIC METHODS
public:
	virtual int				GetValue(void) = 0;

//PROTECTED METHODS
protected:
							TUnitRetType_integer(void){};	///< konstruktor
							~TUnitRetType_integer(void){};	///< destruktor je pristupny pouze 
															///< pro odvozene tridy
//PROTECTED FAKE METHODS
protected:
							TUnitRetType_integer( const TUnitRetType_integer & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TUnitRetType_integer & orig );				///< falesny prirazovaci operator
};
//OK 2007-08-25 18:08:23 B04-315B\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tato trida reprezentuje navratovy typ vypocetni jednotky reprezentujici pole obdelniku.
*
*	Odvozena trida musi reimplementovat metody Release(), GetRectangleCount()
*	a GetRectangle().
*/
class TUnitRetType_rectangles : public TUnitRetTypeInterface
{
//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumUnitType	GetType(void) const { return ENUM_UNITTYPE_RECTANGLES; };
	virtual void			Release(void) = 0;

//PUBLIC METHODS
public:
	virtual DWORD			GetRectangleCount(void) = 0;		///< pocet obdelniku
	virtual	const TRT_Rect*	GetRectangle( DWORD index ) = 0;	///< vrati obdelnik na zadanem indexu
																///< vraci NULL pri chybnem indexu

//PROTECTED METHODS
protected:
							TUnitRetType_rectangles(void){};	///< konstruktor
							~TUnitRetType_rectangles(void){};	///< destruktor je pristupny pouze 
																///< pro odvozene tridy

//PROTECTED FAKE METHODS
protected:
							TUnitRetType_rectangles( const TUnitRetType_rectangles & orig );///< falesny kopirovaci konstruktor
	void					operator=( const TUnitRetType_rectangles & orig );				///< falesny prirazovaci operator
};
//OK 2007-08-25 18:08:20 B04-315B\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tato trida reprezentuje navratovy typ vypocetni jednotky reprezentujici pole obrazku.
*	Ke kazdemu obrazku lze ziskat obdelnik, ktery urcuje platny vyrez v obrazku
*
*	Odvozena trida musi reimplementovat metody 
\verbatim
	GetType()
	Release()
	GetImageCount()
	GetImage()
	GetRectangle()	
\endverbatim
*/
class TUnitRetType_images : public TUnitRetTypeInterface
{
//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumUnitType	GetType(void) const = 0;
	virtual void			Release(void) = 0;

//PUBLIC METHODS
public:
	virtual DWORD			GetImageCount(void) = 0;			///< pocet obrazku (a obdelniku)
	virtual	const TImage*	GetImage( DWORD index ) = 0;		///< vrati obrazek na zadanem indexu
																///< vraci NULL pri chybnem indexu
	virtual	const TRT_Rect*	GetRectangle( DWORD index ) = 0;	///< vrati obdelnik na zadanem indexu
																///< vraci NULL pri chybnem indexu

//PROTECTED METHODS
protected:
							TUnitRetType_images(void){};		///< konstruktor
							~TUnitRetType_images(void){};		///< destruktor je pristupny pouze 
																///< pro odvozene tridy

//PROTECTED FAKE METHODS
protected:
							TUnitRetType_images( const TUnitRetType_images & orig );///< falesny kopirovaci konstruktor
	void					operator=( const TUnitRetType_images & orig );			///< falesny prirazovaci operator

};
//OK 2007-08-25 18:08:18 B04-315B\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/** Tato trida reprezentuje navratovy typ vypocetni jednotky reprezentujici pole 
*	obrazku a obdelniku.
*
*	Obrazky a obdelniky jsou nezavisle - muze jich tedy byt ruzny pocet.
*
*	Odvozena trida musi reimplementovat metody 
\verbatim
	GetType()
	Release()
	GetImageCount()
	GetImage()
	GetRectangleCount()
	GetRectangle()	
\endverbatim
*/
class TUnitRetType_images_rectangles : public TUnitRetTypeInterface
{
//PUBLIC OVERRIDEN METHODS
public:
	virtual EnumUnitType	GetType(void) const = 0;
	virtual void			Release(void) = 0;

//PUBLIC METHODS
public:
	virtual DWORD			GetImageCount(void) = 0;			///< pocet obrazku
	virtual	const TImage*	GetImage( DWORD index ) = 0;		///< vrati obrazek na zadanem indexu
																///< vraci NULL pri chybnem indexu

	virtual DWORD			GetRectangleCount(void) = 0;		///< pocet obdelniku
	virtual	TRT_Rect		GetRectangle( DWORD index ) = 0;	///< vrati obdelnik na zadanem indexu

//PROTECTED METHODS
protected:
							TUnitRetType_images_rectangles(void){};		///< konstruktor
							~TUnitRetType_images_rectangles(void){};	///< destruktor je pristupny pouze 
																		///< pro odvozene tridy

//PROTECTED FAKE METHODS
protected:
							TUnitRetType_images_rectangles( const TUnitRetType_images_rectangles & orig );	///< falesny kopirovaci konstruktor
	void					operator=( const TUnitRetType_images_rectangles & orig );						///< falesny prirazovaci operator

};
//OK 2007-11-13 21:44:17 B04-315A\Tom

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

}; //END of namespace NSPipeline
using namespace NSPipeline;

#endif