/*
*	This file contains simple working unit from that may be simply derived others units.
*
*	Author:
*			Tomas Mrkvicka
*			xmrkvi03@stud.fit.vutbr.cz
*
*/

#include <cassert>

#include "pipeline/SimpleUnit.h"

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// TSimpleUnit

/** Tato staticka metoda tvori telo vlakna ve kterem jednotka
*	zpracovava snimky
*
*	\param	ptr	[in] ukazatel na jednotku typu TSimpleUnit
*/
DWORD WINAPI TSimpleUnit::TSimpleUnitThread(void* ptr)
{
	TSimpleUnit * simple_unit = (TSimpleUnit*)ptr;

	/** Smycka bezi dokud vlakno nezastavi.
	*/
	while( simple_unit->GetState() != ENUM_UNIT_STOP )
	{
		simple_unit->Loop();
	};

	return 0;
}
//OK 2007-08-25 18:42:47 B04-315B\Tom

/** Konstruktor.
*	Vytvori jednotku, ktera musi byt nasledne spustena pomoci metody Start.
*
*	\param	dispatcher	[in] rozhrani na objekt pro ziskavani novych snimku z kamery
*	\param	unit		[in] rozhrani vlastni zpracovavajici jednotky - objekt je jednotkou automaticky
*						znicen v destruktoru
*	\param	lockFrame	[in] tento parametr urcuje zda bude jednotka zamykat snimky
*						Zamknutim snimku jednotka neumozni zobrazit snimek dokud neni kompletne zpracovan.
*/
TSimpleUnit::TSimpleUnit(
	TDispatcherInterface* dispatcher,
	TSimpleUnitProcessingInterface * unit,
	BOOL lockFrame)
{
	m_thread		= NULL;

	m_state			= ENUM_UNIT_LOADFRAME;

	m_dispatcher	= dispatcher;
	m_unit			= unit;

	m_lockFrame		= lockFrame;
	m_frame			= NULL;

	m_interval		= 0;
}
//OK 2007-08-25 18:42:50 B04-315B\Tom

/** Destruktor
*
*	Destruktor smi byt volan pouze po radnem zastaveni jednotky metodou Stop!!!
*	To zajisti, ze jednotka nebude udrzovat reference na zadne snimky.
*/
TSimpleUnit::~TSimpleUnit(void)
{
	//snimek musi byt odstranen pred znicenim - toho je docileno zastavenim jednotky
	assert( m_frame == NULL );

	delete m_unit;
	m_unit = NULL;
}
//OK 2007-08-25 18:42:52 B04-315B\Tom

/** Zniceni objektu.
*
*	Neni zde zadne pocitani referenci, takze jednotka se znici okamzite.
*
*	Metoda smi byt volana v okamziku kdy jiz jednotka nebezi.
*
*	Tato metoda musi byt reimplementovana v odvozene tride !!!
*/
void TSimpleUnit::Release(void)
{
	delete this;
}
//OK 2007-08-25 18:42:54 B04-315B\Tom

/** Spusteni jednotky.
*
*	Vraci TRUE pokud byla jednotka spustena nebo jiz bezi.
*
*	FALSE vraci pri kriticke chybe, kdyz napr. neni mozne vytvorit vlakno apod.
*	V takovem pripade je nutne okmazite ukoncit aplikaci.
*/
BOOL TSimpleUnit::Start(void)
{
	if( ! m_thread )
	{
		//jednotka nebezi

		//nastavime jednotce pocatecni stav
		SetState(ENUM_UNIT_LOADFRAME);

		//spustime vypocetni vlakno
		m_thread = new TThread;
		BOOL res = m_thread->Run( (FUNC_PTR)TSimpleUnitThread,(void*)this);

		if ( res )
		{
			return TRUE;
		}
		else
		{
			//nejaka chyba
			delete m_thread;
			m_thread  = NULL;
			return FALSE;
		}
	}
	else
	{
		//jednotka uz bezi
		return TRUE;
	}
}
//OK 2007-08-25 18:43:10 B04-315B\Tom

/** Zastaveni jednotky.
*
*	Vraci TRUE pokud se jednotku podarilo zastavit.
*/
BOOL TSimpleUnit::Stop(void)
{
	if( m_thread )
	{
		//jednotka byla spustena

		if( m_thread->IsTerminated() )
		{
			//vlakno uz bylo ukonceno - pouze uklidime
			m_thread->Finish();	//zde metoda uspeje
			delete m_thread;
			m_thread = NULL;
		}
		else
		{
			//vlakno stale bezi - pokusime se ho ukoncit standardni cestou
			//zajistime ze vlakno jednotky nezmeni stav
			m_critical.Enter();
				if(m_state == ENUM_UNIT_STOP)
				{
					//jednotka jiz stoji, bude brzy ukoncena v ramci sveho vlakna
				}
				else
				{
					switch(m_state)
					{
						//jednotka pracuje - dame ji pokyn ze po dokonceni uz nema pokracovat
						case ENUM_UNIT_WORK:
						case ENUM_UNIT_LOADFRAME:
							m_state = ENUM_UNIT_FINISHWORK;
						break;

						//jednotka dokoncuje ulohu
						case ENUM_UNIT_FINISHWORK:
							//jednotka se brzy sama zastavi
						break;

						default:
							//nejaka chyba
							m_state = ENUM_UNIT_FINISHWORK;
							assert(false);
						break;
					}		
				}				
			m_critical.Leave();

			//nyni cekame na ukonceni jednotky
			//todo - tady muze pri zavazne chybe vzniknout nekonecny cyklus - zatim nevyresene
			while( ! m_thread->IsTerminated() )
			{
				Sleep(20);
			};

			//nyni jednotka stoji - ukoncime ji
			m_thread->Finish(); //metoda uspeje
			delete m_thread;
			m_thread = NULL;
		}

		//odstranime zbyvajici snimek - pokud existuje
		if( m_frame )
		{
			m_frame->Release();

			//odstranime zamek - pokud jednotka zamyka snimky
			if(m_lockFrame)
			{
				m_frame->ReleaseLock();
			}
			m_frame = NULL;
		}

		return TRUE;
	}
	else
	{
		//jednotka neni spustena
		return TRUE;
	}
}
//OK 2007-08-25 18:43:29 B04-315B\Tom

/** Hlavni smycka jednotky.
*
*	Je volana ze staticke metody TSimpleUnit::Run() a obsahuje ridici logiku cele jednotky.
*/
void TSimpleUnit::Loop(void)
{
	//pouzijime kopii aktualniho stavu - kvuli synchronizaci vlaken
	m_critical.Enter();
		EnumUnitState actState = m_state;
	m_critical.Leave();

	//zde uz externi vlakno muze menit stav jednotky, to se projevi pripadne v dalsim kroku iterace

	switch(actState)
	{
		//jednotka nic nedela - k tomuto by teoreticky nemelo dojit, protoze tento stav znamena ukonceni
		//hlavni smycky vlakna - viz staticka metoda TSimpleUnitThread
		case ENUM_UNIT_STOP:
		{
		}
		break;

		//nacteni noveho snimku		
		case ENUM_UNIT_LOADFRAME:
		{
			//nacteme novy snimek
			TFrame * frame = NULL;
			if( m_lockFrame )
			{
				//musime ziskat zamceny snimek
				frame = m_dispatcher->GetLockedFrame();
			}
			else
			{
				//jednotka snimky nezamyka
				frame = m_dispatcher->GetFrame();
			}			

			if(frame)
			{
				//ziskali jsme snimek
				//pokud existuje nejaky predchozi snimek zjistime zda se nejedna o identicke snimky
				if(m_frame)
				{
					//zjistime zda se jedna o novejsi snimek
					if( frame->GetTimestamp() != m_frame->GetTimestamp() )
					{
						//ano - toto je novejsi (jiny) snimek
						
						//rozdil mezi ID snimkku
						DWORD lastID = m_frame->GetTimestamp().GetID();
						DWORD newID = frame->GetTimestamp().GetID();

						//spocitame interval zahozenych snimku
						if ( newID > lastID )
						{
							m_interval = newID - lastID;
						}
						else
						{
							//pravdepodobne preteceni
							m_interval = 0;
						}

						//predchozi uvolnime
						if( m_lockFrame )
						{
							//jednotka drzi zamek
							m_frame->ReleaseLock();
							m_frame->Release();							
						}
						else
						{
							//pouze reference
							m_frame->Release();							
						}
						//ulozime novy snimek - referenci uz mame (vcetne pripadneho zamku) z metody
						//dispatcheru
						m_frame = frame;

						//pokud mame novy snimek tak prejdeme znovu do stavu zpracovavani snimku
						//musi platit aktualni stav ENUM_UNIT_LOADFRAME abychom mohli prejit do ENUM_UNIT_WORK
						//toto se muze zmenit pomoci metody Stop ktera zmeni stav na ENUM_UNIT_FINISHWORK
						m_critical.Enter();
							if ( m_state == ENUM_UNIT_LOADFRAME ) 
							{
								m_state = ENUM_UNIT_WORK;
							}
						m_critical.Leave();
					}
					else
					{
						//ne - snimky jsou stejne - pokracujeme v nacitani snimku
						//uvolnime referenci na ziskany snimek
						if(m_lockFrame)
						{
							frame->ReleaseLock();
							frame->Release();							
						}
						else
						{							
							frame->Release();							
						}

						//todo - mozna pridat nejaky cekaci stav
						Sleep(10);
					}
				}
				else // if ( m_frame )
				{
					//novy snimek (pravdepodobne prvni)
					m_frame = frame;

					//prechod do noveho stavu pokud jednotka nechce skoncit
					m_critical.Enter();
						if ( m_state == ENUM_UNIT_LOADFRAME )
						{
							// ok - jednotka ma stale pracovat
							m_state = ENUM_UNIT_WORK;
						}
					m_critical.Leave();

					// predchozi snimek nebyl k dispozici a interval je tedy 0
					m_interval = 0;
				}
			}
			else // if ( frame )
			{
				//zadny snimek neprisel
			}
		}
		break;

		//procesor zpracovava snimek
		case ENUM_UNIT_WORK:
		{
			//virtualni metoda zpracuje snimek tak jak je reimplementovana
			//zde musi byt vzdy nacten snimek !!!
			this->m_unit->ProcessFrame( m_frame );

			//snimek bude uvolnen bud pri nacitani dalsiho snimku nebo pri zastaveni jednotky
			//snimek potrebujeme pro porovnani s dalsim nactenym snimkem, proto ho zde nemuze uvolnit

			//musime znovu nacist snimek pokud nebyla volana metoda Stop 
			//podminene tedy prechazime do stavu ENUM_UNIT_LOADFRAME
			//Volani stop by zpusobilo zmenu stavu na ENUM_UNIT_FINISHWORK a tim by byla v pristim cyklu
			//prace jednotka dokoncena
			m_critical.Enter();
				if( m_state == ENUM_UNIT_WORK )
				{
					m_state = ENUM_UNIT_LOADFRAME;
				}
			m_critical.Leave();
		}
		break;

		//ukonceni veskere cinnosti a prechod do stavu ENUM_UNIT_STOP
		//tento stav nastane po zavolani metody Stop behem ENUM_UNIT_LOADFRAME nebo ENUM_UNIT_WORK
		case ENUM_UNIT_FINISHWORK:
		{		
			m_critical.Enter();
				m_state = ENUM_UNIT_STOP;
			m_critical.Leave();
		}
		break;

		//k tomuto by nemelo nikdy dojit
		default:
		{
			//jen pojistka
			assert(false);

			m_critical.Enter();
				m_state = ENUM_UNIT_STOP;
			m_critical.Leave();
		}
		break;
	}
}
//OK 2007-08-25 18:44:43 B04-315B\Tom

// TSimpleUnit
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
