/* RawToImage - converts raw file to pgm image and creates corresponding annotation.*/
/* Author: Michal Hradis @ UPMG FIT VUTBR */
/* Date: 26.3.2007 */

#include <cstdlib>
#include <iostream>

#include <fstream>
#include <string>
#include <sstream>

#include <string>

using namespace std;

bool loadRaw( const string & filename, int &sizeX, int &sizeY, unsigned &sampleCount, unsigned char * &cache);

int main(int argc, char *argv[])
{
    int border = 16;
    int horizontalCount = 25;
    
    string inputRawFileName;
    string outputAnotationFileName;
    string outputImageFileName;

    for( int i = 1; i < argc; i++){
        if( string( argv[ i]) == "-raw"){
            inputRawFileName = argv[ ++i];   
        } else if( string( argv[ i]) == "-img"){
            outputImageFileName = argv[ ++i];
        } else if( string( argv[ i]) == "-annot"){
            outputAnotationFileName = argv[ ++i];
        } else if( string( argv[ i]) == "-h"){
            cout << "Use: RawToImage.exe -raw rawfile -img outputimage [-annot outputannotationfile] [-b borderSize] [-hs horizontalSize] [-h] " << endl;
            cout << " Creates an *.pgm image and annotation from raw file." << endl;
            cout << endl;
            cout << " Options: " << endl;
            cout << " -raw filename     Specify the name of input raw file." << endl;
            cout << " -img filename     Specify the name of output image file." << endl;
            cout << " -annot filename   Specify the name of output annotation file." << endl;
            cout << "                   If not specified the annotation will not be created." << endl;
            cout << " -b borderSize     The size of free space between two samples (in pixels)." << endl;
            cout << " -hs count         The number of samples on horizontal line." << endl;
            cout << " -h                This help." << endl;
            exit( 1);
        } else if( string( argv[ i]) == "-b"){
            border = atoi( argv[ ++i]);
        } else if( string( argv[ i]) == "-hs"){
            horizontalCount = atoi( argv[ ++i]);  
        }
    }

    if( inputRawFileName.empty()){
        cout << " Input raw file missing. Use -h to get more information." << endl;
        exit(1);
    }    
    if( outputImageFileName.empty()){
        cout << " Output image file name missing. Use -h to get more information." << endl;
        exit( 1);   
    }
    int sizeX;
    int sizeY; 
    unsigned sampleCount;
    unsigned char * cache;
    
    if( !loadRaw( inputRawFileName.data(), sizeX, sizeY, sampleCount, cache)){
        cerr << "Error: Unable to load file " << inputRawFileName << endl;
    }
    
    int verticalCount = sampleCount / horizontalCount + 1;
    
    int stepX = (border + sizeX);
    int stepY = (border + sizeY);
    
    int outputSizeX = stepX * horizontalCount + border;
    int outputSizeY = stepY * verticalCount + border;
    
    cout << "Output sizeX: " << outputSizeX << endl;
    cout << "Output sizeY: " << outputSizeY << endl;
    
    unsigned char * outputImage = new unsigned char[ outputSizeX * outputSizeY];
    memset( ( char *) outputImage, 0, outputSizeX * outputSizeY * sizeof( unsigned char)); 
    
    int currentSampleID = 0;
    unsigned char * rawPosition = cache;
    
    int imagePosX = border;
    int imagePosY = border;
    
    ofstream annotationF;
    if( !outputAnotationFileName.empty()){
        annotationF.open( outputAnotationFileName.data(), ios::trunc);
    }

    if( !outputAnotationFileName.empty()){
        annotationF << "<image>  " <<  outputImageFileName << endl;
    }
    
    
    for( rawPosition = cache; currentSampleID < sampleCount; currentSampleID++){
        
        
        for( int i = 0; i < sizeY; i++, rawPosition += sizeX){
            memcpy(  ( char *) outputImage + imagePosX + (imagePosY + i) * outputSizeX,
                     rawPosition,  sizeX * sizeof( unsigned char)); 
        }
        
        if( !outputAnotationFileName.empty()){

            // generate annotation
            annotationF << "    <object> " << endl;
            annotationF << "        obj_type = rectangle" << endl;
            annotationF << "        sub_class        = fromRaw" << endl;
            annotationF << "        roi_LeftTop      = " << imagePosX << ":" << imagePosY << endl;
            annotationF << "        roi_LeftBottom   = " << imagePosX << ":" << imagePosY + sizeY -1 << endl;
            annotationF << "        roi_RightTop     = " << imagePosX + sizeX -1<< ":" << imagePosY << endl;
            annotationF << "        roi_RightBottom  = " << imagePosX + sizeX -1 << ":" << imagePosY + sizeY -1<< endl;
            annotationF << "    </object> " << endl;
        }
        
        
        imagePosX += stepX;
        if( imagePosX >= outputSizeX){
            imagePosX = border;
            imagePosY += stepY;    
        }
    }
    
    if( !outputAnotationFileName.empty()){
        annotationF << "</image>  " << endl;
    }
    
    ofstream outputFile( outputImageFileName.data(), ios::trunc | ios::binary);
    if( !outputFile.good()){
        cerr << "Error: Unable to open output file " << outputImageFileName << endl;
        exit( 1);   
    }

    outputFile.write( "P5\x0D\x0A", 4);

    stringstream ss;
    ss << outputSizeX << ' ' << outputSizeY;
    outputFile.write( ss.str().data(), ss.str().length());      
    outputFile.write( "\x0D\x0A", 2);
    outputFile.write( "255\x0D\x0A", 5);        
    outputFile.write( (char* ) outputImage, outputSizeX * outputSizeY);
    
    
    
    return EXIT_SUCCESS;
}

/// Header of .raw files
static const char rawHeader[5] = "S2D ";

bool loadRaw( const string & filename, int &sizeX, int &sizeY, unsigned &sampleCount, unsigned char * &cache)
{

    ifstream rawFile( filename.c_str(), ios::binary);

    if( !rawFile.good())
     {
        cerr << "Error: Unable to open file \"" << filename << "\"." << endl;
        return false;
    }
    
    char header[5];
    rawFile.read( (char*)header, 4);

    if( strncmp( header, rawHeader, 4) != 0)
    {
        cerr << "Error: Wrong file format \"" << filename << "\"." << endl;
        return false;
    }
    
    int supposedSampleCount;
    rawFile.read((char*)(&supposedSampleCount), 4);  
    rawFile.read((char*)(&sizeX), 4);
    rawFile.read((char*)(&sizeY), 4);
    
    int sizeToLoad = sizeX * sizeY * supposedSampleCount * sizeof( unsigned char);
    cache = new unsigned char[ sizeX * sizeY * supposedSampleCount];
    rawFile.read( (char *)cache, sizeToLoad);
    
    if( !rawFile.good()){
        delete[] cache;
        cache = NULL;
        cerr << "Error: Corrupted raw file \"" << filename << "\"." << endl;
        return false;   
    }
    
    sampleCount = supposedSampleCount;
    return true;
}

using namespace std;

bool loadImage( const std::string &path, int &sizeX, int &sizeY, unsigned char *&data){
    // only PGM image file format is supported at the moment so there is not much to do
    {
        ifstream inputFile( path.c_str(), ios::binary);
        if( inputFile.bad())
        {
            cerr << "Error: Unable to open file \"" << path << "\"." << endl;
            return false;
        }
        char buff[256];
        unsigned levels;
    
        inputFile.getline( buff, 256);
        if( strncmp( buff, "P5", 2) != 0)
        {
            cerr << "Error: Wrong image file format \"" << path << "\"." << endl;
            return false;
        }

        do // skip comments
        {
            inputFile.getline(buff, 256);
        } while (buff[0] == '#');

        if( sscanf( buff, "%d %d", &sizeX, &sizeY) != 2){
            cerr << "Error: Corupted image file \"" << path << "\"." << endl;
            return false;            
        }
        
        inputFile.getline(buff, 256);
        sscanf(buff, "%d", &levels);

        data = new unsigned char[ sizeX * sizeY];
        
        if( data == NULL)
        {
            cerr << "Error: Unable to allocate memory in loadImage()." << endl;
            return false;
        }

        inputFile.read( (char *) data, sizeX * sizeY * sizeof( unsigned char));
        
        if( !inputFile.good()){
            cerr << "Error: Corupted image file \"" << path << "\"." << endl;
            delete[] data;
            return false;
        }
        
        return true;
    }    
}

bool getImageInfo( const std::string &path, int &sizeX, int &sizeY){
    // only PGM image file format is supported at the moment so there is not much to do
    {
        ifstream inputFile( path.c_str(), ios::binary);
        if( inputFile.bad())
        {
            cerr << "Error: Unable to open file \"" << path << "\"." << endl;
            return false;
        }
        char buff[256];
    
        inputFile.getline( buff, 256);
        if( strncmp( buff, "P5", 2) != 0)
        {
            cerr << "Error: Wrong image file format \"" << path << "\"." << endl;
            return false;
        }
  
        do // skip comments
        {
            inputFile.getline(buff, 256);
        } while (buff[0] == '#');

        if( sscanf( buff, "%d %d", &sizeX, &sizeY) != 2){
            cerr << "Error: Corupted image file \"" << path << "\"." << endl;
            return false;            
        }

        return true;
    }        
}



bool getRAWInfo( const string & filename, int &sizeX, int &sizeY, unsigned & sampleCount)
{
    
    ifstream rawFile( filename.c_str(), ios::binary);

    if( !rawFile.good())
    {
        cerr << "Error: Unable to open file \"" << filename << "\"." << endl;
        return false;
    }
    
    char header[5];
    rawFile.read( (char*)header, 4);

    if( strncmp( header, rawHeader, 4) != 0)
    {
        cerr << "Error: Wrong file format \"" << filename << "\"." << endl;
        return false;
    }
    
    
    rawFile.read((char*)(&sampleCount), 4);
    rawFile.read((char*)(&sizeX), 4);
    rawFile.read((char*)(&sizeY), 4);
 
    if( !rawFile.good()){
        cerr << "Error: Corrupted raw file \"" << filename << "\"." << endl;
        return false;   
    }
    return true;
} // getRawInfo()
